package com.bizunited.platform.core.service.serviceable.init;

import com.bizunited.platform.common.service.init.InitProcessService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.service.invoke.InvokeProxyBuilder;
import com.bizunited.platform.core.service.invoke.InvokeRequestHandle;
import com.bizunited.platform.core.service.invoke.InvokeResponseHandle;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodInfo;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodProperty;
import com.bizunited.platform.core.service.serviceable.template.ValueMappingTemplate;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.dom4j.IllegalAddException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * 当进程启动时，该初始化服务负责完成系统中服务源的扫描过程，并加载到内存中
 * @author yinwenjie 
 */
@Component("ServicableMethodInitProcess2")
public class ServicableMethodInitProcess implements InitProcessService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ServicableMethodInitProcess.class);
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private ServicableMethodService servicableMethodService;
  @Autowired
  private BeanFactory beanFactory;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 所有的值映射模板都在这里，有spring ioc容器进行管理
   */
  @Autowired
  private List<ValueMappingTemplate> valueMappingTemplates;
  
  @Override
  public boolean doProcess() {
    // 每次进程启动都要进行初始化动作
    return true;
  }
  
  @Override
  public int sort() {
    // 初始化动作靠前
    return 5;
  }

  @Override
  public boolean stopOnException() {
    // 一旦出现异常，则终止进程启动过程
    return true;
  }

  @Override
  public void init() {
    try {
      this.initServicableMethod();
    } catch(NoSuchMethodException | ClassNotFoundException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage() , e);
    } 
  }
  /**
   * 对已经标注了serviceMethod的注解进行分析、保存处理——在spring启动成功后
   * @throws Exception
   */
  private void initServicableMethod() throws NoSuchMethodException , ClassNotFoundException {
    LOGGER.info("service method scan started .......");
    String beanDefinitionNames[] = applicationContext.getBeanDefinitionNames();
    if(beanDefinitionNames == null || beanDefinitionNames.length == 0) {
      return;
    }  
    Collection<Object> mappingAnnotations = new ArrayList<>(1000);
    for (String beanDefinitionName : beanDefinitionNames) {
      Object currentBean = applicationContext.getBean(beanDefinitionName);
      mappingAnnotations.add(currentBean);
    }
    // 寻找指定的实现类下，其直接实现的接口中的所有ServiceMethod标注
    ArrayList<Method> serviceMethods = new ArrayList<>();
    for (Object mappingAnnotationItem : mappingAnnotations) {
      Class<?> targetClass = this.recursionFindInterface(mappingAnnotationItem.getClass());
      if(targetClass == null) {
        continue;
      }
      Arrays.stream(targetClass.getDeclaredMethods())
        .filter(item -> item.getAnnotation(NebulaServiceMethod.class) != null).forEach(item -> serviceMethods.add(item));
    }
     
    /*
     * 处理过程为：
     * 1、首先判断当前ServiceMethod的基本信息，包括name、desc和returnPropertiesFilter、包名、类名、方法名等，并且检验name的重复性
     * 2、接着分析和处理其形参（最重要的步骤，详见私有方法）
     * 3、然后分析和处理其返回值类型（详见具体的私有方法）
     * 4、将已经完成的各个ServiceMethod方法，写入由nebulaStaticPersistentService控制的缓存
     * */
    List<ServicableMethodInfo> exsitServiceMethodReops = new ArrayList<>();
    int count = 0;
    for (Method currentMethod : serviceMethods) {
      ServicableMethodInfo serviceMethodReop = new ServicableMethodInfo();
      // 1、==============
      NebulaServiceMethod serviceMethodAnno = currentMethod.getAnnotation(NebulaServiceMethod.class);
      String serviceMethodAnnoName = serviceMethodAnno.name();
      if(StringUtils.isBlank(serviceMethodAnnoName)) {
        LOGGER.warn("指定的ServicableMethodProperty[{}]没有设定name，请检查!!", serviceMethodAnnoName);
        continue;
      }
      String serviceMethodAnnoDesc = serviceMethodAnno.desc();
      if(StringUtils.isBlank(serviceMethodAnnoDesc)) {
        LOGGER.warn("指定的ServicableMethodProperty[{}]没有设定desc，请检查!!", serviceMethodAnnoName);
        continue;
      }
      String returnPropertiesFilter = serviceMethodAnno.returnPropertiesFilter();
      // 方法名
      String methodSimpleName = currentMethod.getName();
      // 完整的接口类名
      String className = currentMethod.getDeclaringClass().getName();
      // 和已经分析完成的暴露方法相比，不能有重复的暴露方法名
      Validate.isTrue(!exsitServiceMethodReops.stream().map(ServicableMethodInfo::getName).filter(item -> StringUtils.equals(item, serviceMethodAnnoName)).findAny().isPresent(), "在进行服务源定义分析时，发现有重复的NebulaServiceMethod.name[" + serviceMethodAnnoName + "]，请检查!!");
      // 分析方法基本定义
      serviceMethodReop.setUsedScope(serviceMethodAnno.scope());
      serviceMethodReop.setDescription(serviceMethodAnnoDesc);
      serviceMethodReop.setInterfaceName(className);
      serviceMethodReop.setProjectName(platformContext.getAppName());
      serviceMethodReop.setSimpleMethodName(methodSimpleName);
      serviceMethodReop.setName(serviceMethodAnnoName);
      // 以下是和责任链加载有关的参数，需判定其合法性，并赋值
      this.checkInvokeHandle(serviceMethodAnno, currentMethod, serviceMethodReop);
      serviceMethodReop.setReturnPropertiesFilter(returnPropertiesFilter);
      
      // 2、==============
      analysisMethodParameters(serviceMethodReop, currentMethod);
      
      // 3、==============
      analysisMethodReturnParameter(serviceMethodReop, currentMethod);
      
      // 4、==============
      exsitServiceMethodReops.add(serviceMethodReop);
      servicableMethodService.create(serviceMethodReop);
      LOGGER.info(String.format("servicable named [%s] cached .......", serviceMethodReop.getName())); 
      count++;
    } 
    LOGGER.info(String.format("service method scan completed , [%d] servicable be found .......", count)); 
  }
  
  private void checkInvokeHandle(NebulaServiceMethod serviceMethodAnno , Method ctMethod , ServicableMethodInfo serviceMethodReop) {
    // 方法名
    String methodSimpleName = ctMethod.getName();
    // 完整的类名
    String className = ctMethod.getDeclaringClass().getName();
    String methodName = StringUtils.join(className , "." , methodSimpleName);
    String requestHandleGroup = serviceMethodAnno.requestHandleGroup();
    Class<?> requestHandleTypes[] = serviceMethodAnno.requestHandleTypes();
    // RequestHandleGroup和RequestHandleTypes不能都设定
    if(!StringUtils.isBlank(requestHandleGroup) 
        && (requestHandleTypes.length > 1 || (requestHandleTypes[0] != Void.class))) {
      throw new IllegalArgumentException(String.format("RequestHandleGroup和RequestHandleTypes不能同时设定[%s]", methodName));
    }
    // 如果requestHandleTypes设定了信息，则验证类型的合法性——必须实现了InvokeRequestHandle接口
    if(requestHandleTypes.length > 1 || requestHandleTypes[0] != Void.class) {
      for (Class<?> requestHandleType : requestHandleTypes) {
        Validate.isTrue(InvokeRequestHandle.class.isAssignableFrom(requestHandleType)
          , "类型[%s]并没有实现InvokeRequestHandle接口，不能配置到NebulaServiceMethod注解的requestHandleTypes属性[methodname=%s]，请检查" , requestHandleType.getTypeName() , methodName);
        
      }
      serviceMethodReop.setRequestHandleTypes(requestHandleTypes);
    }
    // 如果requestHandleGroup设定了值，则需要通过beanFactory中能够取出该bean，且该bean的类型是InvokeProxyBuilder
    if(!StringUtils.isBlank(requestHandleGroup)) {
      Object targetBean = null;
      try {
        targetBean = beanFactory.getBean(requestHandleGroup);
      } catch (NoSuchBeanDefinitionException e) {
        throw new IllegalAddException(String.format("未发现指定name的spring bean[%s]，请检查!!", requestHandleGroup));
      }
      Validate.isTrue(targetBean instanceof InvokeProxyBuilder , "bean name[%s]并不是一个InvokeProxyBuilder对象，不能配置到NebulaServiceMethod注解的requestHandleGroup属性[methodname=%s]，请检查" , requestHandleGroup , methodName);
      serviceMethodReop.setRequestHandleGroup(requestHandleGroup);
    }
    
    String responseHandleGroup = serviceMethodAnno.responseHandleGroup();
    Class<?> responseHandleTypes[] = serviceMethodAnno.responseHandleTypes();
    // ResponseHandleGroup和ResponseHandleTypes不能都设定
    if(!StringUtils.isBlank(responseHandleGroup) 
        && (responseHandleTypes.length > 1 || (responseHandleTypes[0] != Void.class))) {
      throw new IllegalArgumentException(String.format("ResponseHandleGroup和ResponseHandleTypes不能同时设定[%s]", methodName));
    }
    // 如果responseHandleTypes设定了信息，则验证类型的合法性——必须实现了InvokeResponseHandle接口
    if(responseHandleTypes.length > 1 || responseHandleTypes[0] != Void.class) {
      for (Class<?> responseHandleType : responseHandleTypes) {
        Validate.isTrue(InvokeResponseHandle.class.isAssignableFrom(responseHandleType)
          , "类型[%s]并没有实现InvokeResponseHandle接口，不能配置到NebulaServiceMethod注解的responseHandleTypes属性[methodname=%s]，请检查" , responseHandleType.getTypeName() , methodName);
      }
      serviceMethodReop.setResponseHandleTypes(responseHandleTypes);
    }
    // 如果responseHandleGroup设定了值，则需要通过beanFactory中能够取出该bean，且该bean的类型是InvokeProxyBuilder
    if(!StringUtils.isBlank(responseHandleGroup)) {
      Object targetBean = null;
      try {
        targetBean = beanFactory.getBean(responseHandleGroup);
      } catch (NoSuchBeanDefinitionException e) {
        throw new IllegalAddException(String.format("未发现指定name的spring bean[%s]，请检查!!", responseHandleGroup));
      }
      Validate.isTrue(targetBean instanceof InvokeProxyBuilder , "bean name[%s]并不是一个InvokeProxyBuilder对象，不能配置到NebulaServiceMethod注解的responseHandleGroup属性[methodname=%s]，请检查" , responseHandleGroup , methodName);
      serviceMethodReop.setResponseHandleGroup(responseHandleGroup);
    }
  }
  
  /**
   * 该私有方法用于分析指定方法的参数值信息
   * @param serviceMethodReop
   * @param ctMethod
   */
  private void analysisMethodParameters(ServicableMethodInfo serviceMethodReop , Method ctMethod) throws ClassNotFoundException {    
    /*
     * 
     * 进行参数分析，参数分析直接采用值映射模板进行，主要分为以下几类
     * a、基本类型和基本类型的一维数组表达，这部分的参数必须有@ServiceMethodParam注解标识，即使参数只有一个，也要注解标识 --->  params
     * 
     * b、以下类型由服务源自行分析绑定，无需进行任务注解设定：
     * Principal类型——和当前调用者身份有关的信息
     * InvokeProxyContext——链式调用上下文信息，其中包括了所有的传参信息
     * 
     * c、以下类型可由服务源自行分析，无需进行注解标识——由InvokeProxyContext中的jsonbody参数信息自动转换而成
     *  c1、存在于nebulaStaticPersistentService服务中的任何类型信息
     *  c2、经过第1.1、1.2处理后，遗留下来的，还没有进行
     * 
     * 注：
     * 
     * a、如果参数类型以上都不是，则抛出异常信息，不能识别的参数类型
     * b、如果找到相应的参数类型需要的值映射器，则将进行缓存记录
     * c、如果没有找到任何的额参数类型值映射器，则会抛出异常
     * */
    String methodName = ctMethod.getName();
    Class<?> targetType = ctMethod.getDeclaringClass();
    Parameter[] parameters = ctMethod.getParameters();
    if(parameters == null || parameters.length == 0) {
      serviceMethodReop.setProperties(null);
      return;
    } 
    
    List<ServicableMethodProperty> servicableMethodProperties = new ArrayList<>(16);
    for (int index = 0 ; index < parameters.length ; index++) {
      ServicableMethodProperty servicableMethodProperty = new ServicableMethodProperty();
      Parameter parameterItem = parameters[index];
      
      // 开始进行所需要的映射模板分析，找到的适用模板放在useValueMappingTemplates这里，以便随后进行缓存写入
      for (ValueMappingTemplate valueMappingTemplateItem : valueMappingTemplates) {
        boolean match = valueMappingTemplateItem.match(targetType, ctMethod, parameterItem, index);
        if(match) {
          servicableMethodProperty.setValueMappingTemplate(valueMappingTemplateItem);
          break;
        }
      }
      // 代码到这里说明当前参数没有找到适用的模板
      Validate.notNull(servicableMethodProperty.getValueMappingTemplate() , "在方法[%s]中，类型[%s]没有发现符合值映射所需的处理器，该类型目前服务源不支持，请检查!!" , methodName, parameterItem.getName());
      // 参数类型
      servicableMethodProperty.setParamType(parameterItem);
      // 参数索引位
      servicableMethodProperty.setPropertyIndex(index);
      // 参数的ServiceMethodParam注解信息
      ServiceMethodParam serviceMethodParam = (ServiceMethodParam)parameterItem.getAnnotation(ServiceMethodParam.class);
      if(serviceMethodParam != null) {
        servicableMethodProperty.setAnnonQualifiedName(serviceMethodParam.name());
        servicableMethodProperty.setHasParamAnnotation(true);
      } else {
        servicableMethodProperty.setAnnonQualifiedName("");
        servicableMethodProperty.setHasParamAnnotation(false);
      }
      servicableMethodProperties.add(servicableMethodProperty);
    }
    //设置方法参数实体数据
    serviceMethodReop.setProperties(servicableMethodProperties);
  }
  
  /**
   * 递归寻找“真实”类的定义
   * @param targetClass
   * @param currentInterfaces
   */
  private Class<?> recursionFindInterface(Class<?> targetClass) {
    // 如果当前类已经被cglib代理了，则需要解析字符串以后，再进行处理
    if(org.springframework.aop.SpringProxy.class.isAssignableFrom(targetClass)) {
      String cglibClassName = targetClass.getName();
      String realClassName = null;
      if(cglibClassName.indexOf("$$") != -1) {
        realClassName = cglibClassName.split("\\$\\$")[0];
      } else if(cglibClassName.indexOf("$") != -1) {
        realClassName = cglibClassName.split("\\$")[0];
      } else {
        return null;
      }
      Class<?> realClass = null;
      try {
        realClass = Class.forName(realClassName);
      } catch (ClassNotFoundException e) {
        LOGGER.debug(e.getMessage());
      }
      return realClass;
    }
    
    // 说明这个类没有被代理，直接返回即可
    return targetClass;
  }

  
  /**
   * 该私有方法用于分析指定方法的返回值信息
   * @param serviceMethodReop
   * @param ctMethod
   */
  private void analysisMethodReturnParameter(ServicableMethodInfo serviceMethodReop , Method ctMethod) throws  NoSuchMethodException {
    boolean isCollectionInterface = false;
    Class<?> returnClass = ctMethod.getReturnType();
    
    // 分析返回值是否是集合性质，即是说实现了java.util.Collection接口
    Class<?> collectionClass = Collection.class;
    Class<?> iterableClass = Iterable.class;
    if(returnClass == null || returnClass == Void.class) {
      returnClass =  Void.class;
    }
    isCollectionInterface = collectionClass.isAssignableFrom(returnClass) || iterableClass.isAssignableFrom(returnClass);
    
    // 如果当前返回值信息是集合性质，则分析其中的泛型，作为返回值类型
    if(isCollectionInterface) {
      Class<?> genericClazz = null;
      // 接着试图取得泛型
      Type genericType = ctMethod.getGenericReturnType();
      // 如果条件成立才说明在反射类型描述中存在泛型信息
      if(genericType instanceof ParameterizedType) {
        ParameterizedType pt = (ParameterizedType) genericType;
        genericClazz = (Class<?>)pt.getActualTypeArguments()[0];
      }
      
      serviceMethodReop.setReturnCollection(true);
      serviceMethodReop.setReturnClass(genericClazz);
    } else if(returnClass == Void.class) {
      serviceMethodReop.setReturnCollection(false);
      serviceMethodReop.setReturnClass(Void.class);
    } else {
      serviceMethodReop.setReturnCollection(false);
      serviceMethodReop.setReturnClass(returnClass);
    }
  }
}
