package com.bizunited.platform.core.service.serviceable.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.NebulaServiceMethod.ScopeType;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodInfo;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodProperty;
import com.bizunited.platform.core.service.serviceable.template.ValueMappingTemplate;
import com.google.common.collect.Lists;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * nebula业务构建平台服务源查询、管理、调用的相关服务实现
 * @author yinwenjie
 */
@Service("KuiperServiceMethodServiceImpl")
public class ServicableMethodServiceImpl implements ServicableMethodService {
  /**
   * 被缓存的服务源信息在这里。
   */
  private static Map<String , ServicableMethodInfo> caches = new ConcurrentHashMap<>();
  
  private static Logger LOGGER = LoggerFactory.getLogger(ServicableMethodService.class);
  
  @Autowired
  private ApplicationContext applicationContext;
  
  @Override
  public void create(ServicableMethodInfo serviceMethod) {
    Validate.notNull(serviceMethod , "将要缓存的服务源信息，不能为空!!");
    Validate.notBlank(serviceMethod.getName() , "将要缓存的服务源信息，必须有为一个name信息");
    caches.put(serviceMethod.getName(), serviceMethod);
  }

  @Override
  public Page<JSONObject> findByConditions(Pageable pageable, String name,
      String description, String interfaceName, String simpleMethodName, String returnClassName,
      String usedScope) {
    
    if(caches == null) {
      return new PageImpl<>(Lists.newArrayList());
    }
    Collection<ServicableMethodInfo> servicableMethodInfos = caches.values();
    
    // 开始根据条件过滤
    List<ServicableMethodInfo> methods = servicableMethodInfos.stream()
      // 名字模糊查询
      .filter(item -> {
        if(StringUtils.isNotBlank(name)) {
          return StringUtils.contains(item.getName(), name);
        }
        return true;
      })
      // 根据描述信息查询
      .filter(item -> {
        if(StringUtils.isNotBlank(description)) {
          return StringUtils.contains(item.getDescription(), description);
        }
        return true;
      })
      // 根据类/接口名查询
      .filter(item -> {
        if(StringUtils.isNotBlank(interfaceName)) {
          return StringUtils.contains(item.getInterfaceName(), interfaceName);
        }
        return true;
      })
      // 简单方法名查询
      .filter(item -> {
        if(StringUtils.isNotBlank(simpleMethodName)) {
          return StringUtils.contains(item.getSimpleMethodName(), simpleMethodName);
        }
        return true;
      })
      // 返回类型查询
      .filter(item -> {
        if(StringUtils.isNotBlank(returnClassName)) {
          return StringUtils.contains(item.getReturnClass().getName(), returnClassName);
        }
        return true;
      })
      .filter(item -> {
        if(StringUtils.isBlank(usedScope)) {
          return true;
        }
        if(StringUtils.equalsIgnoreCase(usedScope, "write")) {
          return item.getUsedScope() == ScopeType.WRITE;
        } else if(StringUtils.equalsIgnoreCase(usedScope, "read")) {
          return item.getUsedScope() == ScopeType.READ;
        }
        return true;
      })
      .collect(Collectors.toList());


    List<JSONObject> results = buildResults(methods);
    return new PageImpl<>(results);
  }
  
  private List<JSONObject> buildResults(List<ServicableMethodInfo> servicableMethodInfos) {
    List<JSONObject> results = new ArrayList<>();
    servicableMethodInfos.forEach(item -> {
      JSONObject jsonitem = this.findDetailJson(item);
      results.add(jsonitem);
    });
    return results;
  }
  /**
   * 此方法用于把servicableMethodInfo对象转化为json格式
   * @author ycd
   */
  private JSONObject findDetailJson(ServicableMethodInfo servicableMethodInfo){
    JSONObject jsonitem = new JSONObject();
    jsonitem.put("name", servicableMethodInfo.getName());
    jsonitem.put("description", servicableMethodInfo.getDescription());
    jsonitem.put("simpleMethodName", servicableMethodInfo.getSimpleMethodName());
    jsonitem.put("interfaceName", servicableMethodInfo.getInterfaceName());
    jsonitem.put("returnCollection", servicableMethodInfo.isReturnCollection());
    jsonitem.put("returnClass", servicableMethodInfo.getReturnClass().getName());
    jsonitem.put("usedScope", servicableMethodInfo.getUsedScope());
    // properties信息
    List<ServicableMethodProperty> servicableMethodProperties = servicableMethodInfo.getProperties();
    if(servicableMethodProperties != null) {
      JSONArray itemps = new JSONArray();
      servicableMethodProperties.forEach(itemp -> {
        JSONObject itempObject = new JSONObject();
        itempObject.put("propertyIndex", itemp.getPropertyIndex());
        itempObject.put("parameter", itemp.getParamType().getType().getName());
        itempObject.put("hasParamAnnotation", itemp.isHasParamAnnotation());
        itempObject.put("annonQualifiedName", itemp.getAnnonQualifiedName());
        itemps.add(itempObject);
      });
      jsonitem.put("properties", itemps);
    } else {
      jsonitem.put("properties", new JSONArray());
    }
    return jsonitem;
  }
  @Override
  public ServicableMethodInfo findDetailsByName(String name) {
    return caches.get(name);
  }

  @Override
  public JSONObject findDetailJsonByName(String name) {
    ServicableMethodInfo servicableMethodInfo = this.findDetailsByName(name);
    if (servicableMethodInfo == null){
      return null;
    }
    JSONObject jsonObject = this.findDetailJson(servicableMethodInfo);
    return jsonObject;
  }

  @Override
  public Object invoke(String serviceName, InvokeParams invokeParams) throws InvokeProxyException {
    Validate.notBlank(serviceName , "进行服务源调用时，服务源唯一标识名称必须传入!!");
    InvokeParams currentInvokeParams = invokeParams;
    if(currentInvokeParams == null) {
      currentInvokeParams = new InvokeParams();
    }
    ServicableMethodInfo servicableMethodInfo = this.findDetailsByName(serviceName);
    Validate.notNull(servicableMethodInfo , "没发现指定的服务源信息，请检查!!");
    
    /*
     * 在进行入参边界校验后，代理调用的处理方式如下所示：
     * 1、根据serviceName取出当前存储的服务源信息信息。并验证合法性，找到IOC容器中相关目标对象（spring bean）
     * 2、根据服务源值映射处理器的信息，查询和组装映射信息（如果当前服务源没有入参信息，则不需要组装）
     * 3、进行调用
     * */
    // 1、========
    String methodParentClassName = servicableMethodInfo.getInterfaceName();
    Class<?> methodParentClass = null;
    try {
      methodParentClass = applicationContext.getClassLoader().loadClass(methodParentClassName);
    } catch (ClassNotFoundException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
    Object targetBean = this.applicationContext.getBean(methodParentClass);
    Method targetMethod = findTargetMethod(methodParentClass, serviceName);
    
    // 2、=======
    List<Object> parameterValues = new ArrayList<>();
    List<ServicableMethodProperty> servicableMethodProperties = servicableMethodInfo.getProperties();
    Class<?> targetClass = targetMethod.getDeclaringClass();
    for(int index = 0 ; servicableMethodProperties != null && index < servicableMethodProperties.size() ; index++) {
      ServicableMethodProperty property = servicableMethodProperties.get(index);
      String qualifiedName = property.getAnnonQualifiedName();
      ValueMappingTemplate valueMappingTemplate = property.getValueMappingTemplate();
      Parameter parameter = property.getParamType();
      
      // 开始映射值，如果条件成立，说明从上下围的K-V信息中取值，否则从BODY中取值
      Object mappingValue = valueMappingTemplate.mapping(targetClass, targetMethod, parameter, index, qualifiedName, currentInvokeParams);
      parameterValues.add(mappingValue);
    }
    
    // 3、=====
    Object result = null;
    try {
      result = targetMethod.invoke(targetBean, parameterValues.toArray(new Object[] {}));
      return result;
    } catch (RuntimeException e) {
      throw e;
    } catch (InvocationTargetException e) {
      Throwable throwable = e.getTargetException();
      if(throwable != null && throwable instanceof RuntimeException) {
        throw (RuntimeException)throwable;
      } else {
        throw new IllegalArgumentException(e);
      }
    } catch(Throwable e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e);
    }
  }
  
  private Method findTargetMethod(Class<?> methodParentClass , String serviceName) {
    // 寻找指定的方法，通过service method name
    Method[] targetMethods = methodParentClass.getDeclaredMethods();
    Validate.notNull(targetMethods , "未在spring ioc proxy找到指定service method name的服务源!!");
    Method targetMethod = null;
    for (Method methodItem : targetMethods) {
      NebulaServiceMethod anno = methodItem.getAnnotation(NebulaServiceMethod.class);
      if(anno == null) {
        continue;
      }
      if(StringUtils.equals(anno.name(), serviceName)) {
        targetMethod = methodItem;
        break;
      }
    }
    Validate.notNull(targetMethod , "未在spring ioc proxy找到指定service method name的服务源!!");
    return targetMethod;
  }

  @Override
  public Object invoke(String serviceName, Object[] values) {
    Validate.notBlank(serviceName , "进行服务源调用时，服务源唯一标识名称必须传入!!");
    ServicableMethodInfo servicableMethodInfo = this.findDetailsByName(serviceName);
    Validate.notNull(servicableMethodInfo , "没发现指定的服务源信息，请检查!!");
    Object[] currentValues = values;
    if(currentValues == null) {
      currentValues = new Object[] {};
    }
    
    /*
     * 在进行了边界校验后：
     * 直接调用方式，相对于代理调用方式要简单很多，处理过程如下：
     * 1、根据serviceName取出当前存储的服务源信息信息。并验证合法性，找到IOC容器中相关目标对象（spring bean）
     * 2、进行调用
     * */
    // 1、========
    String methodParentClassName = servicableMethodInfo.getInterfaceName();
    Class<?> methodParentClass = null;
    try {
      methodParentClass = applicationContext.getClassLoader().loadClass(methodParentClassName);
    } catch (ClassNotFoundException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
    Object targetBean = this.applicationContext.getBean(methodParentClass);
    
    // 2、=====
    Method targetMethod = findTargetMethod(methodParentClass, serviceName);
    Object result = null;
    try {
      result = targetMethod.invoke(targetBean, currentValues);
    } catch (RuntimeException e) {
      throw e;
    } catch (InvocationTargetException e) {
      Throwable throwable = e.getTargetException();
      if(throwable != null && throwable instanceof RuntimeException) {
        throw (RuntimeException)throwable;
      } else {
        throw new IllegalArgumentException(e);
      }
    } catch(Throwable e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e);
    }
    return result;
  }
}