package com.bizunited.platform.core.service.serviceable.template;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.security.Principal;
import java.time.temporal.Temporal;
import java.util.Date;


import com.alibaba.fastjson.JSONException;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.core.service.invoke.InvokeProxyContext;

/**
 * 处理调用时传来的data-json结构
 * @author yinwenjie
 */
@Component
public class ObjectDataValueMappingTemplate extends AbstractObjectValueMappingTemplate {

  @Override
  public boolean match(Class<?> targetClass, Method targetMethod, Parameter parameter, int index) {
    /*
     * 满足当前模板的具体处理条件是：
     * 1、下层实际入参不是一个数组
     * 2、下层实际入参没有ServiceMethodParam注解
     * 3、并不是诸如InvokeProxyContext、Principal这样的保留类
     * 4、不是java中的八大基础类型
     * */
    // 1、========
    Class<?> parameterType = parameter.getType();
    boolean isArray = parameterType.isArray();
    if(isArray) {
      return false;
    }
    
    // 2、========
    ServiceMethodParam serviceMethodParam = parameter.getAnnotation(ServiceMethodParam.class);
    if(serviceMethodParam != null) {
      return false;
    }
    // 被支持的目标类型不能是JSON类型
    if(JSON.class.isAssignableFrom(parameterType)) {
      return false;
    }
    
    // 3、========
    if(parameterType == InvokeProxyContext.class || parameterType == InvokeParams.class || Principal.class.isAssignableFrom(parameterType)
        || Pageable.class.isAssignableFrom(parameterType) || Date.class.isAssignableFrom(parameterType) 
        || Number.class.isAssignableFrom(parameterType) || Temporal.class.isAssignableFrom(parameterType)) {
      return false;
    }
    
    // 4、========
    if(parameterType.isPrimitive()) {
      return false;
    }
    return true;
  }

  @Override
  public Object mapping(Class<?> targetClass, Method targetMethod, Parameter parameter, int index, String parameterName , InvokeParams invokeParams) {
    // 直接返回invokeParams中关于json data的信息即可
    JSONObject jsonObject = invokeParams.getJsonData();
    if(jsonObject == null) {
      return null;
    }
    Class<?> parameterClass = parameter.getType();
    try {
      return this.mappingObject(jsonObject, parameterClass);
    }catch (RuntimeException e){
      String[] megs = e.getMessage().split(",");
      throw new JSONException(String.format("【%s】参数类型转换错误，请检查", megs[megs.length-2]));
    }
  }
}