package com.bizunited.platform.core.service.serviceable.template.base;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.service.invoke.InvokeParams;

/**
 * 处理Integer类型和int基础类型的类转换和值映射
 * @author yinwenjie
 */
@Component
public class IntegerValueMappingTemplate extends AbstractPrimitiveValueMappingTemplate {

  @Override
  public boolean match(Class<?> targetClass, Method targetMethod, Parameter parameter, int index) {
    Class<?> parameterType = parameter.getType();
    boolean isArray = parameterType.isArray();
    
    // 如果end-point入参不是数组，且当前入参类型为Integer类型或者为int，则该值映射模板支持映射体系
    if(!isArray && (parameterType == Integer.class || parameterType == Integer.TYPE)) {
      Validate.isTrue(super.checkServiceMethodParamAnnotation(parameter) , "设定服务源时Integer/int类型的参数[%s]，必须携带ServiceMethodParam注解" , targetMethod.toString());
      return true;
    }
    // 如果end-point入参是数组，且当前入参原始类型为Integer类型或者为int，则该值映射模板支持映射体系
    if(isArray && (parameterType.getComponentType() == Integer.class || parameterType.getComponentType() == Integer.TYPE)) {
      Validate.isTrue(super.checkServiceMethodParamAnnotation(parameter) , "设定服务源时Integer/int类型的参数[%s]，必须携带ServiceMethodParam注解" , targetMethod.toString());
      return true;
    }
    
    return false;
  }

  @Override
  protected Object mappingNoArrayToNoArray(Class<?> targetClass, Method targetMethod, Parameter parameter, int index, String parameterName, InvokeParams invokeParams, Object value) {
    Class<?> parameterType = parameter.getType();
    boolean isTypePrimitive = parameterType.isPrimitive();
    boolean isValuePrimitive = value.getClass().isPrimitive();
    
    // 参数定义是基础类型，入参值不是基础类型
    if(!isTypePrimitive && isValuePrimitive) {
      return Integer.valueOf((int)value);
    } 
    // 参数定义不是基础类型，如参值是基础类型
    else if(isTypePrimitive && !isValuePrimitive){
      return ((Integer)value).intValue();
    }
    
    // 两者都是基础类型，或者两者都不是基础类型，就直接返回值即可
    return value;
  } 

  @Override
  protected Object mappingArrayToArray(Class<?> targetClass, Method targetMethod, Parameter parameter, int index, String parameterName, InvokeParams invokeParams, Object values) {
    Class<?> parameterType = parameter.getType();
    boolean isTypePrimitive = parameterType.isPrimitive();
    boolean isValuePrimitive = values.getClass().isPrimitive();
    
    // 参数定义不是基础类型，入参值是基础类型
    if(!isTypePrimitive && isValuePrimitive) {
      return ArrayUtils.toObject((int[])values);
    } 
    // 参数定义是基础类型，入参值不是基础类型
    else if(isTypePrimitive && !isValuePrimitive) {
      return ArrayUtils.toPrimitive((Integer[])values);
    }
    
    // 两者都是基础类型，或者两者都不是基础类型，就直接返回值即可
    return values;
  }
}
