package com.bizunited.platform.core.aspect;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.entity.log.LoggerInfoEntity;
import com.bizunited.platform.core.entity.log.LoggerTemplateEntity;
import com.bizunited.platform.core.repository.log.LoggerTemplateEntityRepository;
import com.bizunited.platform.core.service.LoggerInfoEntityService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;

@Component
public class NebulaLoggerHandle {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(NebulaLoggerHandle.class);
  private static final String SPECIAL_EXPRESSION = "$[?(@.id = '";
  @Autowired
  private LoggerTemplateEntityRepository loggerTemplateEntityRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private LoggerInfoEntityService loggerInfoService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private ApplicationContext applicationContext;
  
  @Transactional
  public Object handLogger(MethodSignature methodSignature , LoggerTemplateEntity target , ProceedingJoinPoint point) {
    /*
     * 首先，处理分为两种场景：
     * 场景1：不记录详情变化，只记录事件描述：只根据对应的日支模板中的expression表达式，完成事件基本信息和描述的记录
     * 场景2：记录数据详情变化：除了以上工作以外，还需要在compare属性为true时
     * 进行详情查询、对比，并完成数据变化情况的记录
     *
     * 处理过程为：
     * 1、验证日志模板信息的边界（包括明确操作账号和操作时间），
     * 如果compare属性为true，则边界校验还需要确认预查询和后置查询的配置信息；
     * 2、根据preQueryComponent、preQueryMethod、preReturnFilter的信息
     * 进行预查询，以便获得在正式操作前，对象的数据信息
     * 3、开启事务后，正式进行方法调用
     * 4、根据lastQueryComponent、lastQueryMethod、lastReturnFilter的信息
     * 进行后置查询，以便获得正式操作后，对象的数据信息
     * 5、根据expression表单式，以及可能的预查询、后置查询结果，构建日志描述信息并保存
     * 6、如果compare属性为true，则进行数据详细变化明细的比较（JSON形式），并进行保存
     * */

    // 1、========
    this.validate(target);
    String expression = target.getExpression();
    Integer methodParamIndex = target.getMethodParamIndex();
    if(methodParamIndex == null) {
      methodParamIndex = 0;
    }
    Boolean compare = target.getCompare();
    String account = this.getAccount();
    if(StringUtils.isBlank(account)) {
      account = "匿名用户";
    }
    Date opTime = new Date();

    // 2、========
    // 根据methodParamIndex获得调用的入参信息，如果有methodParamProperty的设定，还要进行属性的引用值获取
    String preQueryComponent = target.getPreQueryComponent();
    String preQueryMethod = target.getPreQueryMethod();
    String preReturnFilter = target.getPreReturnFilter();
    Object inputParam = point.getArgs()[methodParamIndex];
    Validate.notNull(inputParam , "错误的入参设定，请检查方法调用的第[%d]个入参信息，" , methodParamIndex);
    String methodParamProperty = target.getMethodParamProperty();
    if(StringUtils.isNotBlank(methodParamProperty)) {
      inputParam = this.analysisValueByField(inputParam, methodParamProperty);
    }
    Object preObject = null;
    if(StringUtils.isNotBlank(preQueryComponent) && StringUtils.isNotBlank(preQueryMethod)
            && inputParam != null) {
      preObject = this.invokeComponent(preQueryComponent, preQueryMethod, preReturnFilter, inputParam);
    }

    // 3、=======
    Object obj = null;
    Object lastObject = null;
    try {
      obj = point.proceed(point.getArgs());
      // 再次试图取唯一标识属性
      this.loggerTemplateEntityRepository.flush();
      inputParam = point.getArgs()[methodParamIndex];
      if(StringUtils.isNotBlank(methodParamProperty)) {
        inputParam = this.analysisValueByField(inputParam, methodParamProperty);
      }

      // 4、=========
      String lastQueryComponent = target.getLastQueryComponent();
      String lastQueryMethod = target.getLastQueryMethod();
      String lastReturnFilter = target.getLastReturnFilter();
      if(StringUtils.isNotBlank(lastQueryComponent) && StringUtils.isNotBlank(lastQueryMethod)
              && inputParam != null) {
        lastObject = this.invokeComponent(lastQueryComponent, lastQueryMethod, lastReturnFilter, inputParam);
      }

      // 5、=========
      String expressionValue = this.analysisExpression(expression, account, opTime, preObject, lastObject);
      // 开始保存基本的日志信息
      String type = target.getType();
      String module = target.getModule();
      LoggerInfoEntity loggerInfo = new LoggerInfoEntity();
      loggerInfo.setAccount(account);
      loggerInfo.setModule(module);
      loggerInfo.setType(type);
      loggerInfo.setMsg(expressionValue);
      // 优先从last中取得
      // 再次从pre中取得
      // 最后在视图从方法入参取得
      Object pkValue = null;
      if(lastObject != null) {
        pkValue = this.analysisValueByField(lastObject, "id");
      }
      if(pkValue == null && preObject != null) {
        pkValue = this.analysisValueByField(preObject, "id");
      }
      if(pkValue == null) {
        pkValue = inputParam;
      }
      Validate.notNull(pkValue , "未找到指定的数据唯一值依据(一般不可能出现)，请检查!!");
      loggerInfo.setOpId(pkValue.toString());
      loggerInfo.setOpTime(opTime);
      loggerInfo.setCreateAccount(SecurityUtils.getUserAccount());
      loggerInfo.setCreateTime(new Date());
      // 6、=======
      // TODO 在该方法后续需要改变为异步方式新增日志信息
      if(compare) {
        this.doCompare(loggerInfo, preObject, lastObject, module, pkValue);
      }
      this.loggerInfoService.create(loggerInfo);
    } catch (RuntimeException | Error ex) {
      throw ex;
    } catch (Throwable thr) {
      Rethrower.rethrow(thr);
    }
    return obj;
  }

  private void validate(LoggerTemplateEntity target) {
    String code = target.getCode();
    Validate.notBlank(code , "日志模板的业务编码必须填写(code)");
    String type = target.getType();
    Validate.notBlank(type , "日志操作类型必须填写[%s]!!" , code);
    String module = target.getModule();
    Validate.notBlank(module , "日志操作模块必须设定[%s]!!" , code);
    String methodName = target.getMethodName();
    Validate.notBlank(methodName , "必须设定日志的操作方法[%s]" , code);
    String expression = target.getExpression();
    Validate.notBlank(expression , "日志表达式必须填写[%s]" , code);
    Boolean compare = target.getCompare();
    // 如果要进行明细比较，那么还必须进行保证前置和后置查询都进行了设定
    String preQueryComponent = target.getPreQueryComponent();
    String preQueryMethod = target.getPreQueryMethod();
    String preReturnFilter = target.getPreReturnFilter();
    String lastQueryComponent = target.getLastQueryComponent();
    String lastQueryMethod = target.getLastQueryMethod();
    String lastReturnFilter = target.getLastReturnFilter();
    if(compare) {
      Validate.isTrue(StringUtils.isNotBlank(preQueryComponent) && StringUtils.isNotBlank(preQueryMethod)
              && StringUtils.isNotBlank(preReturnFilter) , "日志模板[%s]设定了需要进行变化明细比对，但是其前置查询设定不完整，请检查!!" , code);
      Validate.isTrue(StringUtils.isNotBlank(lastQueryComponent) && StringUtils.isNoneBlank(lastQueryMethod)
              && StringUtils.isNotBlank(lastReturnFilter) , "日志模板[%s]设定了需要进行变化明细比对，但是其后置查询设定不完整，请检查!!" , code);
    }
  }

  private void doCompare(LoggerInfoEntity loggerInfo , Object preObject , Object lastObject , String module , Object pkValue){
    JSONObject preJsonObject = null;
    if(preObject != null) {
      preJsonObject = (JSONObject)JSONObject.toJSON(preObject);
    }
    JSONObject lastJsonObject = null;
    if(lastObject != null) {
      lastJsonObject = (JSONObject)JSONObject.toJSON(lastObject);
    }
    JSONArray compareResults = analysisProperties(preJsonObject, lastJsonObject);
    if(compareResults == null) {
      compareResults = new JSONArray();
    }
    // 开始进行对比文件信息的保存（对文件路径进行确认）
    byte[] contents = null;
    try {
      contents = compareResults.toJSONString().getBytes("UTF-8");
    } catch (UnsupportedEncodingException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage() , e);
    }
    String fileName = StringUtils.join("_op_log_" , module , "_" , pkValue , ".txt");
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String relativePath = StringUtils.join("/oplog/" , folderName , "/" , (new Random().nextInt(100) % 10));
    venusFileService.saveFile(relativePath, fileName, fileName, contents);
    loggerInfo.setRelativePath(relativePath);
    loggerInfo.setFileName(fileName);
  }

  /**
   * 该私有方法可以递归分析两个JSON对象结构的字段信息差异
   * 分析得到哪些字段修改了，哪些字段新增了，哪些字段信息删除了
   * @param preObject
   * @param lastObject
   * @return
   */
  private static JSONArray analysisProperties(JSONObject preObject , JSONObject lastObject) {
    /*
     * 处理过程为：
     * 1、首先做基础属性判定
     *   1.1、首先判定新增的字段，既是preObject中存在，但是lastObject中不存在的
     *   1.2、然后再判定删除的字段，既是preObject中不存在，但是lastObject中存在的
     *   1.3、接着判定值修改的字段，
     * 2、然后做jsonObject中引用对象的递归处理，并记录处理结果
     *   2.1、首先判定新增的引用对象，既是preItem中不存在，但是lastItem中存在
     *   2.2、然后判定删除的引用对象，既是preItem中存在，但是lastItem中不存在
     *   2.3、接着判定引用对象内部进行修改的
     * 3、最后做jsonObject中引用集合的递归处理，并记录处理结果
     *
     * */
    // 1、=========
    Set<Entry<String, Object>> preEntris = preObject == null?new HashSet<>():preObject.entrySet();
    final Set<Entry<String, Object>> preProperties = preEntris.stream().filter(item -> {
      Object value = item.getValue();
      return value != null && !(value instanceof JSONObject || value instanceof JSONArray);
    }).collect(Collectors.toSet());
    Set<String> preKeys = preProperties.stream().map(Entry::getKey).collect(Collectors.toSet());
    if(preKeys == null) {
      preKeys = new HashSet<>();
    }
    Set<Entry<String, Object>> lastEntris = lastObject == null?new HashSet<>():lastObject.entrySet();
    final Set<Entry<String, Object>> lastProperties = lastEntris.stream().filter(item -> {
      Object value = item.getValue();
      return value != null && !(value instanceof JSONObject || value instanceof JSONArray);
    }).collect(Collectors.toSet());
    Set<String> lastKeys = lastProperties.stream().map(Entry::getKey).collect(Collectors.toSet());
    if(lastKeys == null) {
      lastKeys = new HashSet<>();
    }

    // 1.1、=======
    JSONArray properties = new JSONArray();
    Set<String> createKeys = Sets.difference(lastKeys , preKeys);
    for (String createKey : createKeys) {
      JSONObject createProperty = new JSONObject();
      Object createPropertyValue = lastObject.get(createKey);
      createProperty.put("name", createKey);
      createProperty.put("changeType", "ADD");
      createProperty.put("beforeValue", null);
      createProperty.put("afterValue", createPropertyValue == null?null:createPropertyValue.toString());
      properties.add(createProperty);
    }
    // 1.2、=======
    Set<String> deleteKeys = Sets.difference(preKeys, lastKeys);
    for (String deleteKey : deleteKeys) {
      JSONObject deleteProperty = new JSONObject();
      Object deletePropertyValue = lastObject.get(deleteKey);
      deleteProperty.put("name", deleteKey);
      deleteProperty.put("changeType", "DELETE");
      deleteProperty.put("beforeValue", null);
      deleteProperty.put("afterValue", deletePropertyValue == null?null:deletePropertyValue.toString());
      properties.add(deleteProperty);
    }
    // 1.3、======
    Set<String> maybeModifyKeys = Sets.intersection(preKeys, lastKeys);
    for (String maybeModifyKey : maybeModifyKeys) {
      String preValue = preObject == null?null:preObject.getString(maybeModifyKey);
      String lastValue = lastObject == null?null:lastObject.getString(maybeModifyKey);
      // 如果条件成立，就任务值发生了变化
      if(!StringUtils.equals(preValue, lastValue)) {
        JSONObject modifyProperty = new JSONObject();
        modifyProperty.put("name", maybeModifyKey);
        modifyProperty.put("changeType", "MODIFY");
        modifyProperty.put("beforeValue", preValue);
        modifyProperty.put("afterValue", lastValue);
        properties.add(modifyProperty);
      }
    }

    // 2、======
    final Set<Entry<String, Object>> preItems = preEntris.stream().filter(item -> {
      Object value = item.getValue();
      return value != null && value instanceof JSONObject;
    }).collect(Collectors.toSet());
    preKeys.clear();
    preKeys = preItems.stream().map(Entry::getKey).collect(Collectors.toSet());
    final Set<Entry<String, Object>> lastItems = lastEntris.stream().filter(item -> {
      Object value = item.getValue();
      return value != null && value instanceof JSONObject;
    }).collect(Collectors.toSet());
    lastKeys.clear();
    lastKeys = lastItems.stream().map(Entry::getKey).collect(Collectors.toSet());
    // 2.1、======
    Set<String> createItemKeys = Sets.difference(lastKeys , preKeys);
    for (String createItemKey : createItemKeys) {
      if(lastObject == null) {
        continue;
      }
      JSONObject createItem = new JSONObject();
      JSONObject createItemObject = lastObject.getJSONObject(createItemKey);
      if(createItemObject != null && createItemObject.getString("id") != null) {
        createItem.put("name", createItemKey);
        createItem.put("changeType", "ADD");
        createItem.put("values", analysisProperties(null, createItemObject));
        properties.add(createItem);
      }
    }
    // 2.2、======
    Set<String> deleteItemKeys = Sets.difference(preKeys, lastKeys);
    for (String deleteItemKey : deleteItemKeys) {
      if(preObject == null) {
        continue;
      }
      JSONObject deleteItem = new JSONObject();
      JSONObject deleteItemObject = preObject.getJSONObject(deleteItemKey);
      if(deleteItemObject != null && deleteItemObject.getString("id") != null) {
        deleteItem.put("name", deleteItemKey);
        deleteItem.put("changeType", "DELETE");
        deleteItem.put("beforeValue", null);
        deleteItem.put("values", analysisProperties(deleteItemObject, null));
        properties.add(deleteItem);
      }
    }
    // 2.3、======
    Set<String> maybeModifyItemKeys = Sets.intersection(preKeys, lastKeys);
    for (String maybeModifyItemKey : maybeModifyItemKeys) {
      JSONObject preValue = preObject == null?new JSONObject():preObject.getJSONObject(maybeModifyItemKey);
      JSONObject lastValue = lastObject == null?new JSONObject():lastObject.getJSONObject(maybeModifyItemKey);
      // 如果条件成立，就任务值发生了变化
      if(!StringUtils.equals(preValue.toJSONString(), lastValue.toJSONString())) {
        JSONObject modifyProperty = new JSONObject();
        modifyProperty.put("name", maybeModifyItemKey);
        modifyProperty.put("changeType", "MODITY");
        modifyProperty.put("values", analysisProperties(preValue, lastValue));
        properties.add(modifyProperty);
      }
    }

    // 3、=======
    final Set<Entry<String, Object>> preArrays = preEntris.stream().filter(item -> {
      Object value = item.getValue();
      return value != null && value instanceof JSONArray;
    }).collect(Collectors.toSet());
    preKeys.clear();
    preKeys = preArrays.stream().map(Entry::getKey).collect(Collectors.toSet());
    final Set<Entry<String, Object>> lastArrays = lastEntris.stream().filter(item -> {
      Object value = item.getValue();
      return value != null && value instanceof JSONArray;
    }).collect(Collectors.toSet());
    lastKeys.clear();
    lastKeys = lastArrays.stream().map(Entry::getKey).collect(Collectors.toSet());
    // 3.1、======
    Set<String> createArrayKeys = Sets.difference(lastKeys , preKeys);
    for (String createArrayKey : createArrayKeys) {
      if(lastObject == null) {
        continue;
      }
      JSONObject createItem = new JSONObject();
      JSONArray createItemObjects = lastObject.getJSONArray(createArrayKey);
      if(createItemObjects != null && !CollectionUtils.isEmpty(createItemObjects)) {
        createItem.put("name", createArrayKey);
        createItem.put("changeType", "ADD");
        createItem.put("values", analysisArrays(new JSONArray(), createItemObjects));
        properties.add(createItem);
      }
    }
    // 3.2、======
    Set<String> deleteArrayKeys = Sets.difference(preKeys, lastKeys);
    for (String deleteArrayKey : deleteArrayKeys) {
      if(preObject == null) {
        continue;
      }
      JSONObject deleteItem = new JSONObject();
      JSONArray deleteItemObjects = preObject.getJSONArray(deleteArrayKey);
      if(deleteItemObjects != null && !CollectionUtils.isEmpty(deleteItemObjects)) {
        deleteItem.put("name", deleteArrayKey);
        deleteItem.put("changeType", "DELETE");
        deleteItem.put("values", analysisArrays(deleteItemObjects , new JSONArray()));
        properties.add(deleteItem);
      }
    }
    // 2.3、======
    Set<String> maybeModifyArrayKeys = Sets.intersection(preKeys, lastKeys);
    for (String maybeModifyArrayKey : maybeModifyArrayKeys) {
      JSONArray preValue = preObject == null?new JSONArray():preObject.getJSONArray(maybeModifyArrayKey);
      JSONArray lastValue = lastObject == null?new JSONArray():lastObject.getJSONArray(maybeModifyArrayKey);
      // 如果条件成立，就任务值发生了变化
      if(!StringUtils.equals(preValue.toJSONString(), lastValue.toJSONString())) {
        JSONObject modifyProperty = new JSONObject();
        modifyProperty.put("name", maybeModifyArrayKey);
        modifyProperty.put("changeType", "MODITY");
        modifyProperty.put("values", analysisArrays(preValue , lastValue));
        properties.add(modifyProperty);
      }
    }

    // 都已其中的id属性未转换
    return properties;
  }

  /**
   * 该私有方法用于分析两个指定的明细属性对象的差异，并生成一个结果性质的json数组对象返回。
   * @param beforeItems 进行“WRITE”操作前的明细属性信息的详情
   * @param afterItems 进行“WRITE”操作后的明细属性信息的详情
   */
  private static JSONObject analysisArrays(JSONArray beforeItems , JSONArray afterItems) {
    /*
     * 明细的变化情况分为三种：
     * 如果在操作前有明细项，但是操作后没有了明细项，那么说明明细项是删除操作
     * 如果在操作前没有明细项，但是操作后有了明细项，那么说明明细项是添加操作
     * 出去以上两种情况，明细项都是修改操作。而明细项中的所有一般属性字段，只有“修改”这种操作类型
     *
     * 那么操作过程如下：
     * 1、首先构造这个item的基本信息，包括propertyName、changeType等信息
     * 并判定当前明细项是以上哪一种操作类型，并分裂成三个集合：
     * 明细项删除集合、明细项新增集合、明细项修改集合
     *
     * 2、对明细项删除集合中的每一条元素进行变化日志记录操作
     * （包括所有一般性属性和关联属性的变化情况）
     * 3、对明细项新增集合中的每一条元素进行变化日志记录操作
     * 4、对明细项修改集合中的每一条元素进行变化日志记录操作
     *
     * (注意：如果1.3所使用的集合，其中没有任何属性发生变化，那么说明这条明细项没有变化)
     * */

    // 1、======
    JSONObject changeObjects = new JSONObject();
    // 判定的依据就是明细项中的id属性
    Set<Object> beforeItemIdArrays = Sets.newHashSet();
    Set<Object> afterItemIdArrays = Sets.newHashSet();
    if(beforeItems != null) {
      Object beforeItemArrayValue = JSONPath.eval(beforeItems, "$['id']");
      JSONArray beforeItemArray = null;
      if(beforeItemArrayValue instanceof JSONArray) {
        beforeItemArray = (JSONArray)beforeItemArrayValue;
      }
      beforeItemIdArrays = beforeItemArray != null?Sets.newHashSet(beforeItemArray.toArray(new Object[]{})):Sets.newHashSet();
    }
    if(afterItems != null) {
      Object afterItemArrayObject = JSONPath.eval(afterItems, "$['id']");
      JSONArray afterItemArray = null;
      if(afterItemArrayObject instanceof JSONArray) {
        afterItemArray = (JSONArray)afterItemArrayObject;
      }
      afterItemIdArrays = afterItemArray != null?Sets.newHashSet(afterItemArray.toArray(new Object[]{})):Sets.newHashSet();
    }
    // 以下集合为删除的集合
    SetView<Object> deleteIds = Sets.difference(beforeItemIdArrays, afterItemIdArrays);
    Set<JSONObject> deleteItems = Sets.newLinkedHashSet();
    if(deleteIds != null && !deleteIds.isEmpty()) {
      deleteIds.stream().forEach(itemId -> {
        JSONArray items = (JSONArray)JSONPath.eval(beforeItems, SPECIAL_EXPRESSION + itemId + "')]");
        deleteItems.add(items.getJSONObject(0));
      });
    }
    // 以下集合为新增的集合
    SetView<Object> newIds = Sets.difference(afterItemIdArrays , beforeItemIdArrays);
    Set<JSONObject> newItems = Sets.newLinkedHashSet();
    if(newIds != null && !newIds.isEmpty()) {
      newIds.stream().forEach(itemId -> {
        JSONArray items = (JSONArray)JSONPath.eval(afterItems, SPECIAL_EXPRESSION + itemId + "')]");
        newItems.add(items.getJSONObject(0));
      });
    }
    // 以下集合为修改的集合
    SetView<Object> updateIds = Sets.intersection(afterItemIdArrays , beforeItemIdArrays);
    Set<JSONObject> updateItems = Sets.newLinkedHashSet();
    if(updateIds != null && !updateIds.isEmpty()) {
      updateIds.stream().forEach(itemId -> {
        JSONArray items = (JSONArray)JSONPath.eval(afterItems, SPECIAL_EXPRESSION + itemId + "')]");
        updateItems.add(items.getJSONObject(0));
      });
    }

    // 2、=======
    JSONArray datas = new JSONArray();
    if(deleteItems != null && !deleteItems.isEmpty()) {
      for (JSONObject deleteItem : deleteItems) {
        JSONObject deleteChange = new JSONObject();
        String id = deleteItem.getString("id");
        deleteChange.put("id", id);
        deleteChange.put("changeType", "DELETE");
        // 一般性属性的判定
        JSONArray deletePropertiesChanges = analysisProperties(deleteItem, new JSONObject());
        deleteChange.put("properties", deletePropertiesChanges);
        datas.add(deleteChange);
      }
    }

    // 3、=====
    if(newItems != null && !newItems.isEmpty()) {
      for (JSONObject newItem : newItems) {
        JSONObject newChange = new JSONObject();
        String id = newItem.getString("id");
        newChange.put("id", id);
        newChange.put("changeType", "NEW");
        // 一般性属性的判定
        JSONArray newPropertiesChanges = analysisProperties(new JSONObject(), newItem);
        newChange.put("properties", newPropertiesChanges);
        datas.add(newChange);
      }
    }

    // 4、====
    if(updateItems != null && !updateItems.isEmpty()) {
      for (JSONObject updateItem : updateItems) {
        JSONObject updateChange = new JSONObject();
        String id = updateItem.getString("id");
        JSONArray beforeUpdateItems = (JSONArray)JSONPath.eval(beforeItems, SPECIAL_EXPRESSION + id + "')]");
        updateChange.put("id", id);
        updateChange.put("changeType", "MODIFY");
        // 一般性属性的判定
        JSONArray updatePropertiesChanges = analysisProperties(beforeUpdateItems.getJSONObject(0), updateItem);
        updateChange.put("properties", updatePropertiesChanges);
        if(!CollectionUtils.isEmpty(updatePropertiesChanges)) {
          datas.add(updateChange);
        }
      }
    }

    if(!datas.isEmpty()) {
      changeObjects.put("datas", datas);
      return changeObjects;
    }
    return null;
  }

  /**
   * 该方法用于根据spring ioc容器中的方法，得到指定方法的调用结果
   * @param queryComponent 可能是bean的名字，也可能是一个完整的特定类名
   * @param queryMethod
   * @param returnFilter
   * @param inputParam
   * @return
   */
  private Object invokeComponent(String queryComponent , String queryMethod , String returnFilter , Object inputParam) {
    // 被代理（或者没有被代理）的sping bean
    Object targetBean = null;
    // 原始的class，以便找到指定方法名对应的方法
    Class<?> sourceClass = null;
    boolean hasBean = true;
    // 1、=======
    try {
      targetBean = this.applicationContext.getBean(queryComponent);
    } catch(NoSuchBeanDefinitionException e) {
      hasBean = false;
    }
    // 如果条件成立，说明queryComponent填写的信息是一个可以被识别的bean的name
    // 否则，试图通过class name的方式找到bean和相关的class信息
    if(hasBean) {
      sourceClass = ClassUtils.getUserClass(targetBean);
    } else {
      ClassLoader classLoader = ClassUtils.getDefaultClassLoader();
      try {
        sourceClass = classLoader.loadClass(queryComponent);
      } catch (ClassNotFoundException e) {
        LOGGER.error(e.getMessage() , e);
      }
      targetBean = this.applicationContext.getBean(sourceClass);
    }
    Validate.notNull(targetBean , "未找到指定的spring bean对象信息，请检查!!");
    Validate.notNull(sourceClass , "未找到指定的source class信息，请检查!!");
    // 找到指定的方法（如果bean中存在多个方法，则）
    Method targetMethod = BeanUtils.findMethodWithMinimalParameters(sourceClass, queryMethod);
    Validate.notNull(targetMethod , "未找到指定的方法信息，请检查方法名[%s]的设定" , queryMethod);

    // 2、======开始进行调用
    Object value = null;
    try {
      value = targetMethod.invoke(targetBean, inputParam);
    } catch (RuntimeException | InvocationTargetException | IllegalAccessException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
    if(value == null) {
      return null;
    }
    Class<?> sourceUserClass = ClassUtils.getUserClass(value.getClass());
    // 进行重度复制，保证不会因为JPA的特性造成前置查询和后置查询的结果因为实际操作被改变
    Validate.isTrue(!Collection.class.isAssignableFrom(sourceUserClass) , "目前日志操作配置不支持对集合性质的批量操作方法进行日志配置，请重新配置!1");
    Object copyValue = null;
    if(StringUtils.isBlank(returnFilter)) {
      copyValue = this.nebulaToolkitService.copyObjectByWhiteList(value, sourceUserClass , HashSet.class, ArrayList.class);
    } else {
      copyValue = this.nebulaToolkitService.copyObjectByWhiteList(value, sourceUserClass , HashSet.class, ArrayList.class , StringUtils.split(returnFilter ,  ","));
    }
    return copyValue;
  }

  private String getAccount() {
    SecurityContext securityContext = SecurityContextHolder.getContext();
    if(securityContext == null) {
      return null;
    }
    Authentication authentication = securityContext.getAuthentication();
    if(authentication == null) {
      return null;
    }
    return authentication.getName();
  }

  private String analysisExpression(String expression , String account , Date opTime , Object preObject , Object lastObject) {
    SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    String opTimeValue = simpleDateFormat.format(opTime);
    String expressionResult = null;

    // 找到{user}标记进行替换
    expressionResult= StringUtils.replace(expression, "{user}", account);
    // 找到{time}标记进行替换
    expressionResult= StringUtils.replace(expressionResult, "{time}", opTimeValue);
    // 试图找到{pre.XXXX}，并进行替换
    int startIndex = -1;
    int endIndex = -1;
    while((startIndex = StringUtils.indexOf(expressionResult, "{pre.")) != -1) {
      Validate.notNull(preObject, "在日志表达式中发现前置对象要求，但是前置查询得到的结果为null，请检查!!");
      endIndex = StringUtils.indexOf(expressionResult, "}");
      Validate.isTrue(endIndex != -1 , "错误的表达式{pre.#}");
      String preIncludeExpression = StringUtils.substring(expressionResult , startIndex, endIndex+1);
      String preExpression = StringUtils.substring(expressionResult , startIndex+5, endIndex);
      Object preExpressionValue = this.analysisValueByField(preObject, preExpression);
      expressionResult = StringUtils.replace(expressionResult, preIncludeExpression, preExpressionValue ==null?"null":preExpressionValue.toString());
    }
    // 试图找到{last.XXXX}，并进行替换
    while((startIndex = StringUtils.indexOf(expressionResult, "{last.")) != -1) {
      Validate.notNull(lastObject, "在日志表达式中发现后置对象要求，但是后置查询得到的结果为null，请检查!!");
      endIndex = StringUtils.indexOf(expressionResult, "}");
      Validate.isTrue(endIndex != -1 , "错误的表达式{last.#}");
      String preIncludeExpression = StringUtils.substring(expressionResult , startIndex, endIndex+1);
      String preExpression = StringUtils.substring(expressionResult , startIndex+6, endIndex);
      Object preExpressionValue = this.analysisValueByField(lastObject, preExpression);
      expressionResult = StringUtils.replace(expressionResult, preIncludeExpression, preExpressionValue ==null?"null":preExpressionValue.toString());
    }
    return expressionResult;
  }

  /**
   *
   */
  private Object analysisValueByField(Object currentObject , String fieldNames) {
    if(currentObject == null) {
      return null;
    }
    return this.analysisValueByField(currentObject, StringUtils.split(fieldNames, ".") , 0);
  }

  /**
   * 该私有方法，基于按照对象属性顺序给定的一个数组，递归分析出对应的值，如果分析过程出现错误则抛出能看懂的异常
   * 该私有方法不支持数组，如果发现是一个集合，则抛出异常
   * @param currentObject
   * @param fieldNames 有序的一组指定的属性，利于a.b.c.field的指定方式，给定的数组顺序就是a对象下面的b对象下面的c对象，下面的field属性
   * @param index 当前递归正在处理有序数组的第index个索引位
   * @return
   */
  private Object analysisValueByField(Object currentObject , String[] fieldNames , int index) {
    Validate.isTrue(index < fieldNames.length , "错误的属性顺序索引位置!!");
    String currentFieldName = fieldNames[index];
    Class<?> currentObjectClass = currentObject.getClass();
    Validate.isTrue(!Collection.class.isAssignableFrom(currentObjectClass) , "表单式中的对象描述不支持“集合”，请检查!!");
    // 取得这个属性对应的对象
    Object currentValue = null;
    try {
      char[] chars = currentFieldName.toCharArray();
      chars[0] -= 32;
      Method getMethod = currentObjectClass.getMethod("get" + String.valueOf(chars));
      currentValue = getMethod.invoke(currentObject);
    } catch (RuntimeException | InvocationTargetException | NoSuchMethodException | IllegalAccessException e) {
      LOGGER.error(e.getMessage());
      throw new IllegalArgumentException(e.getMessage());
    }

    // 如果条件成立，说明不需要再递归找下去，该属性的值就是返回值
    if(index == fieldNames.length - 1 || currentValue == null) {
      return currentValue;
    }
    // 如果以上条件不成立，则需要继续向下递归
    return this.analysisValueByField(currentValue, fieldNames, index+1);
  }

  /**
   * 重新抛出异常
   */
  private static class Rethrower {
    public static void rethrow(final Throwable exception) {
      class CheckedExceptionRethrower<T extends Throwable> {
        @SuppressWarnings("unchecked")
        private void rethrow(Throwable exception) throws T {
          throw (T) exception;
        }
      }
      new CheckedExceptionRethrower<RuntimeException>().rethrow(exception);
    }
  }
}
