package com.bizunited.platform.core.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.DataSourceTableEntity;
import com.bizunited.platform.core.service.DataSourceTableService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * @Author: Paul Chan
 * @Date: 2019-04-15 11:11
 * @Description: 数据表功能业务模型的MVC Controller层实现，基于HTTP Restful风格
 */
@RestController
@RequestMapping("/v1/nebula/dataSourceTables")
public class DataSourceTableController extends BaseController {

  private static final  Logger LOGGER = LoggerFactory.getLogger(DataSourceTableController.class);
  
  @Autowired
  private DataSourceTableService dataSourceTableService;

  /**
   * @ApiOperation(value = "更新数据表，更新字断不能全部为空")
   * @param dataSourceTable
   * @return
   */
  @PatchMapping("")
  @ApiOperation(value = "更新数据表", notes = "更新数据表，更新字断不能全部为空")
  public ResponseModel update(@RequestBody @ApiParam("更新实体") DataSourceTableEntity dataSourceTable){
    try {
      DataSourceTableEntity table = dataSourceTableService.update(dataSourceTable);
      return this.buildHttpResultW(table);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   *分页查询的数据表
   * @param pageable
   * @return
   */
  @GetMapping(value="/findByConditions")
  @ApiOperation(value = "分页查询数据表", notes = "dataSourceId为空则默认为主数据源,页数参数为page，第一页从0开始，分页大小参数为size（size默认为50）")
  public ResponseModel findByConditions(@PageableDefault(value = 50) Pageable pageable,
                                        @RequestParam(name = "tableType", required = false) @ApiParam(name="tableType" , value="数据表性质(0:非业务表, 1:业务表)") Integer tableType,
                                        @RequestParam(name = "dataSourceId", required = false) @ApiParam(name="dataSourceId" , value="数据源ID") String dataSourceId,
                                        @RequestParam(name = "status", required = false) @ApiParam(name="status" , value="数据表状态(0:禁用，1:启用)") Integer status,
                                        @RequestParam(name = "tableName", required = false) @ApiParam(name="tableName" , value="数据表名") String tableName){
    try {
      Page<DataSourceTableEntity> page = dataSourceTableService.findByConditions(pageable, tableType, dataSourceId, status, tableName);
      return this.buildHttpResultW(page, "dataSource");
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 重新加载数据表
   * @return
   */
  @GetMapping(value="/reload")
  @ApiOperation(value = "重新加载数据表", notes = "重新加载所有数据源的数据表")
  public ResponseModel reloadTables(){
    try {
      this.dataSourceTableService.refresh();
      return this.buildHttpResult();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
}
