package com.bizunited.platform.core.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.RemoteServiceAddressEntity;
import com.bizunited.platform.core.service.RemoteServiceAddressService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * RemoteServiceAddressController 远端调用源地址管理
 *
 * @description:
 * @author: yanwe
 * @date: 06/May/2019 09:54
 */
@RestController
@RequestMapping("/v1/nebula/remoteServiceAddress")
public class RemoteServiceAddressController extends BaseController {

  /** 日志 */
  private static final Logger LOGGER =
      LoggerFactory.getLogger(RemoteServiceAddressController.class);

  @Autowired 
  private RemoteServiceAddressService remoteServiceAddressService;

  /** 分页查询，根据编号，别名 */
  @GetMapping("queryPage")
  @ApiOperation(value = "分页查询，根据编号，别名")
  public ResponseModel findByConditions(
      @RequestParam(required = false) @ApiParam("编号code") String code,
      @RequestParam(required = false) @ApiParam("别名alias") String alias,
      @PageableDefault(50) Pageable pageable) {
    try {
      Page<RemoteServiceAddressEntity> page =
          this.remoteServiceAddressService.findByConditions(code, alias, pageable);
      return this.buildHttpResultW(page);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 测试连接一个新的远端调用源地址
   *
   * @param entity
   * @param testJson
   * @return
   */
  @ApiOperation(value = "测试连接一个新的远端调用源地址")
  @RequestMapping(value = "/valid", method = RequestMethod.POST)
  public ResponseModel valid(
      @ApiParam(name = "entity", value = "指定远端调用源地址基本信息") @RequestParam(name = "entity")
          String entity,
      @ApiParam(name = "testJson", value = "测试远端数据源测试数据") @RequestParam(name = "testJson")
          String testJson) {
    try {
      this.remoteServiceAddressService.valid(entity, testJson);
      return this.buildHttpResult();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 创建一个新的远端调用源地址
   *
   * @param entity 指定远端调用源地址基本信息
   */
  @ApiOperation(value = "创建一个新的远端调用源地址")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(
      @ApiParam(name = "entity", value = "指定远端调用源地址基本信息") @RequestParam(name = "entity")
          String entity,
      @ApiParam(name = "testJson", value = "新增远端数据源测试数据") @RequestParam(name = "testJson")
          String testJson) {
    try {
      RemoteServiceAddressEntity remoteServiceAddressEntity =
          this.remoteServiceAddressService.create(entity, testJson);
      return this.buildHttpResultW(remoteServiceAddressEntity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 更新一个新的远端调用源地址
   *
   * @param entity 指定远端调用源地址基本信息
   */
  @ApiOperation(value = "更新一个新的远端调用源地址")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(
      @ApiParam(name = "entity", value = "指定远端调用源地址基本信息") @RequestParam(name = "entity")
          String entity,
      @ApiParam(name = "testJson", value = "新增远端数据源测试数据") @RequestParam(name = "testJson")
          String testJson) {
    try {
      RemoteServiceAddressEntity remoteServiceAddressEntity =
          this.remoteServiceAddressService.update(entity, testJson);
      return this.buildHttpResultW(remoteServiceAddressEntity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /** 启用远端调用源 */
  @ApiOperation(value = "启用远端调用源")
  @RequestMapping(
      value = "/enable",
      method = {RequestMethod.POST})
  public ResponseModel enable(@RequestParam(name = "id") @ApiParam("id") String id) {
    try {
      RemoteServiceAddressEntity entity = this.remoteServiceAddressService.enable(id);
      return this.buildHttpResultW(entity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /** 禁用远端调用源 */
  @ApiOperation(value = "禁用远端调用源")
  @RequestMapping(
      value = "/disable",
      method = {RequestMethod.POST})
  public ResponseModel disable(@RequestParam(name = "id") @ApiParam("id") String id) {
    try {
      RemoteServiceAddressEntity entity = this.remoteServiceAddressService.disable(id);
      return this.buildHttpResultW(entity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
