package com.bizunited.platform.core.controller.dataview;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.dataview.ExcelExportTemplate;
import com.bizunited.platform.core.service.dataview.model.ExecuteParamModel;
import com.bizunited.platform.core.vo.dataview.DataviewExportExcelFieldVo;
import com.fasterxml.jackson.core.type.TypeReference;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.security.Principal;
import java.util.List;
import java.util.Set;

/**
 * DataViewEntity业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author ywt
 */
@RestController
@RequestMapping("/v1/nebula/dataViews")
public class DataViewController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DataViewController.class);
  
  private static final String DATASOURCE_PARAM_NAME = "dataSource";
  private static final String GROUP_PARAM_NAME = "dataViewGroup";
  private static final String FIELDS_PARAM_NAME = "fields";
  private static final String FILTERS_PARAM_NAME = "filters";
  private static final String SYSTEM_PARAM_NAME = "systemFilters";
  private static final String[] FIELD_FILTERS = {DATASOURCE_PARAM_NAME,GROUP_PARAM_NAME,FIELDS_PARAM_NAME,FILTERS_PARAM_NAME,SYSTEM_PARAM_NAME};
  private static final String[] FIELD_FILTERS_F = new String[]{DATASOURCE_PARAM_NAME,GROUP_PARAM_NAME,FIELDS_PARAM_NAME,FILTERS_PARAM_NAME,"filters.field",SYSTEM_PARAM_NAME};

  @Autowired
  private DataViewService dataViewEntityService;
  @Autowired
  private ExcelExportTemplate excelExportTemplate;

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建dataViewEntity中的基本信息以外，还可以对dataViewEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataViewEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建dataViewEntity中的基本信息以外，还可以对dataViewEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataViewEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping(value="")
  public ResponseModel create(@RequestBody @ApiParam(name="dataViewEntity" , value="相关的创建过程，http接口。请注意该创建过程除了可以创建dataViewEntity中的基本信息以外，还可以对dataViewEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataViewEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") DataViewEntity dataViewEntity) {
    try {
      DataViewEntity current = this.dataViewEntityService.create(dataViewEntity , false);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataViewEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataViewEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PatchMapping(value="")
  public ResponseModel update(@RequestBody @ApiParam(name="dataViewEntity" , value="相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataViewEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") DataViewEntity dataViewEntity) {
    try {
      DataViewEntity current = this.dataViewEntityService.update(dataViewEntity);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照数据源code查询 视图、分组、字段等统计信息
   */
  @ApiOperation(value = "按照数据源code查询DataViewEntity")
  @ApiImplicitParam(name = "dataSourceCode", value = "按照数据源ID查询DataViewEntity", required = true)
  @RequestMapping(value="/findByDataSourceCode" , method={RequestMethod.GET})
  public ResponseModel findByDataSourceCode(@RequestParam(required=false) @ApiParam("数据源code") String dataSourceCode) {
    try {
      JSONArray jsonArr = this.dataViewEntityService.findByDataSourceCode(dataSourceCode);
      return this.buildHttpResultW(jsonArr);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照数据源ID查询DataViewEntity
   */
  @ApiOperation(value = "按照数据源ID查询DataViewEntity")
  @ApiImplicitParam(name = "dataSourceId", value = "按照数据源ID查询DataViewEntity", required = true)
  @RequestMapping(value="/findByDataSource" , method={RequestMethod.GET})
  public ResponseModel findByDataSource(@RequestParam("dataSourceId") @ApiParam("数据源ID") String dataSourceId) {
    try {
      List<DataViewEntity> result = this.dataViewEntityService.findByDataSource(dataSourceId);
      return this.buildHttpResultW(result, DATASOURCE_PARAM_NAME);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照数据库分组查询DataViewEntity
   */
  @ApiOperation(value = "按照数据库分组查询DataViewEntity")
  @ApiImplicitParam(name = "dataViewGroupId", value = "按照数据库分组查询DataViewEntity", required = true)
  @RequestMapping(value="/findByDataViewGroup" , method={RequestMethod.GET})
  public ResponseModel findByDataViewGroup(@RequestParam("dataViewGroupId") @ApiParam("数据库分组") String dataViewGroupId) {
    try {
      List<DataViewEntity> result = this.dataViewEntityService.findByDataViewGroup(dataViewGroupId);
      return this.buildHttpResultW(result, GROUP_PARAM_NAME);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  /**
   * 按照结构字段查询DataViewEntity
   */
  @ApiOperation(value = "按照结构字段查询DataViewEntity")
  @ApiImplicitParam(name = "filtersId", value = "按照结构字段查询DataViewEntity", required = true)
  @RequestMapping(value="/findByFields" , method={RequestMethod.GET})
  public ResponseModel findByFields(@RequestParam("fieldsId") @ApiParam("结构字段") String fieldsId) {
    try {
      DataViewEntity result = this.dataViewEntityService.findByFields(fieldsId);
      return this.buildHttpResultW(result, FIELDS_PARAM_NAME);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  /**
   * 按照过滤筛选字段查询DataViewEntity
   */
  @ApiOperation(value = "按照过滤筛选字段查询DataViewEntity")
  @ApiImplicitParam(name = "filtersId", value = "按照过滤筛选字段查询DataViewEntity", required = true)
  @RequestMapping(value="/findByFilters" , method={RequestMethod.GET})
  public ResponseModel findByFilters(@RequestParam("filtersId") @ApiParam("过滤筛选字段") String filtersId) {
    try {
      DataViewEntity result = this.dataViewEntityService.findByFilters(filtersId);
      return this.buildHttpResultW(result, FILTERS_PARAM_NAME);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  /**
   * 按照系统查询字段查询DataViewEntity
   */
  @ApiOperation(value = "按照系统查询字段查询DataViewEntity")
  @ApiImplicitParam(name = "systemFiltersId", value = "按照系统查询字段查询DataViewEntity", required = true)
  @RequestMapping(value="/findBySystemFilters" , method={RequestMethod.GET})
  public ResponseModel findBySystemFilters(@RequestParam("systemFiltersId") @ApiParam("系统查询字段") String systemFiltersId) {
    try {
      DataViewEntity result = this.dataViewEntityService.findBySystemFilters(systemFiltersId);
      return this.buildHttpResultW(result, SYSTEM_PARAM_NAME);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "按照数据视图的唯一业务编号信息进行详情查询（包括关联信息）")
  @RequestMapping(value="/findDetailsByCode" , method={RequestMethod.GET})
  public ResponseModel findDetailsByCode(@RequestParam("code") @ApiParam("数据视图的唯一业务编号信息") String code) {
    try {
      DataViewEntity result = this.dataViewEntityService.findDetailsByCode(code);
      return this.buildHttpResultW(result, FIELD_FILTERS_F);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }


  @ApiOperation(value = "第三方数据源语句输出分析")
  @RequestMapping(value = "/sqlAnalysis",method = {RequestMethod.PATCH})
  public ResponseModel sqlAnalysis(@RequestParam("dataSourceCode") @ApiParam("第三方数据源编码") String dataSourceCode,@RequestBody @ApiParam() DataViewEntity dataView){
    try {
      DataViewEntity result = this.dataViewEntityService.executeResource(dataSourceCode,dataView);
      return this.buildHttpResultW(result, FIELD_FILTERS);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "主数据源语句输出分析")
  @RequestMapping(value = "/mainSqlAnalysis",method = {RequestMethod.PATCH})
  public ResponseModel mainSqlAnalysis(@RequestBody @ApiParam() DataViewEntity dataView){
    try {
      DataViewEntity result = this.dataViewEntityService.executeResource(dataView);
      return this.buildHttpResultW(result, FIELD_FILTERS);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 执行数据视图查询
   * @return
   */
  @PostMapping("executeQuery")
  @ApiOperation("执行数据视图")
  public ResponseModel executeQuery(@RequestBody @ApiParam ExecuteParamModel executeParam) {
    try {
      List<JSONObject> list = dataViewEntityService.executeQuery(executeParam);
      return buildHttpResult(list);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 执行数据视图分页查询
   * @return
   */
  @PostMapping("executePageQuery")
  @ApiOperation("执行数据视图分页查询")
  public ResponseModel executePageQuery(@RequestBody @ApiParam ExecuteParamModel executeParam) {
    try {
      Page<JSONObject> page = dataViewEntityService.executePageQuery(executeParam);
      return buildHttpResult(page);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 导出数据到excel
   * @param dataSourceCode
   * @param listTemplateId
   * @param json
   * @return
   */
  @PostMapping("exportExcel")
  @ApiOperation("导出数据到excel")
  public void exportExcel(HttpServletRequest request, HttpServletResponse response,
                          @RequestParam("executeParam") String executeParamJson,
                          @RequestParam("exportExcelParams") String exportExcelParamsJson) {
    try {
      Principal principal = this.getPrincipal();
      ExecuteParamModel executeParam = null;
      if(StringUtils.isNotBlank(executeParamJson)) {
        executeParam = JsonUtils.json2Obj(executeParamJson, ExecuteParamModel.class);
      }
      List<DataviewExportExcelFieldVo> excelFields = JsonUtils.json2Obj(exportExcelParamsJson, new TypeReference<List<DataviewExportExcelFieldVo>>() {});
      byte[] bytes = dataViewEntityService.exportExcel(executeParam, principal, excelFields);
      String fileName = excelExportTemplate.fileName(executeParam);
      fileName = StringUtils.join(fileName, ".xlsx");
      super.writeResponseFile(request, response, bytes, fileName);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
    }
  }

  /**
   * 将某视图移动至新的分组下
   * @param dataViewId
   * @param dataViewGroupId
   * @return
   */
  @ApiOperation(value = "将某视图移动至新的分组下")
  @RequestMapping(value="/moveGroup" , method={RequestMethod.PATCH})
  public ResponseModel moveGroup(@RequestParam("dataViewId") @ApiParam("数据视图ID") String dataViewId,@RequestParam("dataViewGroupId") @ApiParam("移至分组ID") String dataViewGroupId) {
    try {
      this.dataViewEntityService.moveGroup(dataViewId,dataViewGroupId);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据数据视图和指定权限预言执行结果的字段
   * @param code
   * @param authCode
   * @return
   */
  @GetMapping("findFieldsByCodeAndAuthCode")
  @ApiOperation(value = "根据数据视图和指定权限预言执行结果的字段", notes = "权限编码不传或传空则返回数据视图的所有字段")
  public ResponseModel findFieldsByCodeAndAuthCode(@RequestParam @ApiParam(value = "数据视图编码", required = true) String code,
                                                   @RequestParam(required = false) @ApiParam("权限编码") String authCode) {
    try {
      Set<String> fields = dataViewEntityService.findFieldsByCodeAndAuthCode(code, authCode);
      return buildHttpResultW(fields);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }


} 
