package com.bizunited.platform.core.repository.dataview;

import com.bizunited.platform.core.entity.DataSourceEntity;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import javax.transaction.Transactional;
import java.util.List;

/**
 * DataSourceEntity业务模型的数据库方法支持
 * @author saturn
 */
@Repository("_DataSourceRepository")
public interface DataSourceRepository
    extends
    JpaRepository<DataSourceEntity, String>,
    JpaSpecificationExecutor<DataSourceEntity>,
    DataSourceRepositoryCustom {

  /**
   * 按照数据源编号进行查询
   * 
   * @param code
   * @return
   */
  public DataSourceEntity findByCode(String code);

  /**
   * 按照状态标识进行查询
   * 
   * @param tstatus
   * @return
   */
  @Query("select dataSourceEntity from DataSourceEntity dataSourceEntity  where dataSourceEntity.tstatus = :tstatus and (dataSourceEntity.projectName = '' OR dataSourceEntity.projectName IS NULL ) order by dataSourceEntity.createTime desc ")
  public List<DataSourceEntity> findByTstatusAndBlankProjectName(@Param("tstatus") Integer tstatus);

  /**
   * 将指定的数据源code设置为“不可用”
   * 
   * @param codes 多个指定的codes信息
   */
  @Transactional
  @Modifying
  @Query(value = "update engine_datasource set t_status = 0 where code in (:codes)", nativeQuery = true)
  public void updateDisable(@Param("codes") String[] codes);

  /**
   * 将指定的数据源code设置为“可用”
   * 
   * @param codes 多个指定的codes信息
   */
  @Transactional
  @Modifying
  @Query(value = "update engine_datasource set t_status = 1 where code in (:codes)", nativeQuery = true)
  public void updateEnable(@Param("codes") String[] codes);

  /**
   * 根据数据库编号code（唯一键值），查询第三方数据源信息
   * 
   * @param code
   * @return
   */
  @Query("select dataSourceEntity from DataSourceEntity dataSourceEntity  where dataSourceEntity.code=:code ")
  public DataSourceEntity findDetailsByCode(@Param("code") String code);

  /**
   * 按照状态标识和项目名进行查询
   * @param tstatus
   * @param projectName
   * @return
   */
  @Query("select dataSourceEntity from DataSourceEntity dataSourceEntity  where dataSourceEntity.tstatus = :tstatus and dataSourceEntity.projectName = :projectName order by dataSourceEntity.createTime desc ")
  List<DataSourceEntity> findByTstatusAndProjectName(@Param("tstatus") Integer tstatus, @Param("projectName") String projectName);

  /**
   * 根据项目名查询
   * @param projectName
   * @return
   */
  @Query("select dataSourceEntity from DataSourceEntity dataSourceEntity  where dataSourceEntity.projectName = :projectName order by dataSourceEntity.createTime desc ")
  List<DataSourceEntity> findByProjectName(@Param("projectName") String projectName);

  /**
   * 查询无项目名的数据
   * @return
   */
  @Query("select dataSourceEntity from DataSourceEntity dataSourceEntity  where dataSourceEntity.projectName = '' OR dataSourceEntity.projectName IS NULL ")
  List<DataSourceEntity> findByBlankProjectName();
}
