package com.bizunited.platform.core.repository.dataview;

import com.bizunited.platform.core.entity.DataViewEntity;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;

/**
 * DataViewEntity业务模型的数据库方法支持
 *
 * @author saturn
 */
@Repository("_DataViewRepository")
public interface DataViewRepository
    extends
    JpaRepository<DataViewEntity, String>
    , JpaSpecificationExecutor<DataViewEntity>, DataViewRepositoryCustom {
  /**
   * 按照数据源ID进行查询
   *
   * @param dataSourceId 数据源ID
   */
  @Query("from DataViewEntity dataViewEntity "
      + " left join fetch dataViewEntity.dataSource dataSource "
      + " where dataSource.id = :dataSourceId")
  public List<DataViewEntity> findByDataSource(@Param("dataSourceId") String dataSourceId);

  /**
   * 按照数据库分组进行查询
   *
   * @param dataViewGroupId 数据库分组
   */
  @Query("from DataViewEntity dataViewEntity "
      + " left join fetch dataViewEntity.dataViewGroup dataViewGroup "
      + " where dataViewGroup.id = :dataViewGroupId")
  public List<DataViewEntity> findByDataViewGroup(@Param("dataViewGroupId") String dataViewGroupId);

  /**
   * 按照结构字段进行查询
   *
   * @param fieldsId 结构字段
   */
  @Query("from DataViewEntity dataViewEntity "
      + " left join fetch dataViewEntity.fields fields "
      + " where fields.id = :fieldsId")
  public DataViewEntity findByFields(@Param("fieldsId") String fieldsId);

  /**
   * 按照过滤筛选字段进行查询
   *
   * @param filtersId 过滤筛选字段
   */
  @Query("from DataViewEntity dataViewEntity "
      + " left join fetch dataViewEntity.filters filters "
      + " where filters.id = :filtersId")
  public DataViewEntity findByFilters(@Param("filtersId") String filtersId);

  /**
   * 按照系统查询字段进行查询
   *
   * @param systemFiltersId 系统查询字段
   */
  @Query("from DataViewEntity dataViewEntity "
      + " left join fetch dataViewEntity.systemFilters systemFilters "
      + " where systemFilters.id = :systemFiltersId")
  public DataViewEntity findBySystemFilters(@Param("systemFiltersId") String systemFiltersId);

  /**
   * 按照业务编号进行详情查询（包括关联信息）。注意，这里的详细查询并不包括数据权限信息
   *
   * @param code 数据视图的唯一业务编号信息
   */
  @Query("select dataViewEntity from DataViewEntity dataViewEntity "
      + " left join  fetch dataViewEntity.dataSource dataViewEntity_dataSource "
      + " left join  fetch dataViewEntity.dataViewGroup dataViewEntity_dataViewGroup "
      + " left join  fetch dataViewEntity.fields dataViewEntity_fields "
      + " left join  fetch dataViewEntity.filters dataViewEntity_filters "
      + " left join  fetch dataViewEntity.systemFilters dataViewEntity_systemFilters "
      + " where dataViewEntity.code=:code ")
  public DataViewEntity findDetailsByCode(@Param("code") String code);

  /**
   * 可以按照多个数据ID进行详情查询（包括关联信息）。注意，这里的详细查询并不包括数据权限信息
   *
   * @param code 数据视图的唯一业务编号信息
   */
  @Query("select distinct dataViewEntity from DataViewEntity dataViewEntity "
      + " left join  fetch dataViewEntity.dataSource dataViewEntity_dataSource "
      + " left join  fetch dataViewEntity.dataViewGroup dataViewEntity_dataViewGroup "
      + " left join  fetch dataViewEntity.fields dataViewEntity_fields "
      + " left join  fetch dataViewEntity.filters dataViewEntity_filters "
      + " left join  fetch dataViewEntity.systemFilters dataViewEntity_systemFilters "
      + " where dataViewEntity.id in (:ids) ")
  List<DataViewEntity> findDetailsByIds(@Param("ids") String[] ids);

  /**
   * 根据CODE查询
   *
   * @param code
   * @return
   */
  DataViewEntity findByCode(String code);

  /**
   * 按照数据源code查询 视图、分组、字段等统计信息
   *
   * @param dataSourceCode
   * @return
   */
  @Query(value = "select dataview.id,dataview.code,dataview.name,dataview.data_source_group datasourcegroup,outfields.outCount,displayfields.displayCount "
      + "from engine_dataview dataview "
      + "left join engine_datasource dataSource on dataview.data_source=dataSource.id "
      + "left join (select data_view,count(data_view) outCount from engine_dataview_field group by data_view) outfields on outfields.data_view=dataview.id  "
      + "left join (select data_view,count(data_view) displayCount from engine_dataview_field where t_status=1 group by data_view) displayfields  on displayfields.data_view=dataview.id  "
      + "where dataview.t_status=1 and dataSource.code =:dataSourceCode ", nativeQuery = true)
  public List<Object[]> findByDataSourceCode(@Param("dataSourceCode") String dataSourceCode);

  /**
   * 查询主数据源的 视图、分组、字段等统计信息
   *
   * @return
   */
  @Query(value = "select dataview.id,dataview.code,dataview.name,dataview.data_source_group datasourcegroup,outfields.outCount,displayfields.displayCount "
      + "from engine_dataview dataview "
      + "left join engine_datasource dataSource on dataview.data_source=dataSource.id "
      + "left join (select data_view,count(data_view) outCount from engine_dataview_field group by data_view) outfields on outfields.data_view=dataview.id  "
      + "left join (select data_view,count(data_view) displayCount from engine_dataview_field where t_status=1 group by data_view) displayfields  on displayfields.data_view=dataview.id  "
      + "where dataview.t_status=1 and dataSource.code is null ", nativeQuery = true)
  public List<Object[]> findByDataSourceCode();

  /**
   * 按照指定的数据视图ID信息，查询最终满足这些id编号的数据条目。<br>
   * 这个方法一般用于验证一批给定的id，是否都能在系统中找到对应的数据
   *
   * @param ids 指定的id信息
   */
  @Query(value = "select count(id) from engine_dataview where engine_dataview.id in (:ids)", nativeQuery = true)
  public int countByIds(@Param("ids") String[] ids);

  /**
   * 根据分组ID统计查询
   * @param groupId
   * @return
   */
  @Query("select count(*) from DataViewEntity dv inner join dv.dataViewGroup dvg where dvg.id = :groupId")
  long countByDataViewGroupId(@Param("groupId") String groupId);

  /**
   * 查询数量
   * @param code
   * @return
   */
  long countByCode(String code);
}