package com.bizunited.platform.core.repository.dataview;

import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.orm.hibernate5.SessionFactoryUtils;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Repository("DataViewRepositoryCustomImpl")
public class DataViewRepositoryCustomImpl implements DataViewRepositoryCustom{

  @Lazy
  @Autowired
  private DynamicDataSourceManager dynamicDataSourceManager;
  @Autowired
  @PersistenceContext
  private EntityManager entityManager;
  private static final String SYSTEM_PARAMS_PATTERN = "\\{\\:[a-zA-Z]{1}\\w*\\}";
  private static final String ERROR_DATAVIEW = "数据视图不能为空！";
  private static final String ERROR_SQL = "原始SQL不能为空！";
  private static final String ERROR_SQL_EXECUTE = "SQL执行异常，请检查!!详情：";

  @Override
  public DataViewEntity executeResource(DataViewEntity dataView) {
    Validate.notNull(dataView,ERROR_DATAVIEW);
    Validate.notBlank(dataView.getSourceSql(), ERROR_SQL);
    Session session = (Session) entityManager.getDelegate();
    String sourceSql = dataView.getSourceSql();
    this.replaceSysParam(dataView);
    session.doWork(connection -> {
      Set<DataViewFieldEntity> fields = new LinkedHashSet<>();
      try(
          Statement statement = connection.createStatement();
          ResultSet result = statement.executeQuery(dataView.getSourceSql())
      ){
        ResultSetMetaData metaData = result.getMetaData();
        //遍历结果集中每一个字段，得到字段信息并存入fields
        Set<String> fieldNames = new HashSet<>();
        //用户参数不能重名
        for(int i=1;i<=metaData.getColumnCount();i++){
          DataViewFieldEntity field = new DataViewFieldEntity();
          field.setFieldName(metaData.getColumnLabel(i));
          //如果查询出的字段集中含有相同的字段名，那么报错提醒用户
          Validate.isTrue(fieldNames.add(metaData.getColumnLabel(i)),"返回字段集有相同字段名：%s,请为其取别名，或做其他调整。" , metaData.getColumnLabel(i));
          field.setFieldType(metaData.getColumnClassName(i));
          field.setPhysical(!StringUtils.isEmpty(metaData.getTableName(i)));
          field.setSchemaName(metaData.getSchemaName(i));
          field.setTargetTable(metaData.getTableName(i));
          field.setTargetField(metaData.getColumnName(i));
          field.setSortIndex(i);
          fields.add(field);
        }
        fieldNames.clear();
      }
      dataView.setFields(fields);
    });
    dataView.setSourceSql(sourceSql);
    return dataView;
  }

  @Override
  public void checkTable(String tableName) {
    Validate.notBlank(tableName,"输入数据表名不能为空，请检查！");
    Session session = (Session) entityManager.getDelegate();
    session.doWork(connection -> {
      DatabaseMetaData md = connection.getMetaData();
      try(ResultSet rs = md.getTables(null, null, tableName, null)){
        Validate.isTrue(rs.next(),"所设置基础表【%s】在主数据源不存在，请检查！", tableName);
      }
    });
  }

  @Override
  public DataViewEntity executeResource(String dataSourceCode, DataViewEntity dataView) {
    Validate.notNull(dataView,ERROR_DATAVIEW);
    Validate.notBlank(dataView.getSourceSql(), ERROR_SQL);
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory,"未找到该数据源！");
    String sourceSql = dataView.getSourceSql();
    Set<DataViewFieldEntity> fields = new LinkedHashSet<>();
    DataSource dataSource = SessionFactoryUtils.getDataSource(sessionFactory);
    DataViewEntity currentDataView = this.replaceSysParam(dataView);
    try(Connection connection = dataSource.getConnection();
        Statement statement = connection.createStatement();
        ResultSet result = statement.executeQuery(currentDataView.getSourceSql())) {
      ResultSetMetaData metaData = result.getMetaData();
      //遍历结果集中每一个字段，得到字段信息并存入fields
      Set<String> fieldNames = new HashSet<>();
      //用户参数与系统参数也不能重名
      Set<String> sysParamsStrs = currentDataView.getSystemFilters().stream().map(DataViewSystemEntity::getParamName).collect(Collectors.toSet());
      fieldNames.addAll(sysParamsStrs);
      for(int i=1;i<=metaData.getColumnCount();i++){
        DataViewFieldEntity field = new DataViewFieldEntity();
        field.setFieldName(metaData.getColumnLabel(i));
        //如果查询出的字段集中含有相同的字段名，那么报错提醒用户
        Validate.isTrue(fieldNames.add(metaData.getColumnLabel(i)),"返回字段集或与系统参数中含有相同字段名：%s,请为其取别名，或做其他调整。" , metaData.getColumnLabel(i));
        field.setFieldType(metaData.getColumnClassName(i));
        field.setPhysical(!StringUtils.isEmpty(metaData.getTableName(i)));
        field.setSchemaName(metaData.getSchemaName(i));
        field.setTargetTable(metaData.getTableName(i));
        field.setTargetField(metaData.getColumnName(i));
        field.setSortIndex(i);
        fields.add(field);
      }
      fieldNames.clear();
      currentDataView.setFields(fields);
    }catch (SQLException e){
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }
    currentDataView.setSourceSql(sourceSql);
    return currentDataView;
  }

  @Override
  public void checkTable(String dataSourceCode, String tableName) {
    Validate.notBlank(tableName,"输入数据表名不能为空，请检查！");
    Validate.notBlank(dataSourceCode,"输入第三方数据源编码不能为空，请检查！");
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    try (Connection connection = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();){
      DatabaseMetaData md = connection.getMetaData();
      try(ResultSet rs = md.getTables(null, null, tableName, null);){
        Validate.isTrue(rs.next(),"所设置第三方数据源中基础表不存在，请检查！");
      }
    } catch (Exception e){
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }
  }

  /**
   * 分析SQL，获取其中的系统参数，并将系统参数值变为空字符串
   * 替换查询SQL中的系统参数,
   * @param dataView
   * @return
   */
  private DataViewEntity replaceSysParam(DataViewEntity  dataView){
    Validate.notNull(dataView,ERROR_DATAVIEW);
    Validate.notBlank(dataView.getSourceSql(), ERROR_SQL);
    String sql = dataView.getSourceSql();
    Set<DataViewSystemEntity> systemEntities = new LinkedHashSet<>();
    Set<String> sysParams = this.extractMessageByRegular(sql);
    if(!CollectionUtils.isEmpty(sysParams)){
      for(String param : sysParams){
        //取出系统参数字段
        DataViewSystemEntity dataViewSystem = new DataViewSystemEntity();
        dataViewSystem.setParamName(param);
        systemEntities.add(dataViewSystem);
        //将原SQL字段设置为空
        String replaceStr = "{:" + param + "}";
        // 这里加上括号是因为不确定前面的操作符是=或in
        sql = sql.replace(replaceStr,"('')");
      }
    }
    dataView.setSystemFilters(systemEntities);
    dataView.setSourceSql(sql);
    return dataView;
  }

  /**
   * 正则获取特定格式的系统参数
   * @param msg
   * @return
   */
  private Set<String> extractMessageByRegular(String msg){
    Set<String> list=new LinkedHashSet<>();
    Pattern p = Pattern.compile(SYSTEM_PARAMS_PATTERN);
    Matcher m = p.matcher(msg);
    while(m.find()){
      //截取符合规则的系统参数字符串
      list.add(m.group().substring(2, m.group().length()-1));
    }
    return list;
  }


}
