package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.enums.SqlOperatorEnum;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewFilterEntity;
import com.bizunited.platform.core.service.dataview.model.ExecuteContextModel;
import com.bizunited.platform.core.service.dataview.model.ExecuteParamModel;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 用于数据视图filter用户筛选条件的分析以及构造SQL工具类
 */
class SQLConditionFilterAnalysis {

  /**
   * 构建sql的条件部分
   * @param executeContext
   */
  static void buildSQLConditions(ExecuteContextModel executeContext) {
    handleInputParams(executeContext);
    handleKeyParams(executeContext);
  }

  /**
   * 处理传入的参数条件
   * @param executeContext
   */
  private static void handleInputParams(ExecuteContextModel executeContext) {
    Set<DataViewFilterEntity> filters = executeContext.getDataViewFilters();
    if(CollectionUtils.isEmpty(filters)) {
      return;
    }
    ExecuteParamModel executeParam = executeContext.getExecuteParam();
    for (DataViewFilterEntity filter : filters) {
      DataViewFieldEntity field = filter.getField();
      Validate.notNull(field, "未找到参数的对应字段：%s", filter.getDisplayName());
      String fieldName = field.getFieldName();
      Object value = executeParam.getInputParam(filter.getParamName());
      if(!filter.getNullable()) {
        Validate.notNull(value, "未发现必传的筛选参数：%s", field.getFieldName());
      }
      if(value == null) {
        continue;
      }
      SqlParamAnalysis.handleParamCondition(executeContext, fieldName, field.getFieldType(), filter.getOpType(), value);
    }
  }

  /**
   * 处理主键条件的参数
   * @param executeContext
   */
  private static void handleKeyParams(ExecuteContextModel executeContext) {
    // 处理关键字查询条件
    ExecuteParamModel executeParam = executeContext.getExecuteParam();
    Map<String, Object> keyParams = executeParam.getKeyParams();
    if(CollectionUtils.isEmpty(keyParams)) {
      return;
    }
    Set<DataViewFieldEntity> fields = executeContext.getDataViewFields();
    Map<String, DataViewFieldEntity> fieldMap = fields.stream().collect(Collectors.toMap(DataViewFieldEntity::getFieldName, f -> f));
    for (Map.Entry<String, Object> entry : keyParams.entrySet()) {
      String key = entry.getKey();
      Object value = entry.getValue();
      Validate.notNull(value, "主键查询值不能为空");
      Validate.notBlank(key, "主键字段名不能为空");
      DataViewFieldEntity field = fieldMap.get(key);
      Validate.notNull(field, "未找到字段：%s", key);
      SqlParamAnalysis.handleParamCondition(executeContext, field.getFieldName(), field.getFieldType(), SqlOperatorEnum.IN.name(), value);
    }
  }

}
