package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.constant.DataViewPresetParamKey;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.userGroup.UserGroupService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_QUERY_COVER_CODE;

/**
 * 处理sql中的预制值信息
 * @Author: Paul Chan
 * @Date: 2020-04-20 10:59
 */
public class SQLPresetValueAnalysis {

  /**
   * 缓存信息
   */
  private final Map<String, Object> cache = new ConcurrentHashMap<>();
  /**
   * 角色服务
   */
  private RoleService roleService;
  /**
   * 岗位服务
   */
  private PositionService positionService;
  /**
   * 用户组服务
   */
  private UserGroupService userGroupService;
  /**
   * 组织机构服务
   */
  private OrganizationService organizationService;

  public SQLPresetValueAnalysis(ApplicationContext applicationContext) {
    Validate.notNull(applicationContext, "spring 上下文不能为空");
    this.roleService = applicationContext.getBean(RoleService.class);
    this.positionService = applicationContext.getBean(PositionService.class);
    this.userGroupService = applicationContext.getBean(UserGroupService.class);
    this.organizationService = applicationContext.getBean(OrganizationService.class);
  }

  /**
   * 根据设置获取预制值
   * @param paramKey
   * @return
   */
  public Object getPresetValue(String paramKey, Integer level) {
    if(StringUtils.isBlank(paramKey)) {
      return null;
    }
    UserVo user = SecurityUtils.getCurrentUser();
    Object value = cache.get(paramKey);
    if(value != null) {
      return value;
    }
    switch (paramKey) {
      //用户相关
      case DataViewPresetParamKey.OPT_USER_ID:
        value = user.getId();
        break;
      case DataViewPresetParamKey.OPT_USER_NAME:
        value = user.getUserName();
        break;
      case DataViewPresetParamKey.OPT_USER_ACCOUNT:
        value = user.getAccount();
        break;
      case DataViewPresetParamKey.OPT_USER_CODE:
        value = user.getId();
        break;
      case DataViewPresetParamKey.OPT_USER_PHONE:
        value = user.getPhone();
        break;
      case DataViewPresetParamKey.OPT_USER_STATUS:
        value = user.getUseStatus();
        break;
      case DataViewPresetParamKey.OPT_USER_IDCARD:
        value = user.getIdcard();
        break;
      //岗位相关
      case DataViewPresetParamKey.OPT_USER_POSITION_ID:
      case DataViewPresetParamKey.OPT_USER_POSITION:
      case DataViewPresetParamKey.OPT_USER_POSITION_CODE:
      case DataViewPresetParamKey.OPT_USER_POSITION_STATUS:
        value = this.getPositionPreset(user, paramKey);
        break;
      //组织机构相关
      case DataViewPresetParamKey.OPT_USER_ORG_STATUS:
      case DataViewPresetParamKey.OPT_USER_ORG_DESC:
      case DataViewPresetParamKey.OPT_USER_ORG_TYPE:
      case DataViewPresetParamKey.OPT_USER_ORG_ID:
      case DataViewPresetParamKey.OPT_USER_ORG:
      case DataViewPresetParamKey.OPT_CUR_ORGS:
      case DataViewPresetParamKey.OPT_PARENT_ORGS:
      case DataViewPresetParamKey.OPT_PARENT_ALL_ORGS:
      case DataViewPresetParamKey.OPT_CHILDREN_ORGS:
      case DataViewPresetParamKey.OPT_CHILDREN_ALL_ORGS:
      case DataViewPresetParamKey.OPT_PARENT_LEVEL_ORGS:
      case DataViewPresetParamKey.OPT_CHILDREN_LEVEL_ORGS:
        value = this.getOrgPreset(user, paramKey, level);
        break;
        // 用户组相关
      case DataViewPresetParamKey.OPT_USER_GROUP:
        value = this.getUserGroupPreset(user, paramKey);
        break;
        // 用户角色相关
      case DataViewPresetParamKey.OPT_USER_ROLE:
        value = this.getUserRolePreset(user, paramKey);
        break;
      default:
        // 其他的，如时间相关的
        value = this.getOtherPreset(user, paramKey);
        break;
    }
    if(value != null) {
      cache.put(paramKey, value);
    }
    return value;
  }

  private Object getOtherPreset(UserVo user, String paramKey) {
    Object value;
    switch (paramKey) {
      case DataViewPresetParamKey.CURRENT_USER_LOGIN_TIME:
        value = user.getLastloginTime();
        break;
      case DataViewPresetParamKey.NOW_DATE_MONTH:
        value = new SimpleDateFormat("yyyy-MM").format(new Date());
        break;
      case DataViewPresetParamKey.NOW_DATE_DAY:
        value = new SimpleDateFormat("yyyy-MM-dd").format(new Date());
        break;
      case DataViewPresetParamKey.NOW_DATE_MINUTE:
        value = new SimpleDateFormat("yyyy-MM-dd HH:mm").format(new Date());
        break;
      case DataViewPresetParamKey.NOW_DATE_SECOND:
        value = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date());
        break;
      default:
        throw new IllegalArgumentException(String.format("不能识别的预制值类别：%s", paramKey));
    }
    return value;
  }

  /**
   * 获取用户角色预制值
   * @param user
   * @param paramKey
   * @return
   */
  private Object getUserRolePreset(UserVo user, String paramKey) {
    List<RoleVo> roles = this.roleService.findAllByUserId(user.getId(), 0);
    if(CollectionUtils.isEmpty(roles)) {
      return Lists.newArrayList(DEFAULT_QUERY_COVER_CODE);
    }
    return roles.stream().map(RoleVo::getRoleCode).collect(Collectors.toList());
  }

  /**
   * 获取用户组的预制值信息
   * @param user
   * @param paramKey
   * @return
   */
  private Object getUserGroupPreset(UserVo user, String paramKey) {
    Set<UserGroupVo> groups = userGroupService.findByUserId(user.getId());
    if(CollectionUtils.isEmpty(groups)) {
      return Lists.newArrayList(DEFAULT_QUERY_COVER_CODE);
    }
    return groups.stream().map(UserGroupVo::getGroupCode).collect(Collectors.toList());
  }

  /**
   * 获取岗位的预制值
   * @param applicationContext
   * @param user
   * @param paramKey
   * @return
   */
  private Object getPositionPreset(UserVo user, String paramKey) {
    PositionVo position = positionService.findMainPositionByUserId(user.getId());
    if(position == null) {
      return DEFAULT_QUERY_COVER_CODE;
    }
    switch (paramKey) {
      case DataViewPresetParamKey.OPT_USER_POSITION_ID:
        return position.getId();
      case DataViewPresetParamKey.OPT_USER_POSITION:
      case DataViewPresetParamKey.OPT_USER_POSITION_CODE:
        return position.getCode();
      case DataViewPresetParamKey.OPT_USER_POSITION_STATUS:
        return position.getTstatus();
      default:
        break;
    }
    return position.getCode();
  }

  /**
   * 获取组织机构类型的预制值
   * @param applicationContext
   * @param user
   * @param paramKey
   * @return
   */
  private Object getOrgPreset(UserVo user, String paramKey, Integer level) {
    Set<OrganizationVo> orgs = null;
    Function<OrganizationVo, Object> function = null;
    switch (paramKey) {
      case DataViewPresetParamKey.OPT_USER_ORG_STATUS:
        orgs = organizationService.findOrgByUserId(user.getId());
        function = OrganizationVo::getTstatus;
        break;
      case DataViewPresetParamKey.OPT_USER_ORG_DESC:
        orgs = organizationService.findOrgByUserId(user.getId());
        function = OrganizationVo::getDescription;
        break;
      case DataViewPresetParamKey.OPT_USER_ORG_TYPE:
        orgs = organizationService.findOrgByUserId(user.getId());
        function = OrganizationVo::getType;
        break;
      case DataViewPresetParamKey.OPT_USER_ORG_ID:
        orgs = organizationService.findOrgByUserId(user.getId());
        function = OrganizationVo::getId;
        break;
      case DataViewPresetParamKey.OPT_USER_ORG:
      case DataViewPresetParamKey.OPT_CUR_ORGS:
        orgs = organizationService.findOrgByUserId(user.getId());
        function = OrganizationVo::getCode;
        break;
      case DataViewPresetParamKey.OPT_PARENT_ORGS:
        orgs = this.findParentOrgsByUserId(user.getId());
        function = OrganizationVo::getCode;
        break;
      case DataViewPresetParamKey.OPT_PARENT_ALL_ORGS:
        orgs = this.findAllParentOrgsByUserId(user.getId());
        function = OrganizationVo::getCode;
        break;
      case DataViewPresetParamKey.OPT_CHILDREN_ORGS:
        orgs = this.findChildrenOrgsByUserId(user.getId());
        function = OrganizationVo::getCode;
        break;
      case DataViewPresetParamKey.OPT_CHILDREN_ALL_ORGS:
        orgs = this.findAllChildrenOrgsByUserId(user.getId());
        function = OrganizationVo::getCode;
        break;
      case DataViewPresetParamKey.OPT_PARENT_LEVEL_ORGS:
        orgs = this.findLevelParentOrgsByUserId(user.getAccount(), level);
        function = OrganizationVo::getCode;
        break;
      case DataViewPresetParamKey.OPT_CHILDREN_LEVEL_ORGS:
        orgs = this.findLevelChildrenOrgsByUserId(user.getAccount(), level);
        function = OrganizationVo::getCode;
        break;
      default:
        throw new IllegalArgumentException(String.format("不能识别的组织机构预制值：%s", paramKey));
    }
    if(CollectionUtils.isEmpty(orgs)) {
      return Lists.newArrayList(DEFAULT_QUERY_COVER_CODE);
    }
    return orgs.stream().map(function).collect(Collectors.toList());
  }

  /**
   * 获取用户指定层级的下级组织
   * @param account
   * @param level
   * @return
   */
  private Set<OrganizationVo> findLevelChildrenOrgsByUserId(String account, Integer level) {
    Set<OrganizationVo> orgs = organizationService.findByTypeAndLevel(2, account, level);
    Set<OrganizationVo> result = new HashSet<>();
    for(OrganizationVo org : orgs){
      this.getChlidrenOrgs(org, result);
    }
    return result;
  }

  /**
   * 构建下级集合
   * @param org
   * @param result
   */
  private void getChlidrenOrgs(OrganizationVo org, Set<OrganizationVo> result) {
    Set<OrganizationVo> children = org.getChild();
    if(CollectionUtils.isEmpty(children)){
      return;
    }
    result.addAll(children);
    for(OrganizationVo item :children) {
      this.getChlidrenOrgs(item, result);
    }
  }

  /**
   * 获取用户指定层级的上级组织
   * @param account
   * @param level
   * @return
   */
  private Set<OrganizationVo> findLevelParentOrgsByUserId(String account, Integer level) {
    Set<OrganizationVo> orgs = organizationService.findByTypeAndLevel(1, account, level);
    Set<OrganizationVo> result = new HashSet<>();
    for(OrganizationVo org : orgs){
      this.getParentOrgs(org, result);
    }
    return result;
  }

  /**
   * 构建上级集合
   * @param org
   * @param result
   */
  private void getParentOrgs(OrganizationVo org, Set<OrganizationVo> result) {
    OrganizationVo parent = org.getParent();
    if(parent == null){
      return;
    }
    result.add(parent);
    this.getParentOrgs(parent, result);
  }


  /**
   * 获取用户的下级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findChildrenOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> children = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      List<OrganizationVo> son = organizationService.findByParent(org.getId());
      if (!CollectionUtils.isEmpty(son)) {
        children.addAll(son);
      }
    }
    return children;
  }

  /**
   * 获取用户的所有下级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findAllChildrenOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> children = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      this.findChildrenOrgs(children, org.getId());
    }
    return children;
  }

  /**
   * 寻找一个组织机构的所有子级组织机构
   * @param orgs
   * @param parentOrgId
   */
  private void findChildrenOrgs(Set<OrganizationVo> orgs, String parentOrgId) {
    List<OrganizationVo> children = organizationService.findByParent(parentOrgId);
    if (CollectionUtils.isEmpty(children)) {
      return;
    }
    orgs.addAll(children);
    for (OrganizationVo org : children) {
      findChildrenOrgs(orgs, org.getId());
    }
  }

  /**
   * 获取用户的上级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findParentOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> parents = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      OrganizationVo parent = organizationService.findByChild(org.getId());
      if (parent != null) {
        parents.add(parent);
      }
    }
    return parents;
  }

  /**
   * 获取用户的所有上级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findAllParentOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> parents = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      findParentOrg(organizationService, parents, org.getId());
    }
    return parents;
  }

  /**
   * 寻找一个组织机构的所有父级组织机构
   * @param orgs
   * @param childOrgId
   */
  private void findParentOrg(OrganizationService organizationService, Set<OrganizationVo> orgs, String childOrgId) {
    OrganizationVo parent = organizationService.findByChild(childOrgId);
    if (parent == null) {
      return;
    }
    orgs.add(parent);
    findParentOrg(organizationService, orgs, parent.getId());
  }

}
