package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.common.enums.ClassTypeNameEnum;
import com.bizunited.platform.core.common.enums.SqlOperatorEnum;
import com.bizunited.platform.core.service.dataview.model.ExecuteContextModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static com.bizunited.platform.core.repository.dataview.analysis.Constants.SQL_PLACE_HOLDER;

/**
 * sql参数生成器
 * @Author: Paul Chan
 * @Date: 2020-04-18 16:52
 */
public class SqlParamAnalysis {

  private SqlParamAnalysis() {
    throw new IllegalStateException("静态类不能实例化");
  }

  /**
   * 处理sql参数，并且返回生成的占位符，根据操作符的类型和参数生成参数的sql部分，如 a.name = ?
   * @param executeContext
   * @param paramName
   * @param paramType
   * @param opType
   * @param value
   * @return
   */
  public static void handleParamCondition(ExecuteContextModel executeContext, String paramName, String paramType, String opType, Object value) {
    Validate.notBlank(opType, "操作符类型不能为空");
    SqlOperatorEnum operator = SqlOperatorEnum.valueOf(opType);
    Validate.notNull(operator, "不支持的操作符：%s", opType);
    switch (operator) {
      case NEQ:
      case EQ:
      case GT:
      case LT:
      case GE:
      case LE:
        handleNormalCondition(executeContext, paramName, paramType, operator, value);
        break;
      case LC_RC:
      case LO_RC:
      case LO_RO:
      case LC_RO:
      case BETWEEN:
        handleRegionCondition(executeContext, paramName, paramType, operator, value);
        break;
      case IN:
      case NIN:
        handleInCondition(executeContext, paramName, paramType, operator, value);
        break;
      case LLIKE:
      case RLIKE:
      case LIKE:
        handleLikeCondition(executeContext, paramName, paramType, operator, value);
        break;
      default:
        throw new IllegalArgumentException(String.format("不支持的操作符:%s", operator.getOprtValue()));
    }
  }

  /**
   * 处理普通的操作符条件，如等于、不等于等
   * @param executeContext
   * @param paramName
   * @param paramType
   * @param operator
   * @param value
   */
  private static void handleNormalCondition(ExecuteContextModel executeContext, String paramName, String paramType, SqlOperatorEnum operator, Object value) {
    Object formatValue = SqlParamFormater.format(paramType, value);
    String sqlAlias = executeContext.getWrapperSqlAlias();
    String condition = String.format(" and %s.%s %s %s", sqlAlias, paramName, operator.getOprtValue(), SQL_PLACE_HOLDER);
    executeContext.appendExecuteSql(condition);
    executeContext.addExecuteSqlParameter(formatValue);
  }

  /**
   * 处理模糊查询的条件
   * @param executeContext
   * @param paramName
   * @param paramType
   * @param operator
   * @param value
   */
  private static void handleLikeCondition(ExecuteContextModel executeContext, String paramName, String paramType, SqlOperatorEnum operator, Object value) {
    SqlAnalysis sqlAnalysis = executeContext.getSqlAnalysis();
    String sqlAlias = executeContext.getWrapperSqlAlias();
    String concat;
    switch (operator) {
      case LLIKE:
        concat = sqlAnalysis.concatSql("'%%'", SQL_PLACE_HOLDER);
        break;
      case RLIKE:
        concat = sqlAnalysis.concatSql(SQL_PLACE_HOLDER, "'%%'");
        break;
      case LIKE:
        concat = sqlAnalysis.concatSql("'%%'", SQL_PLACE_HOLDER, "'%%'");
        break;
      default:
        throw new IllegalArgumentException(String.format("不支持的操作符:%s", operator.getOprtValue()));
    }
    String condition = String.format(" and %s.%s like %s", sqlAlias, paramName, concat);
    executeContext.appendExecuteSql(condition);
    executeContext.addExecuteSqlParameter(value == null ? value : value.toString());
  }

  /**
   * 处理包含或不包含的参数
   * @param executeContext
   * @param paramName
   * @param paramType
   * @param operator
   * @param value
   */
  private static void handleInCondition(ExecuteContextModel executeContext, String paramName, String paramType, SqlOperatorEnum operator, Object value) {
    List<Object> values = getArrayValue(value);
    String sqlAlias = executeContext.getWrapperSqlAlias();
    String replace = handleAndGetParamReplaceStr(executeContext, paramName, paramType, values);
    String condition = String.format(" and %s.%s %s %s ", sqlAlias, paramName, operator.getOprtValue(), replace);
    executeContext.appendExecuteSql(condition);
  }

  /**
   * 处理区间的参数条件
   * @param executeContext
   * @param paramName
   * @param paramType
   * @param operator
   * @param value
   */
  private static void handleRegionCondition(ExecuteContextModel executeContext, String paramName, String paramType, SqlOperatorEnum operator, Object value) {
    String sqlAlias = executeContext.getWrapperSqlAlias();
    Validate.isTrue(isArray(value), "参数【%s】在使用操作符【%s】时传入的参数【%s】非两位数组", paramName, operator.getOprtValue(), value);
    List<Object> values = getArrayValue(value);
    Validate.isTrue(values.size() == 2, "参数【%s】在使用操作符【%s】时传入的参数【%s】非两位数组", paramName, operator.getOprtValue(), value);
    String format;
    switch (operator) {
      case LC_RC:
        format = " and %s.%s >= ? and %s.%s <= ? ";
        break;
      case LO_RC:
        format = " and %s.%s > ? and %s.%s <= ? ";
        break;
      case LO_RO:
        format = " and %s.%s > ? and %s.%s < ? ";
        break;
      case LC_RO:
        format = " and %s.%s >= ? and %s.%s < ? ";
        break;
      case BETWEEN:
        format = " and %s.%s between ? and ? ";
        break;
      default:
        throw new IllegalArgumentException(String.format("不支持的操作符:%s", operator.getOprtValue()));
    }
    String condition;
    if(operator.equals(SqlOperatorEnum.BETWEEN)) {
      condition = String.format(format, sqlAlias, paramName);
    } else {
      condition = String.format(format, sqlAlias, paramName, sqlAlias, paramName);
    }
    executeContext.appendExecuteSql(condition);
    ClassTypeNameEnum classType = ClassTypeNameEnum.valueOfClassName(paramType);
    if(classType != null && (classType == ClassTypeNameEnum.DATE || classType == ClassTypeNameEnum.TIMESTAMP)) {
      values = handleRegionDateValues(values);
    }
    for (Object v : values) {
      executeContext.addExecuteSqlParameter(SqlParamFormater.format(paramType, v));
    }
  }

  /**
   * 处理时间区间的参数，原则上是把开始日期的时分秒毫秒设置成0，把结束日期的时间秒毫秒设置成最大值
   * @param values
   * @return
   */
  private static List<Object> handleRegionDateValues(List<Object> values) {
    Object start = SqlParamFormater.formatDate(values.get(0));
    Object end = SqlParamFormater.formatDate(values.get(1));
    if(start != null && Date.class.isAssignableFrom(start.getClass())) {
      Calendar calendar = Calendar.getInstance();
      calendar.setTime((Date) start);
      calendar.set(Calendar.HOUR_OF_DAY, 0);
      calendar.set(Calendar.MINUTE, 0);
      calendar.set(Calendar.SECOND, 0);
      calendar.set(Calendar.MILLISECOND, 0);
      values.set(0, calendar.getTime());
    }
    if(end != null && Date.class.isAssignableFrom(end.getClass())) {
      Calendar calendar = Calendar.getInstance();
      calendar.setTime((Date) end);
      calendar.set(Calendar.HOUR_OF_DAY, 23);
      calendar.set(Calendar.MINUTE, 59);
      calendar.set(Calendar.SECOND, 59);
      calendar.set(Calendar.MILLISECOND, 999);
      values.set(1, calendar.getTime());
    }
    return values;
  }


  /**
   * 处理sql参数，并且返回生成的占位符，如果是单个对象，占位符是单个？，如果是集合，则返回以逗号(,)分割的多个问号
   * @param executeContext
   * @param paramName
   * @param paramType
   * @param value
   * @return
   */
  public static String handleAndGetParamReplaceStr(ExecuteContextModel executeContext, String paramName, String paramType, Object value) {
    if(value == null) {
      executeContext.addExecuteSqlParameter(value);
      return SQL_PLACE_HOLDER;
    }
    if(value.getClass().isArray() || Iterable.class.isAssignableFrom(value.getClass())) {
      List<Object> values = getArrayValue(value);
      Validate.notEmpty(values, "传入的参数为空：%s", paramName);
      StringBuilder replace = new StringBuilder();
      replace.append("(");
      List<String> holders = new ArrayList<>();
      for (Object obj : values) {
        holders.add(SQL_PLACE_HOLDER);
        executeContext.addExecuteSqlParameter(SqlParamFormater.format(paramType, obj));
      }
      replace.append(StringUtils.join(holders, ","));
      replace.append(")");
      return replace.toString();
    }
    executeContext.addExecuteSqlParameter(SqlParamFormater.format(paramType, value));
    return SQL_PLACE_HOLDER;
  }

  /**
   * 判断一个对象是否是集合
   * @param value
   * @return
   */
  private static boolean isArray(Object value) {
    if(value == null) {
      return false;
    }
    return value.getClass().isArray() || Iterable.class.isAssignableFrom(value.getClass());
  }

  /**
   * 将参数转换为集合
   * @param value
   * @return
   */
  @SuppressWarnings("unchecked")
  private static List<Object> getArrayValue(Object value) {
    List<Object> values = new ArrayList<>();
    if(value.getClass().isArray()) {
      Object[] objs = (Object[]) value;
      for (Object obj : objs) {
        values.add(obj);
      }
    } else if(Iterable.class.isAssignableFrom(value.getClass())) {
      Iterable<? extends Object> iterable = (Iterable<? extends Object>) value;
      iterable.forEach(v -> {
        values.add(v);
      });
    } else {
      values.add(value);
    }
    return values;
  }

}
