package com.bizunited.platform.core.repository.dynamic;

import com.bizunited.platform.core.entity.DataSourceEntity;
import org.hibernate.SessionFactory;

/**
 * 动态数据源管理器
 * @author yinwenjie
 */
public interface DynamicDataSourceManager {
  /**
   * 该方法在应用程序启动时，初始化当前系统已经设定的，且状态正常的第三方数据源信息
   * @throws 如果初始化过程出现错误，则将当前数据源置为不可用，并继续初始化下一个第三方数据源。
   */
  public void init();
  /**
   * 创建一个数据源，并供后续数据视图查询使用。<br>
   * 如果创建成功那么这个数据源基本连接信息将会写入到数据库中，并且状态为正常
   * @param dataSource 
   */
  public DataSourceEntity create(DataSourceEntity dataSource);
  /**
   * 对指定的数据源进行修改操作，只能修改其中部分信息包括：连接地址、用户名、密码、url参数信息。其它的不能修改了
   * @param dataSource 需要修改的数据源信息，必须有id
   * @return 修改后的信息将被返回
   */
  public DataSourceEntity update(DataSourceEntity dataSource);
  /**
   * 重新加载指定编号的数据源，只有处于“正常”状态的数据源能够进行重加载。<br>
   * 一旦启动失败，当前数据编号的数据源将被自动设定为“禁用”
   * @param dataSourceCode 指定的数据源编号
   */
  public void reload(String dataSourceCode);
  /**
   * 检查指定的数据源是否可用。检查方式就是用dataSource对象描述的信息基于最原始的java jdbc连接进行操作。
   * 如果当前连接成功，则测试成功。
   * @param dataSource 数据源对象描述
   * @throws 如果连接过程出现问题，则抛出运行时异常
   */
  public void check(DataSourceEntity dataSource);
  /**
   * 禁用并停止当前指定编号的数据源的工作。当前的数据源将被设定为“禁用”状态
   * @param dataSourceCode 指定的数据源编号
   */
  public void disable(String dataSourceCode);
  /**
   * 启用一个指定编号对应的第三方数据源
   * @param dataSourceCode 
   */
  public void enable(String dataSourceCode);
  /**
   * 删除指定的数据源，只有当前的数据源没有建立数据视图的情况下，这个数据源才能被删除。
   * @param dataSourceCode 指定的数据源编号
   */
  public void delete(String dataSourceCode);
  /**
   * 获取当前指定编号的数据源的session工厂。如果当前指定的数据源工作状态不正确，则返回null
   * @param dataSourceCode 指定的数据源编号
   * @return 
   */
  public SessionFactory getCurrentSessionFactory(String dataSourceCode); 
  /**
   * 基于当前使用本对象的线程，加读锁。<br>
   * 一般来说，在动态数据源被使用进行数据查询任务时，就会加读锁。
   */
  public void readLock();
  /**
   * 基于当前使用本对象的线程，解除读锁。<br>
   * 一般来说，在动态数据源被使用进行数据查询任务结束时，就会解除读锁。
   */
  public void unReadLock();
  /**
   * 基于当前使用本对象的线程，加写锁。<br>
   * 一般来说，在动态数据源管理器（本对象），运行数据源重载、停止、启动操作时，会加写锁。以避免其它线程正在使用数据源的情况下，后者配置发生变化
   */
  public void writeLock();
  /**
   * 基于当前使用本对象的线程，解除写锁。<br>
   * 一般来说，在动态数据源管理器（本对象），运行数据源重载、停止、启动操作结束后（无论成功结束还是异常技术），会解除写锁。以便其它运行数据源查询的任务能够正常运行。
   */
  public void unWriteLock();
}