package com.bizunited.platform.core.repository.dynamic;

import com.alibaba.druid.pool.DruidDataSource;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataSourceTableEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.repository.DataSourceTableRepository;
import com.bizunited.platform.core.repository.dataview.DataSourceRepository;
import com.bizunited.platform.core.repository.dataview.DataViewGroupRepository;
import com.bizunited.platform.core.repository.dataview.DataViewRepository;
import com.bizunited.platform.rbac.server.crypto.password.Aes2PasswordEncoder;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * 第三方数据源管理类的实现，注意:应用程序的主数据源不在这里进行管理
 * @author yinwenjie
 */
@Component("DynamicDataSourceManagerImpl")
public class DynamicDataSourceManagerImpl implements DynamicDataSourceManager {

  @Autowired
  private PlatformContext platformContext;
  @Autowired
  private ApplicationContext ctx;
  @Autowired
  private DataSourceRepository dataSourceRepository;
  @Autowired
  private DataViewGroupRepository dataViewGroupRepository;
  @Autowired
  private DataViewRepository dataViewRepository;
  @Autowired
  private Aes2PasswordEncoder passwordEncoder;
  @Autowired
  private DataSourceTableRepository dataSourceTableRepository;
  // 可重入的读写分离乐观锁（AQS），以便避免同一进程中同时对两个数据源进行操作
  private ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock();
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicDataSourceManagerImpl.class);
  private static final String ERROR_CODE = "未找到指定的数据源名称信息";
  private static final String ERROR_CODE_NOTNULL = "数据源名称必须传入!!";
  private static final String MESS_MYSQL = "mysql";
  private static final String MESS_ORACLE = "oracle";
  private static Map<String, String> jdbcDrives = new HashMap<>();
  
  static {
    jdbcDrives.put(MESS_MYSQL, "com.mysql.jdbc.Driver");
    jdbcDrives.put(MESS_ORACLE, "oracle.jdbc.driver.OracleDriver");
  }
  
  /**
   * 数据源bean name的前缀
   */
  private static final String BEANNAMEPREFIX = "_datasource_";
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#init()
   */
  @Override
  public void init() {
    /*
     * 初始化过程将会完成所有数据库中状态“正常”的数据源的注册：
     * 1、首先查询数据库中状态“正常”的数据源
     * 2、进行数据源设置状况的验证
     * 注意，那些初始化失败的数据源会被初始化为
     * 3、然后依次注入到spring ioc容器中（基于SessionFactory注册）
     * */
    // 1、========
    List<DataSourceEntity> dataSources = this.dataSourceRepository.findByTstatusAndProjectName(1, platformContext.getAppName());
    if(dataSources == null || dataSources.isEmpty()) {
      return;
    }
    
    // 2、进行数据源验证
    List<String> arrayErrors = new ArrayList<>(); 
    List<String> errorCode = new ArrayList<>();
    for (DataSourceEntity dataSourceEntity : dataSources) {
      String code = dataSourceEntity.getCode();
      // 进行验证，验证不通过，不能创建数据源，还要讲当前数据源置为不可用
      try {
        this.validate(dataSourceEntity);
      } catch(RuntimeException e) {
        errorCode.add(code);
        arrayErrors.add(String.format("数据源%s验证失败，原因：%s；", code , e.getMessage()));
      }
    }
    if(!arrayErrors.isEmpty()) {
      // 设置错误的数据源为不可用
      this.dataSourceRepository.updateDisable(errorCode.toArray(new String[]{}));
      throw new IllegalArgumentException(Arrays.toString(arrayErrors.toArray()));
    }
    
    // 3、进行初始化
    try {
      this.writeLock();
      for (DataSourceEntity dataSourceEntity : dataSources) {
        this.buildSessionFactory(dataSourceEntity);
      }
    } finally {
      this.unWriteLock();
    }
  }
  
  /**
   * 在IOC容器中，创建/重新创建 指定的SessionFactory对象,
   * 注：dataSourceEntity对象中密码必须为加密状态
   * @param dataSourceEntity
   */
  private void buildSessionFactory(DataSourceEntity dataSourceEntity) {
    String type = dataSourceEntity.getType();
    String password = passwordEncoder.decode(dataSourceEntity.getPassword());
    String userName = dataSourceEntity.getUserName();
    String code = dataSourceEntity.getCode();

    String url = this.getUrlFromDataSourceEntity(dataSourceEntity);

    // bean name
    String beanName = StringUtils.join(BEANNAMEPREFIX , code);
    this.unregistSessionFactory(beanName);
    DruidDataSource currentDataSource = this.getDataSource(type, userName, password, url);
    this.registSessionFactory(type , beanName , currentDataSource);
  }
  
  /**
   * 试图注销指定的bean（SessionFactory）
   * @param beanName
   */
  private void unregistSessionFactory(String beanName) {
    // 如果当前ioc容器中已经有这个bean name了，则首先移除，再进行添加
    DefaultListableBeanFactory defaultListableBeanFactory = (DefaultListableBeanFactory) ctx.getAutowireCapableBeanFactory();
    if(defaultListableBeanFactory.containsBean(beanName)) {
      defaultListableBeanFactory.removeBeanDefinition(beanName);
    }
  }
  
  /**
   * 接着依据已经生成的DataSource，注册hibernate下面的sessionFactory信息
   * TODO 后续支持Sql Server或者Oracle时，该方法将被调整
   * @return 
   */
  private void registSessionFactory(String type , String beanName , DataSource dataSource) {
    DefaultListableBeanFactory defaultListableBeanFactory = (DefaultListableBeanFactory) ctx.getAutowireCapableBeanFactory();
    BeanDefinitionBuilder beanDefinitionBuilder = BeanDefinitionBuilder.genericBeanDefinition(LocalSessionFactoryBean.class);
    beanDefinitionBuilder.addPropertyValue("dataSource", dataSource);
    Map<String, String> properties = new HashMap<>();
    // 这里可以设置session的其它参数
    if(StringUtils.equalsIgnoreCase(type, MESS_MYSQL)) {
      properties.put("hibernate.dialect", "org.hibernate.dialect.MySQLDialect");
    } else {
      properties.put("hibernate.dialect", "org.hibernate.dialect.Oracle10gDialect");
    }
    
    properties.put("hibernate.current_session_context_class", "thread");
    properties.put("hibernate.show_sql", "true");
    properties.put("hibernate.format_sql", "true");
    beanDefinitionBuilder.addPropertyValue("hibernateProperties", properties);
    beanDefinitionBuilder.setLazyInit(false);
    
    AbstractBeanDefinition beanDefinition = beanDefinitionBuilder.getBeanDefinition();
    defaultListableBeanFactory.registerBeanDefinition(beanName , beanDefinition);
  }
  
  /**
   * 获取DataSource
   * @param type 数据库类型，目前支持mysql和oracle
   * @param userName 用户名
   * @param password 密码
   * @param url url连接信息
   * @return
   */
  private DruidDataSource getDataSource(String type , String userName , String password , String url) {
    DruidDataSource dataSource = new DruidDataSource();
    if(StringUtils.equalsIgnoreCase(MESS_MYSQL, type)) {
      dataSource.setDriverClassName(jdbcDrives.get(MESS_MYSQL));
      dataSource.setPoolPreparedStatements(false);
    } else {
      dataSource.setDriverClassName(jdbcDrives.get(MESS_ORACLE));
    }
    dataSource.setPassword(password);
    dataSource.setUsername(userName);
    dataSource.setUrl(url);
    // TODO 以下设定值暂时是固定的，后期再开放
    dataSource.setInitialSize(5);
    dataSource.setMinIdle(2); 
    dataSource.setMaxActive(10);
    
    return dataSource;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#check(com.bizunited.platform.kuiper.entity.DataSourceEntity)
   */
  @Override
  public void check(DataSourceEntity dataSource) {
    /*
     * 检查过程很简单，既是使用当前对象中的数据连接信息，直接通过JDBC进行目标数据库的链接
     * 这个过程如果出现任何异常情况，则报错。
     * 特别需要注意，因为密码做了加密处理，所以在修改验证时，会将传入的密码与数据库的密码比较是否一致，
     * 如果一致则解密，否则作为新密码处理
     * */
    this.validate(dataSource);
    String type = dataSource.getType();
    try {
      if(StringUtils.equalsIgnoreCase(MESS_MYSQL, type)) {
        Class.forName(jdbcDrives.get(MESS_MYSQL));
      } else {
        Class.forName(jdbcDrives.get(MESS_ORACLE));
      }
    } catch(ClassNotFoundException  e) {
      LOGGER.error(e.getMessage(),e);
      throw new IllegalArgumentException(e);
    }
    
    // 开始建立连接
    String username = dataSource.getUserName();
    String password = dataSource.getPassword();
    if(StringUtils.isNotBlank(dataSource.getCode())) {
      // 如果传入的密码与数据库存在的密码一致，则解密密码
      DataSourceEntity dbDataSource = dataSourceRepository.findByCode(dataSource.getCode());
      if(dbDataSource != null && dbDataSource.getPassword().equals(dataSource.getPassword())) {
        password = passwordEncoder.decode(dataSource.getPassword());
      }
    }

    String url = this.getUrlFromDataSourceEntity(dataSource);
    
    Connection conn = null;
    try {
      conn = DriverManager.getConnection(url, username, password);
    } catch (SQLException e) {
      throw new IllegalArgumentException(e);
    } finally {
      this.closeConnection(conn);
    }
  }

  private String getUrlFromDataSourceEntity(DataSourceEntity dataSource) {
    if(StringUtils.equalsIgnoreCase(MESS_MYSQL, dataSource.getType())) {
      if (StringUtils.isBlank(dataSource.getUrlParams())) {
        return String.format("jdbc:mysql://%s:%d/%s", dataSource.getAddress(), dataSource.getPort(), dataSource.getDbName());
      }
      return String.format("jdbc:mysql://%s:%d/%s?%s", dataSource.getAddress(), dataSource.getPort(), dataSource.getDbName(), dataSource.getUrlParams());
    }

    return String.format("jdbc:oracle:thin:@%s:%d:%s", dataSource.getAddress() , dataSource.getPort() , dataSource.getDbName());
  }
  
  private void closeConnection(Connection conn) {
    try {
      if(conn != null) {
        conn.close();
      }
    } catch(SQLException e) {
      throw new IllegalArgumentException(e);
    } 
  }
  
  /**
   * 该私有方法负责进行检验
   * @param dataSource
   */
  private void validate(DataSourceEntity dataSource) {
    Validate.notNull(dataSource , "进行操作时，必须对数据源进行描述");
    String address = dataSource.getAddress();
    Validate.notBlank(address , "进行操作时，必须传入数据库连接地址（IP或者域名都可）");
    String dbName = dataSource.getDbName();
    Validate.notBlank(dbName , "进行操作时，必须传入连接的数据库名");
    String password = dataSource.getPassword();
    Validate.notBlank(password , "操作时，必须传入数据库密码");
    Integer port = dataSource.getPort();
    Validate.notNull(port , "数据库连接端口必须填写!!");
    String urlParams = dataSource.getUrlParams();
    if(StringUtils.isBlank(urlParams)) {
      dataSource.setUrlParams("");
    }
    String username = dataSource.getUserName();
    Validate.notBlank(username , "操作时，必须传入数据库用户名");
    String type = dataSource.getType();
    Validate.notBlank(type , "操作时，必须传入数据库类型（Mysql/Oracle）");
    Validate.isTrue(StringUtils.equalsIgnoreCase(type, MESS_MYSQL) || StringUtils.equalsIgnoreCase(type, MESS_ORACLE)
      , "目前数据视图只支持MySQL和Oracle数据库!!");
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#create(com.bizunited.platform.kuiper.entity.DataSourceEntity)
   */
  @Override
  public DataSourceEntity create(DataSourceEntity dataSource) {
    /*
     * 创建数据源的动作包括：
     * 1、进行信息的基本检查，以及重复性检查
     * 2、检查这个数据源的连接性，如果连接不上，则提示错误
     * 3、生成数据源对象，并在IOC容器中注册相关session factory
     * 4、插入到数据库
     * */
    // 1、=======
    this.validate(dataSource);
    String code = dataSource.getCode();
    DataSourceEntity oldDataSource = this.dataSourceRepository.findByCode(code);
    Validate.isTrue(oldDataSource == null , "当前数据源名称已经存在，请重新填写!!");
    // 2、=======
    this.check(dataSource);
    // 3和4、=======
    try {
      this.writeLock();
      dataSource.setPassword(passwordEncoder.encode(dataSource.getPassword()));
      this.buildSessionFactory(dataSource);
      dataSource.setCreateTime(new Date());
      dataSource.setProjectName(platformContext.getAppName());
      this.dataSourceRepository.save(dataSource);
    } finally {
      this.unWriteLock();
    }
    
    return dataSource;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#update(com.bizunited.platform.kuiper.entity.DataSourceEntity)
   */
  @Override
  public DataSourceEntity update(DataSourceEntity dataSource) {
    /*
     * 修改操作只能对地址、用户名、密码、端口、数据库名、参数进行修改
     * */
    Validate.notNull(dataSource , "修改信息必须传入!!");
    String code = dataSource.getCode();
    Validate.notBlank(code , "编码信息必须传入!!");
    DataSourceEntity current = this.dataSourceRepository.findByCode(dataSource.getCode());
    Validate.notNull(current , "没有查到当前编码对应的数据库信息!!");
    Validate.isTrue(current.getTstatus() == NormalStatusEnum.DISABLE.getStatus(), "只有数据源状态禁用的情况下，才能进行修改!!");
    
    current.setAddress(dataSource.getAddress());
    current.setDbName(dataSource.getDbName());
    current.setUserName(dataSource.getUserName());
    if(!current.getPassword().equals(dataSource.getPassword())) {
      current.setPassword(passwordEncoder.encode(dataSource.getPassword()));
    }
    current.setPort(dataSource.getPort());
    current.setUrlParams(dataSource.getUrlParams());
    current.setTstatus(dataSource.getTstatus());
    this.validate(current);
    this.dataSourceRepository.save(current);
    if(current.getTstatus().equals(NormalStatusEnum.ENABLE.getStatus())) {
      // 检查数据源是否能连接，并且完成数据源重加在操作
      this.reload(current.getCode());
    }
    return current;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#reload(java.lang.String)
   */
  @Override
  public void reload(String dataSourceCode) {
    /*
     * 重新加载过程如下：
     * 1、首先查询这个code的信息是否存在，状态是否正确
     * 2、然后检测连接是否可用
     * 3、调用buildSessionFactory方法，重新进行注册（这里会自动删除之前的IOC对象）
     * */
    // 1、=====
    DataSourceEntity currentDataSource = this.dataSourceRepository.findByCode(dataSourceCode);
    Validate.notNull(currentDataSource , ERROR_CODE);
    // 2、=====
    try {
      this.check(currentDataSource);
    } catch(RuntimeException e) {
      // 如果出现异常，则将当前数据源置为不可用
      this.dataSourceRepository.updateDisable(new String[]{dataSourceCode});
      throw new IllegalArgumentException(e.getMessage() , e);
    }
    // 3、======
    try {
      this.writeLock();
      this.buildSessionFactory(currentDataSource);
    } finally {
      this.unWriteLock();
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#disable(java.lang.String)
   */
  @Override
  public void disable(String dataSourceCode) {
    Validate.notBlank(dataSourceCode , ERROR_CODE_NOTNULL);
    DataSourceEntity currentDataSource = this.dataSourceRepository.findByCode(dataSourceCode);
    Validate.notNull(currentDataSource , ERROR_CODE);
    /*
     * 禁用操作如下：
     * 1、无论数据库中的状态如何，都先禁用数据库中的信息
     * 2、然后无论数据库中的数据状态如何，都试图从IOC容器中进行注销
     * */
    
    // 1、=====
    this.dataSourceRepository.updateDisable(new String[]{dataSourceCode});
    
    // 2、===== 
    String code = currentDataSource.getCode();
    String beanName = StringUtils.join(BEANNAMEPREFIX , code);
    try {
      this.writeLock();
      this.unregistSessionFactory(beanName);
    } finally {
      this.unWriteLock();
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#enable(java.lang.String)
   */
  @Override
  public void enable(String dataSourceCode) {
    Validate.notBlank(dataSourceCode , ERROR_CODE_NOTNULL);
    DataSourceEntity currentDataSource = this.dataSourceRepository.findByCode(dataSourceCode);
    Validate.notNull(currentDataSource , ERROR_CODE);
    
    /*
     * 处理过程如下：
     * 1、首先无论数据库中记录的状态如何，这里都修改一次数据库中的状态
     * 2、接着再视图重新注册IOC容器中的数据源对象
     * */
    // 1、=======
    this.dataSourceRepository.updateEnable(new String[]{dataSourceCode});
    
    // 2、这里只需要重新加载一次数据源即可
    try {
      this.writeLock();
      this.reload(dataSourceCode);
    } finally {
      this.unWriteLock();
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#delete(java.lang.String)
   */
  @Override
  public void delete(String dataSourceCode) {
    // 注意，这里是真删除
    //1.判断是否有视图存在，不管视图是否禁用，只要存在，就不能被真删除
    //2.判断是否分组可以被删除，原则上分组中只要存在其他的视图，就不能被真删除，因为有外键信息关联数据源
    //3.如果进行数据源删除，必须先删除视图，再删除分组，最后才能删除数据源
    Validate.notBlank(dataSourceCode , ERROR_CODE_NOTNULL);
    DataSourceEntity currentDataSource = this.dataSourceRepository.findByCode(dataSourceCode);
    Validate.notNull(currentDataSource , "根据数据源名称信息，未能获取数据源信息，请检查!!");
    Validate.notBlank(currentDataSource.getId(), "根据数据源名称信息，未能获取数据源信息，请检查!!");
    List<DataViewGroupEntity> groupList = this.dataViewGroupRepository.findByDataSourceCode(dataSourceCode);
    List<DataViewEntity> viewList = this.dataViewRepository.findByDataSource(currentDataSource.getId());
    Validate.isTrue(CollectionUtils.isEmpty(viewList) , "当前数据源已存在正在使用的数据视图信息，不能强制删除!!");
    
    if(!CollectionUtils.isEmpty(groupList)) {
      for(DataViewGroupEntity group : groupList) {
        List<DataViewEntity> tempViewList =  dataViewRepository.findByDataViewGroup(group.getId());
        Validate.isTrue(CollectionUtils.isEmpty(tempViewList), "%s分组中还存在其他的视图信息，不能清除分组信息!!",group.getGroupName());
      }
      dataViewGroupRepository.deleteAll(groupList);
    }
    List<DataSourceTableEntity> dataSourceTables = dataSourceTableRepository.findByDataSource(currentDataSource.getId());
    Validate.isTrue(CollectionUtils.isEmpty(dataSourceTables), "有模板正在使用该数据源，不能删除");
    this.dataSourceRepository.delete(currentDataSource);
    
    // 试图注销可能存在在IOC容器中的bean
    try {
      this.writeLock();
      String beanName = StringUtils.join(BEANNAMEPREFIX , dataSourceCode);
      this.unregistSessionFactory(beanName);
    } finally {
      this.unWriteLock();
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#getCurrentSessionFactory(java.lang.String)
   */
  @Override
  public SessionFactory getCurrentSessionFactory(String dataSourceCode) {
    if(StringUtils.isBlank(dataSourceCode)) {
      return null;
    }
    
    // 只有存在于ctx ioc容器中的bean能够被找到
    String beanName = StringUtils.join(BEANNAMEPREFIX , dataSourceCode);
    return ctx.getBean(beanName,SessionFactory.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#readLock()
   */
  @Override
  public void readLock() {
    this.readWriteLock.readLock().lock();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#unReadLock()
   */
  @Override
  public void unReadLock() {
    this.readWriteLock.readLock().unlock();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#writeLock()
   */
  @Override
  public void writeLock() {
    this.readWriteLock.writeLock().lock();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.repository.dynamic.DynamicDataSourceManager#unWriteLock()
   */
  @Override
  public void unWriteLock() {
    readWriteLock.writeLock().unlock();
  }
}
