package com.bizunited.platform.core.repository.internal;

import com.bizunited.platform.common.repository.PageRepositoryImpl;
import com.bizunited.platform.core.entity.CodeRuleEntity;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Repository;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;

/**
 * CodeRuleRepositoryImpl
 *
 * @description:
 * @author: yanwe
 * @date: 12/Jun/2019 15:01
 */
@Repository("CodeRuleRepositoryImpl")
public class CodeRuleRepositoryImpl implements CodeRuleRepositoryCustom, PageRepositoryImpl {

  @Autowired
  @PersistenceContext
  private EntityManager entityManager;
  private static final String RULE_NAME = "ruleName";
  private static final String RULE_CODE = "ruleCode";
  private static final String RULE_TYPE = "ruleType";
  private static final String LANGUAGE = "language";
  
  @Override
  public Page<CodeRuleEntity> findByConditions(Pageable pageable, Map<String, Object> conditions) {
    /*
     * 目前的查询条件是：表单名、业务领域、项目名
     * */
    StringBuilder hql = new StringBuilder("from CodeRuleEntity cgr where 1=1");
    StringBuilder countHql = new StringBuilder("select count(*) from CodeRuleEntity cgr where 1=1");
    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>();

    // 编码生成规则的名称
    Object ruleName = conditions.get(RULE_NAME);
    if(ruleName != null) {
      condition.append(" AND cgr.ruleName = :ruleName ");
      parameters.put(RULE_NAME, ruleName);
    }
    // 编码生成规则的编号
    Object ruleCode = conditions.get(RULE_CODE);
    if(ruleCode != null) {
      condition.append(" AND cgr.ruleCode = :ruleCode ");
      parameters.put(RULE_CODE, ruleCode);
    }

    // 编码生成规则的类别
    Object ruleType = conditions.get(RULE_TYPE);
    if(ruleType != null) {
      condition.append(" AND cgr.ruleType = :ruleType ");
      parameters.put(RULE_TYPE, ruleType);
    }

    // 编码生成规则的说明
    Object language = conditions.get(LANGUAGE);
    if(language != null) {
      condition.append(" AND cgr.language = :language ");
      parameters.put(LANGUAGE, language);
    }

    // 项目名称
    String projectName = (String) conditions.get(PROJECT_NAME);
    if(StringUtils.isNotBlank(projectName)) {
      condition.append(" AND cgr.projectName = :projectName ");
      parameters.put(PROJECT_NAME, projectName);
    }else {
      condition.append(" AND (cgr.projectName = '' OR cgr.projectName IS NULL)");
    }
    hql.append(condition).append(" order by cgr.createTime desc  ");
    countHql.append(condition);
    return queryByConditions(entityManager, hql.toString(), countHql.toString(), parameters, pageable, false, null);
  }

  /**
   * 按照条件搜索编码生成规则，不支持模糊查询,不分页
   * @param codeRule
   * @return
   */
  @SuppressWarnings("unchecked")
  @Override
  public List<CodeRuleEntity> findAllByConditions(CodeRuleEntity codeRule) {
    StringBuilder hql = new StringBuilder("from CodeRuleEntity cgr where 1=1");
    StringBuilder condition = new StringBuilder();
    Map<String, Object> parameters = new HashMap<>();

    // 编码生成规则的名称
    if(StringUtils.isNotBlank(codeRule.getRuleName())) {
      condition.append(" AND cgr.ruleName = :ruleName ");
      parameters.put(RULE_NAME, codeRule.getRuleName());
    }
    // 编码生成规则的编号
    if(StringUtils.isNotBlank(codeRule.getRuleCode())) {
      condition.append(" AND cgr.ruleCode = :ruleCode ");
      parameters.put(RULE_CODE, codeRule.getRuleCode());
    }

    // 编码生成规则的类别
    if(StringUtils.isNotBlank(codeRule.getRuleType())) {
      condition.append(" AND cgr.ruleType = :ruleType ");
      parameters.put(RULE_TYPE, codeRule.getRuleType());
    }

    // 编码生成规则的说明
    if(StringUtils.isNotBlank(codeRule.getLanguage())) {
      condition.append(" AND cgr.language = :language ");
      parameters.put(LANGUAGE, codeRule.getLanguage());
    }

    // 项目名称
    if(StringUtils.isNotBlank(codeRule.getProjectName())) {
      condition.append(" AND cgr.projectName = :projectName ");
      parameters.put(PROJECT_NAME, codeRule.getProjectName());
    }else {
      condition.append(" AND (cgr.projectName = '' OR cgr.projectName IS NULL)");
    }
    hql.append(condition).append(" order by cgr.createTime desc  ");
    Query query = entityManager.createQuery(hql.toString());
    parameters.forEach((k, v) -> {
      query.setParameter(k, v);
    });
    return query.getResultList();
  }
}
