package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.vo.UuidVo;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthHorizontalRelationRepository;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalRelationService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * DataViewAuthHorizontalRelationEntity业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("DataViewAuthHorizontalRelationServiceImpl")
public class DataViewAuthHorizontalRelationServiceImpl implements DataViewAuthHorizontalRelationService {
  @Autowired
  private DataViewAuthHorizontalRelationRepository dataViewAuthHorizontalRelationRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private DataViewAuthHorizontalService dataViewAuthHorizontalService;
  @Autowired
  private DataViewAuthRelationService dataViewAuthRelationService;

  @Override
  public Set<DataViewAuthHorizontalRelationEntity> findDetailsByAuthHorizontal(String authHorizontal) {
    if (StringUtils.isBlank(authHorizontal)) {
      return Sets.newHashSet();
    }
    DataViewAuthHorizontalEntity horizontal = dataViewAuthHorizontalService.findById(authHorizontal);
    if (horizontal == null) {
      return Sets.newHashSet();
    }
    Set<DataViewAuthHorizontalRelationEntity> relations = this.dataViewAuthHorizontalRelationRepository.findDetailsByAuthHorizontal(authHorizontal);
    if (CollectionUtils.isEmpty(relations)) {
      return relations;
    }
    for (DataViewAuthHorizontalRelationEntity relation : relations) {
      UuidVo relationObj = dataViewAuthRelationService.findRelation(relation.getAuthRelationIds(), horizontal.getAuthType());
      relation.setRelation(relationObj);
    }
    return relations;
  }

  @Override
  public DataViewAuthHorizontalRelationEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.dataViewAuthHorizontalRelationRepository.findDetailsById(id);
  }

  @Override
  public DataViewAuthHorizontalRelationEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.dataViewAuthHorizontalRelationRepository.findById(id).orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    Optional<DataViewAuthHorizontalRelationEntity> op = this.dataViewAuthHorizontalRelationRepository.findById(id);
    op.ifPresent(dataViewAuthHorizontalRelationEntity -> this.dataViewAuthHorizontalRelationRepository.delete(dataViewAuthHorizontalRelationEntity));
  }

  @Override
  @Transactional
  public void deleteByAuthHorizontal(String authHorizontal) {
    Validate.notBlank(authHorizontal, "进行删除时，必须给定横向数据权限信息!");
    this.dataViewAuthHorizontalRelationRepository.deleteByAuthHorizontal(authHorizontal);
  }

  /**
   * 处理可能的更新、增添、删除权限领域信息
   *
   * @param dataViewAuthHorizontalEntity
   * @param authRelations
   */
  @Override
  @Transactional
  public Set<DataViewAuthHorizontalRelationEntity> save(DataViewAuthHorizontalEntity authHorizontal, Set<DataViewAuthHorizontalRelationEntity> authRelations) {
    this.saveValidation(authHorizontal, authRelations);
    if (authRelations == null) {
      authRelations = Sets.newHashSet();
    }
    Map<String, DataViewAuthHorizontalRelationEntity> authRelationsMap = authRelations.stream().collect(Collectors.toMap(DataViewAuthHorizontalRelationEntity::getId, r -> r, (r1, r2) -> r1));
    Set<DataViewAuthHorizontalRelationEntity> dbAuthRelations = dataViewAuthHorizontalRelationRepository.findDetailsByAuthHorizontal(authHorizontal.getId());
    Set<DataViewAuthHorizontalRelationEntity> deletes = Sets.newHashSet();
    Set<DataViewAuthHorizontalRelationEntity> updates = Sets.newHashSet();
    Set<DataViewAuthHorizontalRelationEntity> creates = Sets.newHashSet();
    nebulaToolkitService.collectionDiscrepancy(authRelations, dbAuthRelations, DataViewAuthHorizontalRelationEntity::getId, deletes, updates, creates);
    for (DataViewAuthHorizontalRelationEntity delete : deletes) {
      dataViewAuthHorizontalRelationRepository.delete(delete);
    }
    for (DataViewAuthHorizontalRelationEntity update : updates) {
      DataViewAuthHorizontalRelationEntity relation = authRelationsMap.get(update.getId());
      update.setAuthRelationIds(relation.getAuthRelationIds());
      update.setSortIndex(relation.getSortIndex());
      dataViewAuthHorizontalRelationRepository.save(update);
    }
    for (DataViewAuthHorizontalRelationEntity create : creates) {
      create.setAuthHorizontal(authHorizontal);
      dataViewAuthHorizontalRelationRepository.save(create);
    }
    updates.addAll(creates);
    return updates;
  }

  /**
   * 保存数据的验证
   *
   * @param authHorizontal
   * @param authRelations
   */
  private void saveValidation(DataViewAuthHorizontalEntity authHorizontal, Set<DataViewAuthHorizontalRelationEntity> relations) {
    Validate.notNull(authHorizontal, "横向权限不能为空");
    Validate.notBlank(authHorizontal.getId(), "横向权限ID不能为空");
    DataViewAuthHorizontalEntity horizontal = dataViewAuthHorizontalService.findById(authHorizontal.getId());
    Validate.notNull(horizontal, "未找到横向权限：%s", authHorizontal.getId());
    if (CollectionUtils.isEmpty(relations)) {
      return;
    }
    for (DataViewAuthHorizontalRelationEntity authRelation : relations) {
      Validate.notBlank(authRelation.getAuthRelationIds(), "权限值不能为空");
      Validate.notNull(authRelation.getSortIndex(), "权限值排序不能为空");
    }
  }

}
