package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.entity.DataViewAuthInterceptorEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthRepository;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.bizunited.platform.core.service.dataview.DataViewAuthInterceptorService;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * 数据视图权限服务的接口实现
 * @Author: Paul Chan
 * @Date: 2020-04-23 11:19
 */
@Service("DataViewAuthServiceImpl")
public class DataViewAuthServiceImpl implements DataViewAuthService {

  @Autowired
  private DataViewAuthRepository dataViewAuthRepository;
  @Autowired
  private DataViewService dataViewService;
  @Autowired
  private DataViewAuthVerticalService dataViewAuthVerticalService;
  @Autowired
  private DataViewAuthHorizontalService dataViewAuthHorizontalService;
  @Autowired
  private DataViewAuthInterceptorService dataViewAuthInterceptorService;

  @Override
  @Transactional
  public DataViewAuthEntity create(DataViewAuthEntity dataViewAuth) {
    this.createValidation(dataViewAuth);
    // 这里的调用主要是为了验证
    SecurityUtils.getCurrentUser();
    dataViewAuth.setCreateTime(new Date());
    dataViewAuthRepository.save(dataViewAuth);
    return dataViewAuth;
  }

  /**
   * 创建验证
   * @param dataViewAuth
   */
  private void createValidation(DataViewAuthEntity dataViewAuth) {
    Validate.notNull(dataViewAuth, "传入的对象不能为空");
    Validate.notNull(dataViewAuth.getDataView(), "关联的数据视图不能为空");
    String dataViewId = dataViewAuth.getDataView().getId();
    Validate.notBlank(dataViewId, "关联的数据视图ID不能为空");
    DataViewEntity dataView = dataViewService.findById(dataViewId);
    Validate.notNull(dataView, "未找到数据视图:%s", dataViewId);
    Validate.notBlank(dataViewAuth.getCode(), "权限编码不能为空");
    Validate.notBlank(dataViewAuth.getName(), "权限名称不能为空");
    Validate.isTrue(dataViewAuth.getId() == null, "新增数据不能传入ID");
    long count = dataViewAuthRepository.countByCode(dataViewAuth.getCode());
    Validate.isTrue(count  == 0L, "重复的权限编码：%s", dataViewAuth.getCode());
  }

  /**
   * 更新数据验证
   * @param dataViewAuth
   */
  private void updateValidation(DataViewAuthEntity dataViewAuth) {
    Validate.notNull(dataViewAuth, "传入的对象不能为空");
    Validate.notBlank(dataViewAuth.getCode(), "权限编码不能为空");
    Validate.notBlank(dataViewAuth.getName(), "权限名称不能为空");
    Validate.notBlank(dataViewAuth.getId(), "更新数据必须传入ID");
    long count = dataViewAuthRepository.countByCodeExcludeId(dataViewAuth.getCode(), dataViewAuth.getId());
    Validate.isTrue(count  == 0L, "重复的权限编码：%s", dataViewAuth.getCode());
  }

  @Override
  @Transactional
  public DataViewAuthEntity update(DataViewAuthEntity dataViewAuth) {
    this.updateValidation(dataViewAuth);
    DataViewAuthEntity dbAuth = dataViewAuthRepository.findById(dataViewAuth.getId()).orElse(null);
    Validate.notNull(dbAuth, "更新的对象不存在");
    dbAuth.setCode(dataViewAuth.getCode());
    dbAuth.setName(dataViewAuth.getName());
    dataViewAuthRepository.save(dbAuth);
    return dbAuth;
  }

  @Override
  public DataViewAuthEntity findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return dataViewAuthRepository.findById(id).orElse(null);
  }

  @Override
  public DataViewAuthEntity findDetailsByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    DataViewAuthEntity auth = dataViewAuthRepository.findDetailsByCode(code);
    if(auth == null) {
      return auth;
    }
    Set<DataViewAuthVerticalEntity> verticals = dataViewAuthVerticalService.findDetailsByAuthId(auth.getId());
    Set<DataViewAuthHorizontalEntity> horizontals = dataViewAuthHorizontalService.findDetailsByAuthId(auth.getId());
    Set<DataViewAuthInterceptorEntity> interceptors = dataViewAuthInterceptorService.findByAuthId(auth.getId());
    auth.setVerticalAuths(verticals);
    auth.setHorizontalAuths(horizontals);
    auth.setInterceptors(interceptors);
    return auth;
  }

  @Override
  public Set<DataViewAuthEntity> findByDataView(String dataViewCode) {
    if(StringUtils.isBlank(dataViewCode)) {
      return Sets.newHashSet();
    }
    return dataViewAuthRepository.findByDataViewCode(dataViewCode);
  }

  @Override
  public DataViewAuthEntity findDetailsByDataViewCodeAndCode(String dataViewCode, String authCode) {
    if(StringUtils.isAnyBlank(dataViewCode, authCode)) {
      return null;
    }
    return dataViewAuthRepository.findDetailsByDataViewCodeAndCode(dataViewCode, authCode);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    Validate.notBlank(id, "主键ID不能为空");
    DataViewAuthEntity auth = dataViewAuthRepository.findById(id).orElse(null);
    if(auth == null) {
      return;
    }
    dataViewAuthVerticalService.deleteByAuthId(id);
    dataViewAuthHorizontalService.deleteByAuthId(id);
    dataViewAuthInterceptorService.deleteByAuthId(id);
    dataViewAuthRepository.deleteById(id);
  }

  @Override
  public Set<DataViewAuthEntity> findDetailsByDataViewIds(String[] dataviewIds) {
    if(dataviewIds == null || dataviewIds.length == 0) {
      return Sets.newHashSet();
    }
    Set<DataViewAuthEntity> auths = dataViewAuthRepository.findDetailsByDataViewIds(dataviewIds);
    if(CollectionUtils.isEmpty(auths)) {
      return auths;
    }
    for (DataViewAuthEntity auth : auths) {
      Set<DataViewAuthVerticalEntity> verticals = dataViewAuthVerticalService.findDetailsByAuthId(auth.getId());
      Set<DataViewAuthHorizontalEntity> horizontals = dataViewAuthHorizontalService.findDetailsByAuthId(auth.getId());
      Set<DataViewAuthInterceptorEntity> interceptors = dataViewAuthInterceptorService.findByAuthId(auth.getId());
      auth.setVerticalAuths(verticals);
      auth.setHorizontalAuths(horizontals);
      auth.setInterceptors(interceptors);
    }
    return auths;
  }

  @Override
  @Transactional
  public void importAuths(List<DataViewAuthEntity> auths, MigrateImportModel importModel) {
    if(CollectionUtils.isEmpty(auths)) {
      importModel.appendLine("不需要导入数据权限");
      return;
    }
    importModel.appendLine("开始批量导入数据权限");
    for (DataViewAuthEntity auth : auths) {
      this.importAuth(auth, importModel);
    }
    importModel.appendLine("批量导入数据权限成功");
    importModel.appendLine("完成数据权限导入");
  }

  @Override
  @Transactional
  public void importAuth(DataViewAuthEntity importAuth, MigrateImportModel importModel) {
    Validate.notNull(importAuth, "导入的数据权限不能为空");
    Validate.notBlank(importAuth.getCode(), "导入的数据权限编码不能为空");
    importModel.append("开始导入数据权限：").appendLine(importAuth.getName());
    DataViewEntity importDataView = importAuth.getDataView();
    Validate.notNull(importDataView, "导入的数据权限缺失数据视图，请检查数据！！");
    DataViewEntity dataView = dataViewService.findByCode(importDataView.getCode());
    Validate.notNull(dataView, "未找到数据权限关联的数据视图：%s", importDataView.getCode());
    importAuth.setDataView(dataView);
    this.handleImportId(importAuth);
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    long count = dataViewAuthRepository.countByCode(importAuth.getCode());
    if(count > 0 && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("数据权限已存在，跳过本次导入");
      return;
    }
    if(count > 0 && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("数据权限已存在，执行更新导入");
      this.handleUpdateData(importAuth);
      return;
    }
    if(count == 0) {
      importModel.appendLine("执行数据权限的新增导入");
      this.handleCreateData(importAuth);
    }
  }

  /**
   * 处理更新的数据
   * @param importAuth
   */
  private void handleUpdateData(DataViewAuthEntity importAuth) {
    DataViewAuthEntity dbAuth = dataViewAuthRepository.findDetailsByCode(importAuth.getCode());
    Validate.notNull(dbAuth, "数据权限不存在");
    importAuth.setId(dbAuth.getId());
    this.update(importAuth);
    this.saveAuthDetails(importAuth);
  }

  /**
   * 保存数据权限的详情信息
   */
  private void saveAuthDetails(DataViewAuthEntity importAuth) {
    dataViewAuthVerticalService.save(importAuth.getId(), importAuth.getVerticalAuths());
    dataViewAuthHorizontalService.save(importAuth.getId(), importAuth.getHorizontalAuths());
    dataViewAuthInterceptorService.save(importAuth.getId(), importAuth.getInterceptors());
  }

  /**
   * 处理新增的数据权限
   * @param importAuth
   */
  private void handleCreateData(DataViewAuthEntity importAuth) {
    dataViewAuthRepository.save(importAuth);
    this.saveAuthDetails(importAuth);
  }


  /**
   * 处理导入数据的ID为null
   * @param importAuth
   */
  private void handleImportId(DataViewAuthEntity importAuth) {
    importAuth.setId(null);
    Set<DataViewAuthHorizontalEntity> horizontalAuths = importAuth.getHorizontalAuths();
    if(!CollectionUtils.isEmpty(horizontalAuths)) {
      for (DataViewAuthHorizontalEntity horizontalAuth : horizontalAuths) {
        horizontalAuth.setId(null);
        Set<DataViewAuthHorizontalRelationEntity> relations = horizontalAuth.getAuthRelations();
        if(!CollectionUtils.isEmpty(relations)) {
          for (DataViewAuthHorizontalRelationEntity relation : relations) {
            relation.setId(null);
          }
        }
      }
    }
    Set<DataViewAuthVerticalEntity> verticalAuths = importAuth.getVerticalAuths();
    if(!CollectionUtils.isEmpty(verticalAuths)) {
      for (DataViewAuthVerticalEntity verticalAuth : verticalAuths) {
        verticalAuth.setId(null);
        Set<DataViewAuthVerticalRelationEntity> relations = verticalAuth.getAuthRelations();
        if(!CollectionUtils.isEmpty(relations)) {
          for (DataViewAuthVerticalRelationEntity relation : relations) {
            relation.setId(null);
          }
        }
      }
    }
    Set<DataViewAuthInterceptorEntity> interceptors = importAuth.getInterceptors();
    if(!CollectionUtils.isEmpty(interceptors)) {
      for (DataViewAuthInterceptorEntity interceptor : interceptors) {
        interceptor.setId(null);
      }
    }
  }

}
