package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.vo.UuidVo;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthVerticalRelationRepository;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalRelationService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * DataViewAuthVerticalRelationEntity业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("DataViewAuthVerticalRelationServiceImpl")
public class DataViewAuthVerticalRelationServiceImpl implements DataViewAuthVerticalRelationService {
  @Autowired
  private DataViewAuthVerticalRelationRepository dataViewAuthVerticalRelationRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private DataViewAuthVerticalService dataViewAuthVerticalService;
  @Autowired
  private DataViewAuthRelationService dataViewAuthRelationService;

  @Override
  public Set<DataViewAuthVerticalRelationEntity> findDetailsByAuthVertical(String authVertical) {
    if (StringUtils.isBlank(authVertical)) {
      return Sets.newHashSet();
    }
    DataViewAuthVerticalEntity vertical = dataViewAuthVerticalService.findById(authVertical);
    if (vertical == null) {
      return Sets.newHashSet();
    }
    Set<DataViewAuthVerticalRelationEntity> relations = this.dataViewAuthVerticalRelationRepository.findDetailsByAuthVertical(authVertical);
    if (CollectionUtils.isEmpty(relations)) {
      return relations;
    }
    for (DataViewAuthVerticalRelationEntity relation : relations) {
      UuidVo relationObj = dataViewAuthRelationService.findRelation(relation.getAuthRelationIds(), vertical.getAuthType());
      relation.setRelation(relationObj);
    }
    return relations;
  }

  @Override
  public DataViewAuthVerticalRelationEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.dataViewAuthVerticalRelationRepository.findDetailsById(id);
  }

  @Override
  public DataViewAuthVerticalRelationEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<DataViewAuthVerticalRelationEntity> op = this.dataViewAuthVerticalRelationRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    Optional<DataViewAuthVerticalRelationEntity> op = this.dataViewAuthVerticalRelationRepository.findById(id);
    op.ifPresent(dataViewAuthVerticalRelationEntity -> this.dataViewAuthVerticalRelationRepository.delete(dataViewAuthVerticalRelationEntity));
  }

  @Override
  @Transactional
  public void deleteByAuthVertical(String authVerticalId) {
    Validate.notBlank(authVerticalId, "进行删除时，必须给定纵向数据权限主键信息!!");
    dataViewAuthVerticalRelationRepository.deleteByAuthVertical(authVerticalId);
  }

  @Override
  @Transactional
  public Set<DataViewAuthVerticalRelationEntity> save(DataViewAuthVerticalEntity authVertical, Set<DataViewAuthVerticalRelationEntity> authRelations) {
    this.saveValidation(authVertical, authRelations);
    if (authRelations == null) {
      authRelations = Sets.newHashSet();
    }
    Map<String, DataViewAuthVerticalRelationEntity> authRelationsMap = authRelations.stream().collect(Collectors.toMap(DataViewAuthVerticalRelationEntity::getId, r -> r, (r1, r2) -> r1));
    Set<DataViewAuthVerticalRelationEntity> dbAuthRelations = dataViewAuthVerticalRelationRepository.findDetailsByAuthVertical(authVertical.getId());
    Set<DataViewAuthVerticalRelationEntity> deletes = Sets.newHashSet();
    Set<DataViewAuthVerticalRelationEntity> updates = Sets.newHashSet();
    Set<DataViewAuthVerticalRelationEntity> creates = Sets.newHashSet();
    nebulaToolkitService.collectionDiscrepancy(authRelations, dbAuthRelations, DataViewAuthVerticalRelationEntity::getId, deletes, updates, creates);
    for (DataViewAuthVerticalRelationEntity delete : deletes) {
      dataViewAuthVerticalRelationRepository.delete(delete);
    }
    for (DataViewAuthVerticalRelationEntity update : updates) {
      DataViewAuthVerticalRelationEntity relation = authRelationsMap.get(update.getId());
      update.setAuthRelationIds(relation.getAuthRelationIds());
      update.setSortIndex(relation.getSortIndex());
      update.setRelation(relation.getRelation());
      dataViewAuthVerticalRelationRepository.save(update);
    }
    for (DataViewAuthVerticalRelationEntity create : creates) {
      create.setAuthVertical(authVertical);
      dataViewAuthVerticalRelationRepository.save(create);
    }
    updates.addAll(creates);
    return updates;
  }

  /**
   * 保存数据的验证
   *
   * @param authVertical
   * @param authRelations
   */
  private void saveValidation(DataViewAuthVerticalEntity authVertical, Set<DataViewAuthVerticalRelationEntity> authRelations) {
    Validate.notNull(authVertical, "竖向权限不能为空");
    Validate.notBlank(authVertical.getId(), "竖向权限ID不能为空");
    DataViewAuthVerticalEntity vertical = dataViewAuthVerticalService.findById(authVertical.getId());
    Validate.notNull(vertical, "未找到竖向权限：%s", authVertical.getId());
    if (CollectionUtils.isEmpty(authRelations)) {
      return;
    }
    for (DataViewAuthVerticalRelationEntity authRelation : authRelations) {
      Validate.notNull(authRelation.getAuthRelationIds(), "权限值不能为空");
      Validate.notNull(authRelation.getSortIndex(), "权限值排序不能为空");
      UuidVo relationObj = dataViewAuthRelationService.findRelation(authRelation.getAuthRelationIds(), vertical.getAuthType());
      Validate.notNull(relationObj, "未找到权限值：%s", authRelation.getAuthRelationIds());
      authRelation.setRelation(relationObj);
    }
  }

}
