package com.bizunited.platform.core.service.dataview.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.ChineseCharUtils;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.repository.dataview.DataViewGroupRepository;
import com.bizunited.platform.core.repository.dataview.DataViewRepository;
import com.bizunited.platform.core.repository.dataview.DataViewThirdDataSourceRepository;
import com.bizunited.platform.core.service.dataview.DataSourceService;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.bizunited.platform.core.service.dataview.DataViewGroupService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import static com.bizunited.platform.core.common.constant.Constants.CHILDREN;
import static com.bizunited.platform.core.common.constant.Constants.JSON_OBJ_TYPE_AUTH;
import static com.bizunited.platform.core.common.constant.Constants.JSON_OBJ_TYPE_DATA_VIEW;
import static com.bizunited.platform.core.common.constant.Constants.JSON_OBJ_TYPE_GROUP;
import static com.bizunited.platform.core.common.constant.Constants.TITLE;
import static com.bizunited.platform.core.common.constant.Constants.TYPE;

/**
 * DataViewGroupEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewGroupEntityServiceImpl")
public class DataViewGroupServiceImpl implements DataViewGroupService {

  @Autowired
  private DataViewRepository dataViewRepository;
  @Autowired
  private DataViewGroupRepository dataViewGroupEntityRepository;
  @Autowired
  private DataViewThirdDataSourceRepository dataViewThirdDataSourceRepository;
  @Autowired
  private DataViewService dataViewService;
  @Autowired
  private DataSourceService dataSourceService;
  @Autowired
  private DataViewAuthService dataViewAuthService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 在创建一个新的DataViewGroupEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  @Override
  public void createValidation(DataViewGroupEntity dataViewGroupEntity) {
    Validate.notNull(dataViewGroupEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewGroupEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    dataViewGroupEntity.setId(null);
    Validate.notBlank(dataViewGroupEntity.getGroupName(), "分组名(中文名)不能为空！");
    Validate.notBlank(dataViewGroupEntity.getCode(), "编码不能为空！");
    Validate.notBlank(dataViewGroupEntity.getTargetTable(), "基准数据表不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dataViewGroupEntity.getGroupName() == null || dataViewGroupEntity.getGroupName().length() < 64 , "分组名(中文名),填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewGroupEntity.getTargetTable() == null || dataViewGroupEntity.getTargetTable().length() < 64 , "基准数据表,填入值超过了限定长度(64)，请检查!");
    long count = dataViewGroupEntityRepository.countByCode(dataViewGroupEntity.getCode());
    Validate.isTrue(count == 0L, "重复分组编码：%s", dataViewGroupEntity.getCode());
    //找出同一数据源下所有分组
    List<DataViewGroupEntity> existGroups = dataViewGroupEntityRepository.findDetails();
    Set<DataViewGroupEntity> mainGroups = new HashSet<>();//主数据源组
    Set<DataViewGroupEntity> threeGroups = new HashSet<>();//第三方数据源组
    //分别提取主数据源组和第三方数据源组
    if(!CollectionUtils.isEmpty(existGroups)) {
      existGroups.stream().forEach(e -> {
        if(e.getDataSource() == null) {
          mainGroups.add(e);
        }else {
          threeGroups.add(e);
        }
      });
    }else {
      return;
    }
    //验证基准表再各自分组中，是否重复
    if(dataViewGroupEntity.getDataSource() == null) {
      Validate.isTrue(mainGroups.stream().allMatch(e -> !StringUtils.equals(e.getTargetTable(), dataViewGroupEntity.getTargetTable())), 
        "新增基准表已存在于主数据源分组:%s,请检查", dataViewGroupEntity.getTargetTable());
    }else {
      Validate.isTrue(threeGroups.stream().allMatch(e -> !StringUtils.equals(e.getTargetTable(), dataViewGroupEntity.getTargetTable())), 
        "新增基准表已存在于数据源分组:%s,请检查", dataViewGroupEntity.getTargetTable());
    }
    this.validParent(dataViewGroupEntity);
  }
  @Transactional
  @Override
  public DataViewGroupEntity create(DataViewGroupEntity dataViewGroupEntity, boolean ignoreValidate) {
    // 为了新增字段后不让前端报错，新增时如果没有CODE，默认添加一个
    if(StringUtils.isBlank(dataViewGroupEntity.getCode())){
      dataViewGroupEntity.setCode(UUID.randomUUID().toString());
    }
    if(!ignoreValidate) {
      this.createValidation(dataViewGroupEntity);
    }
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    DataSourceEntity dataSourceEntity = null;
    if(null != dataViewGroupEntity.getDataSource()){
      DataSourceEntity dataSource = dataViewGroupEntity.getDataSource();
      Validate.notNull(dataSource.getId(),"所选数据源ID不能为空");
      dataSourceEntity = dataSourceService.findById(dataSource.getId());
      Validate.notNull(dataSourceEntity,"未在数据库中找到第三方数据源");
      Validate.isTrue(!ChineseCharUtils.hasChinese(dataViewGroupEntity.getTargetTable()),"基准表表名不能含有中文，请检查！");
    }
    //验证该基础数据表是否存在
    //区分是主数据源还是第三方数据源
    if (null == dataSourceEntity) {
      // 主数据源
      dataViewRepository.checkTable(dataViewGroupEntity.getTargetTable());
    }else {
      // 第三方
      dataViewThirdDataSourceRepository.checkTable(dataSourceEntity.getCode(),dataViewGroupEntity.getTargetTable());
    }
    dataViewGroupEntity.setCreateTime(new Date());
    return this.dataViewGroupEntityRepository.save(dataViewGroupEntity);
  }

  @Transactional
  @Override
  public DataViewGroupEntity update(DataViewGroupEntity dataViewGroupEntity) { 
    this.updateValidation(dataViewGroupEntity);
    // ===================基本信息
    String currentId = dataViewGroupEntity.getId();
    DataViewGroupEntity currentDataViewGroupEntity = this.dataViewGroupEntityRepository.findById(currentId).orElse(null);
    Validate.notNull(currentDataViewGroupEntity ,"未发现指定的原始模型对象信");

    // 开始重新赋值，首先是一般对象，updateable = true形式的
    currentDataViewGroupEntity.setGroupName(dataViewGroupEntity.getGroupName());
    currentDataViewGroupEntity.setTargetTable(dataViewGroupEntity.getTargetTable());
    if(dataViewGroupEntity.getParent() != null) {
      currentDataViewGroupEntity.setParent(dataViewGroupEntity.getParent());
    }
    
    this.dataViewGroupEntityRepository.saveAndFlush(currentDataViewGroupEntity);
    // =============
    // ManyToMany 结构的关联关系，需要开发人员自行处理
    // 分组下数据视图的移动与更新，由其他接口负责。
    // =============
    return currentDataViewGroupEntity;
  }
  /**
   * 在更新一个已有的DataViewGroupEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  @Override
  public void updateValidation(DataViewGroupEntity dataViewGroupEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dataViewGroupEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null 且 updateable == true
    Validate.notBlank(dataViewGroupEntity.getGroupName(), "分组名(中文名)不能为空！");
    Validate.notBlank(dataViewGroupEntity.getTargetTable(), "基准数据表不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewGroupEntity.getGroupName() == null || dataViewGroupEntity.getGroupName().length() < 64 , "分组名(中文名),填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewGroupEntity.getTargetTable() == null || dataViewGroupEntity.getTargetTable().length() < 64 , "基准数据表,填入值超过了限定长度(64)，请检查!");
    //因为此处前端传回实体中没有分组信息，需要重新查询
    DataViewGroupEntity dataViewGroup = dataViewGroupEntityRepository.findDetailsById(dataViewGroupEntity.getId());
    //找出同一数据源下所有分组
    List<DataViewGroupEntity> existGroups = dataViewGroupEntityRepository.findDetails();
    if(!CollectionUtils.isEmpty(existGroups)){
      List<DataViewGroupEntity> sameSourceGroups = existGroups.stream().filter(o -> {
        //主数据源
        if(null == dataViewGroup.getDataSource()){
          return null == o.getDataSource();
        }else {
          //第三方数据源
          return dataViewGroup.getDataSource().getId().equals(o.getDataSource().getId());
        }
      }).collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(sameSourceGroups)) {
        sameSourceGroups.stream().forEach(o -> {
              if (o.getId().equals(dataViewGroupEntity.getId())) {
                return;
              }
              //修改组名不能重复
              Validate.isTrue(!o.getGroupName().equals(dataViewGroupEntity.getGroupName()), "同一数据源下，修改数据视图分组名不能重复！请检查");
            }
        );
      }
    }
    this.validParent(dataViewGroupEntity);
  }

  /**
   * 验证父级
   * @param group
   */
  private void validParent(DataViewGroupEntity group) {
    DataViewGroupEntity parent = group.getParent();
    if(parent == null) {
      return;
    }
    Validate.notBlank(parent.getId(), "父级ID不能为空");
    DataViewGroupEntity dbParent = dataViewGroupEntityRepository.findById(parent.getId()).orElse(null);
    Validate.notNull(dbParent, "未找到父级对象");
    if(StringUtils.isNotBlank(group.getId())) {
      Validate.isTrue(!parent.getId().equals(group.getId()), "不能将自己设置成自己的父级");
      this.validCircle(group, parent.getParent());
    }
  }

  /**
   * 检查数据是否形成循环依赖
   * @param group
   * @param parent
   */
  private void validCircle(DataViewGroupEntity group, DataViewGroupEntity parent) {
    if(parent == null) {
      return;
    }
    Validate.isTrue(!group.getId().equals(parent.getId()), "数据形成循环依赖，不能将子级设置成自己的父级，请检查");
    this.validCircle(group, parent.getParent());
  }

  @Override
  public List<DataViewGroupEntity> findByDataSource(String dataSourceId) { 
    if(StringUtils.isBlank(dataSourceId)) { 
      return dataViewGroupEntityRepository.findByDataSource();
    }
    return this.dataViewGroupEntityRepository.findByDataSource(dataSourceId);
  }
  @Override
  public DataViewGroupEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewGroupEntityRepository.findDetailsById(id);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    DataViewGroupEntity group = this.dataViewGroupEntityRepository.findById(id).orElse(null);
    Validate.isTrue(CollectionUtils.isEmpty(group.getChildren()), "该分组存在子分组，不能删除");
    long count = dataViewService.countByDataViewGroupId(group.getId());
    Validate.isTrue(count == 0L, "该分组下存在数据视图，不能删除!!");
    dataViewGroupEntityRepository.delete(group);
  }

  @Override
  public List<DataViewGroupEntity> findByDataSourceCode(String code) {
    List<DataViewGroupEntity> gourpList = null;
    if(StringUtils.isBlank(code)) {
      gourpList = dataViewGroupEntityRepository.findByDataSource();
    }else {
      gourpList = dataViewGroupEntityRepository.findByDataSourceCode(code);
    }
    return gourpList;
  }

  public List<JSONObject> findDetailsByDataSource(String dataSourceId, boolean isAuth) {
    //dataSourceId可以为空，为空代表主数据源
    //根据数据源id查询分组
    List<DataViewGroupEntity> rootGroups;
    if(StringUtils.isBlank(dataSourceId)) {
      rootGroups = dataViewGroupEntityRepository.findByNullDataSourceAndNullParent();
    } else {
      rootGroups = dataViewGroupEntityRepository.findByDataSourceIdAndNullParent(dataSourceId);
    }
    if(CollectionUtils.isEmpty(rootGroups)) {
      return Lists.newArrayList();
    }
    List<JSONObject> result = new ArrayList<>();
    for (DataViewGroupEntity group : rootGroups) {
      JSONObject jsonObject = this.convert2JSONObject(group, isAuth);
      result.add(jsonObject);
    }
    return result;
  }

  /**
   * 转换成JSONObject
   * @param group
   * @return
   */
  private JSONObject convert2JSONObject(DataViewGroupEntity group, boolean isAuth) {
    DataViewGroupEntity cpGroup = nebulaToolkitService.copyObjectByWhiteList(group, DataViewGroupEntity.class, HashSet.class, ArrayList.class);
    JSONObject json = (JSONObject) JSON.toJSON(cpGroup);
    json.put(TITLE, group.getGroupName());
    json.put(TYPE, JSON_OBJ_TYPE_GROUP);
    JSONArray children = new JSONArray();
    // 先加载数据视图
    children.addAll(this.getDataViewJsonArray(group, isAuth));
    // 再加载下级分组
    Set<DataViewGroupEntity> groupChildren = group.getChildren();
    if(!CollectionUtils.isEmpty(groupChildren)) {
      for (DataViewGroupEntity groupChild : groupChildren) {
        JSONObject jsonObject = this.convert2JSONObject(groupChild, isAuth);
        children.add(jsonObject);
      }
    }
    if(!children.isEmpty()) {
      json.put(CHILDREN, children);
    }
    return json;
  }

  /**
   * 获取分组下的数据视图
   * @param group
   * @return
   */
  private JSONArray getDataViewJsonArray(DataViewGroupEntity group, boolean isAuth) {
    List<DataViewEntity> dataViews = dataViewService.findByDataViewGroup(group.getId());
    if(CollectionUtils.isEmpty(dataViews)) {
      return new JSONArray();
    }
    JSONArray jsonArray = new JSONArray();
    for (DataViewEntity dataView : dataViews) {
      DataViewEntity cpDataView = nebulaToolkitService.copyObjectByWhiteList(dataView, DataViewEntity.class, HashSet.class, ArrayList.class);
      JSONObject jsonObject = (JSONObject) JSON.toJSON(cpDataView);
      jsonObject.put(TITLE, dataView.getName());
      jsonObject.put(TYPE, JSON_OBJ_TYPE_DATA_VIEW);
      if(isAuth) {
        this.loadAuthJsonArray(dataView, jsonObject);
      }
      jsonArray.add(jsonObject);
    }
    return jsonArray;
  }

  /**
   * 加载数据权限数据
   * @param dataView
   * @param jsonObject
   * @return
   */
  private void loadAuthJsonArray(DataViewEntity dataView, JSONObject jsonObject) {
    Set<DataViewAuthEntity> auths = dataViewAuthService.findByDataView(dataView.getCode());
    if(CollectionUtils.isEmpty(auths)) {
      return;
    }
    JSONArray jsonArray = new JSONArray();
    for (DataViewAuthEntity auth : auths) {
      DataViewAuthEntity cpAuth = nebulaToolkitService.copyObjectByWhiteList(auth, DataViewAuthEntity.class, HashSet.class, ArrayList.class);
      JSONObject json = (JSONObject) JSON.toJSON(cpAuth);
      json.put(TITLE, auth.getName());
      json.put(TYPE, JSON_OBJ_TYPE_AUTH);
      jsonArray.add(json);
    }
    jsonObject.put(CHILDREN, jsonArray);
  }

  @Override
  public DataViewGroupEntity findByCode(String code) {
    if(StringUtils.isBlank(code)){
      return null;
    }
    return dataViewGroupEntityRepository.findByCode(code);
  }

  @Override
  public DataViewGroupEntity findById(String dataViewGroupId) {
    if(StringUtils.isBlank(dataViewGroupId)) {
      return null;
    }
    return dataViewGroupEntityRepository.findById(dataViewGroupId).orElse(null);
  }
} 
