package com.bizunited.platform.core.service.dataview.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ChineseCharUtils;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewFilterEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.repository.dataview.DataViewRepository;
import com.bizunited.platform.core.repository.dataview.analysis.SQLAuthVerticalAnalysis;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import com.bizunited.platform.core.service.dataview.DataSourceService;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.bizunited.platform.core.service.dataview.DataViewFieldService;
import com.bizunited.platform.core.service.dataview.DataViewFilterService;
import com.bizunited.platform.core.service.dataview.DataViewGroupService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.dataview.DataViewSystemService;
import com.bizunited.platform.core.service.dataview.executor.DataViewExecutor;
import com.bizunited.platform.core.service.dataview.model.ExecuteParamModel;
import com.bizunited.platform.core.service.dataview.model.PageableModel;
import com.bizunited.platform.core.vo.dataview.DataviewExportExcelFieldVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.security.Principal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGE_SIZE;
import static com.bizunited.platform.common.constant.MigrateDataConstants.DATA_VIEW_FILENAME;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * DataViewEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewEntityServiceImpl")
public class DataViewServiceImpl implements DataViewService {



  private static final Logger LOGGER = LoggerFactory.getLogger(DataViewServiceImpl.class);
  private static final String ERROR_MESS_SQL = "原始SQL不能为空！";
  /**
   * 默认时间格式
   */
  private static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";
  /**
   * 默认列宽
   */
  private static final int DEFAULT_COLUMN_WIDTH = 5000;

  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private DataViewRepository dataViewEntityRepository;
  @Autowired
  private DataViewFieldService dataViewFieldService;
  @Autowired
  private DataViewFilterService dataViewFilterService;
  @Autowired
  private DataViewSystemService dataViewSystemService;
  @Autowired
  private DataSourceService dataSourceService;
  @Autowired
  private DataViewAuthService dataViewAuthService;
  @Autowired
  private DataViewGroupService dataViewGroupService;
  @Autowired
  private DynamicDataSourceManager dynamicDataSourceManager;

  /**
   * 在创建一个新的DataViewEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewEntity dataViewEntity) {
    Validate.notNull(dataViewEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewEntity.getId()), "添加信息时，当前信息的数据编号（主键）不能有值！");
    dataViewEntity.setId(null);
    Validate.notBlank(dataViewEntity.getCode(), "数据视图编号不能为空！");
    DataViewEntity currentEntity = dataViewEntityRepository.findByCode(dataViewEntity.getCode());
    Validate.isTrue(null==currentEntity,"该CODE视图已存在，请改变编码！");
    Validate.notBlank(dataViewEntity.getName(), "视图中文名不能为空！");
    Validate.isTrue(!ChineseCharUtils.hasChinese(dataViewEntity.getCode()),"视图编码不能含有中文，请检查！");
    Validate.notBlank(dataViewEntity.getSourceSql(), ERROR_MESS_SQL);
    Validate.notNull(dataViewEntity.getTstatus(), "视图状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(dataViewEntity.getCode() == null || dataViewEntity.getCode().length() < 64 , "数据视图编号,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getName() == null || dataViewEntity.getName().length() < 64 , "视图中文名,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getSourceSql() == null || dataViewEntity.getSourceSql().length() < 4096 , "原始SQL,填入值超过了限定长度(4096)，请检查!");
  }

  @Transactional
  @Override
  public DataViewEntity create(DataViewEntity dataViewEntity , boolean ignoreValidate) {
    if(!ignoreValidate) {
      this.createValidation(dataViewEntity);
    }

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================
    DataSourceEntity dataSource = dataViewEntity.getDataSource();
    if(dataSource != null && StringUtils.isNotBlank(dataSource.getId())) {
      DataSourceEntity currentDataSource = dataSourceService.findById(dataSource.getId());
      Validate.notNull(currentDataSource,"未找到该数据源!");
    }

    DataViewGroupEntity dataViewGroup = dataViewEntity.getDataViewGroup();
    Validate.notNull(dataViewGroup, "视图分组信息不能为空，请检查!!");
    Validate.notBlank(dataViewGroup.getId(), "视图所选分组id不能为空，请检查!!");

    this.dataViewEntityRepository.save(dataViewEntity);

    Set<DataViewFieldEntity> fields = dataViewEntity.getFields();
    this.dataViewFieldService.create(fields, dataViewEntity);

    Set<DataViewSystemEntity> systemFilters = dataViewEntity.getSystemFilters();
    this.dataViewSystemService.create(systemFilters, dataViewEntity);

    Set<DataViewFilterEntity> filters = dataViewEntity.getFilters();
    this.dataViewFilterService.create(filters, fields, dataViewEntity);

    dataViewEntity.setFields(fields);
    dataViewEntity.setFilters(filters);
    dataViewEntity.setSystemFilters(systemFilters);
    return dataViewEntity;
  }

  @Override
  @Transactional
  public DataViewEntity update(DataViewEntity dataView) {
    this.updateValidation(dataView);
    DataViewEntity dbDataView = dataViewEntityRepository.findById(dataView.getId()).orElse(null);
    Validate.notNull(dbDataView, "未找到数据视图对象");
    dbDataView.setCode(dataView.getCode());
    dbDataView.setName(dataView.getName());
    dbDataView.setSourceSql(dataView.getSourceSql());
    dbDataView.setTstatus(dataView.getTstatus());
    dataViewEntityRepository.save(dbDataView);
    Set<DataViewFieldEntity> fields = dataViewFieldService.update(dbDataView, dataView.getFields());
    dbDataView.setFields(fields);
    Set<DataViewFilterEntity> filters = dataViewFilterService.update(dbDataView, dataView.getFilters());
    dbDataView.setFilters(filters);
    Set<DataViewSystemEntity> systemFilters = dataViewSystemService.update(dbDataView, dataView.getSystemFilters());
    dbDataView.setSystemFilters(systemFilters);
    return dbDataView;
  }

  /**
   * 在更新一个已有的DataViewEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  @Override
  public void updateValidation(DataViewEntity dataViewEntity) {
    Validate.isTrue(!StringUtils.isBlank(dataViewEntity.getId()), "修改信息时，当前信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null 且 updateable == true
    Validate.notBlank(dataViewEntity.getCode(), "数据视图编号不能为空！");
    Validate.notBlank(dataViewEntity.getName(), "视图中文名不能为空！");
    Validate.notBlank(dataViewEntity.getSourceSql(), ERROR_MESS_SQL);
    Validate.notNull(dataViewEntity.getTstatus(), "视图状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewEntity.getCode() == null || dataViewEntity.getCode().length() < 64 , "数据视图编号,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getName() == null || dataViewEntity.getName().length() < 64 , "视图中文名,填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewEntity.getSourceSql() == null || dataViewEntity.getSourceSql().length() < 4096 , "原始SQL,填入值超过了限定长度(4096)，请检查!");
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewEntity> op = this.dataViewEntityRepository.findById(id);
    op.ifPresent(dataViewEntity -> this.dataViewEntityRepository.delete(dataViewEntity));
  }


  @Override
  public byte[] exportExcel(ExecuteParamModel executeParam, Principal principal, List<DataviewExportExcelFieldVo> excelFields) {
    if(CollectionUtils.isEmpty(excelFields)) {
      return new byte[0];
    }
    List<JSONObject> list = this.executeQuery(executeParam);
    Workbook workbook = this.buildWorkbook(list, excelFields);
    byte[] bytes = new byte[]{};
    try(ByteArrayOutputStream os = new ByteArrayOutputStream();) {
      workbook.write(os);
      bytes = os.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
    }
    return bytes;
  }

  /**
   * 构建excel
   * @param list
   * @param excelFields
   * @return
   */
  private Workbook buildWorkbook(List<JSONObject> list, List<DataviewExportExcelFieldVo> excelFields) {
    Workbook workbook = new XSSFWorkbook();
    Sheet sheet = workbook.createSheet();
    List<DataviewExportExcelFieldVo> sortedExcelFields = excelFields.stream().sorted(Comparator.comparing(DataviewExportExcelFieldVo::getSortIndex)).collect(Collectors.toList());
    if(CollectionUtils.isEmpty(list)) {
      this.initTitle(sheet, sortedExcelFields);
      return workbook;
    }
    this.initColumnWidth(sheet, list.size());
    this.initTitle(sheet, sortedExcelFields);
    this.initRows(sheet, list, sortedExcelFields);
    return workbook;
  }

  /**
   * 初始化行
   * @param sheet
   * @param list
   * @param fields
   */
  private void initRows(Sheet sheet, List<JSONObject> list, List<DataviewExportExcelFieldVo> fields){
    for (int i = 0; i < list.size(); i++) {
      JSONObject jsonObject = list.get(i);
      Row row = sheet.createRow(i + 1);
      this.initRow(row, jsonObject, fields);
    }
  }

  /**
   * 初始化行
   * @param row
   * @param jsonObject
   * @param fields
   */
  private void initRow(Row row, JSONObject jsonObject, List<DataviewExportExcelFieldVo> fields) {
    for (int i = 0; i < fields.size(); i++) {
      DataviewExportExcelFieldVo field = fields.get(i);
      Cell cell = row.createCell(i);
      this.setCellValue(cell, jsonObject, field);
    }
  }

  /**
   * 获取数据字典显示
   * @param dicts
   * @param obj
   * @return
   */
  private String getCellDictKey(Map<String, String> dicts, Object obj) {
    String[] values = obj.toString().split(",");
    List<String> keys = new ArrayList<>();
    Map<String, String> vDicts = dicts.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (a, b) -> b, () -> new HashMap<>(16)));
    for (String value : values) {
      String key = vDicts.get(value);
      if(key == null) {
        keys.add(value);
      } else {
        keys.add(key);
      }
    }
    return StringUtils.join(keys, "、");
  }

  /**
   * 设置单元格的值
   * @param cell
   * @param jsonObject
   * @param field
   */
  private void setCellValue(Cell cell, JSONObject jsonObject, DataviewExportExcelFieldVo field) {
    Object obj = jsonObject.get(field.getFieldName());
    if(obj == null) {
      cell.setCellValue("");
      return;
    }
    Map<String, String> dicts = field.getDicts();
    if(dicts != null && dicts.size() > 0) {
      String key = this.getCellDictKey(dicts, obj);
      cell.setCellValue(key);
      return;
    }
    if(obj instanceof Integer) {
      cell.setCellValue((Integer)obj);
    } else if (obj instanceof Double) {
      cell.setCellValue((Double)obj);
    } else if (obj instanceof Float) {
      cell.setCellValue((Float)obj);
    } else if (obj instanceof String) {
      cell.setCellValue((String) obj);
    } else if (obj instanceof Date) {
      String format = DEFAULT_DATE_FORMAT;
      if(StringUtils.isNotBlank(field.getFormat())) {
        format = field.getFormat();
      }
      SimpleDateFormat dateFormat = new SimpleDateFormat(format);
      cell.setCellValue(dateFormat.format((Date)obj));
    } else if (obj instanceof Long) {
      Long data = (Long)obj;
      if(Objects.equals("java.sql.Timestamp", field.getFieldType())
        || Objects.equals("java.util.Date", field.getFieldType())) {
        String format = DEFAULT_DATE_FORMAT;
        if(StringUtils.isNotBlank(field.getFormat())) {
          format = field.getFormat();
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat(format);
        cell.setCellValue(dateFormat.format(data));
      } else {
        cell.setCellValue((Long)obj);
      }
    } else if (obj instanceof Boolean) {
      cell.setCellValue((Boolean) obj);
    } else if (obj instanceof BigDecimal) {
      cell.setCellValue(((BigDecimal)obj).toString());
    } else if (obj instanceof Byte) {
      cell.setCellValue((Byte)obj);
    } else if (obj instanceof Short) {
      cell.setCellValue((Short)obj);
    } else {
      cell.setCellValue(obj.toString());
    }
  }

  /**
   * 初始化列宽
   * @param sheet
   */
  private void initColumnWidth(Sheet sheet, int columnCount) {
    for (int i = 0; i < columnCount; i++) {
      sheet.setColumnWidth(i, DEFAULT_COLUMN_WIDTH);
    }
  }

  /**
   * 初始化标题
   * @param sheet
   */
  private void initTitle(Sheet sheet, List<DataviewExportExcelFieldVo> fields) {
    Row row = sheet.createRow(0);
    for (int i = 0; i < fields.size(); i++) {
      Cell cell = row.createCell(i);
      cell.setCellValue(fields.get(i).getDisplayName());
    }
  }

  /**
   * 使用一个指定的第三方数据源，进行数据视图中设定的最原始SQL的预执行。预执行后，将向调用者返回执行后的SQL输出结构、以及要求填写的系统参数信息</br>
   * 注意，该方法只会对原始SQL进行执行（自动替换原始SQL中的系统参数信息），并不会将预执行后的结构做持久化保存；要进行持久化保存，请使用本服务中的creat方法
   * @param dataSourceCode 指定的第三方数据源
   * @param dataView 这个对象包括了数据视图最原始的SQL设定信息。
   */
  @Override
  public DataViewEntity executeResource(String dataSourceCode, DataViewEntity dataView) {
    this.checkSQL(dataView);
    DataViewEntity dataViewEntity = dataViewEntityRepository.executeResource(dataSourceCode, dataView);
    return this.handleDataView(dataViewEntity);
  }

  /**
   * 处理数据视图
   * @param dataView
   * @return
   */
  private DataViewEntity handleDataView(DataViewEntity dataView) {
    if(dataView == null) {
      return null;
    }
    if (dataView.getFilters() == null){
      dataView.setFilters(Sets.newHashSet());
    }
    return dataView;
  }


  /**
   * 使用本本进程的主数据源进行数据视图中设定的最原始SQL的保存。其它操作特性请参见:</br>
   * executeResource(String dataSourceCode , DataViewEntity dataView)方法
   */
  @Override
  public DataViewEntity executeResource(DataViewEntity dataView) {
    this.checkSQL(dataView);
    DataViewEntity dataViewEntity = dataViewEntityRepository.executeResource(dataView);
    return this.handleDataView(dataViewEntity);
  }

  @Override
  public List<JSONObject> executeQuery(ExecuteParamModel executeParam) {
    if(executeParam == null) {
      return Lists.newArrayList();
    }
    DataViewExecutor executor = new DataViewExecutor(applicationContext, executeParam);
    return executor.executeQuery();
  }

  @Override
  public Page<JSONObject> executePageQuery(ExecuteParamModel executeParam) {
    if(executeParam == null) {
      return new PageImpl<>(Lists.newArrayList());
    }
    PageableModel pageable = executeParam.getPageable();
    if(pageable == null) {
      pageable = new PageableModel(0, DEFAULT_PAGE_SIZE);
      executeParam.setPageable(pageable);
    }
    DataViewExecutor executor = new DataViewExecutor(applicationContext, executeParam);
    return executor.executePageQuery();
  }

  /**
   * 检查sql是否合规
   * @param dataView
   */
  private void checkSQL(DataViewEntity dataView){
    String[] notAllowed = new String[]{"delete","insert","drop","create","alter",";"};
    List<String> notAllowedList = Arrays.asList(notAllowed);
    Validate.notNull(dataView,"数据视图不能为空！");
    Validate.notBlank(dataView.getSourceSql(), ERROR_MESS_SQL);
    String sql = dataView.getSourceSql();
    List<String> words = Arrays.asList(StringUtils.split(sql," "));
    if(!CollectionUtils.isEmpty(words)){
      Validate.isTrue("select".equals(StringUtils.lowerCase(words.get(0))),"查询语句只允许以'select'起始的查询语句");
      words.forEach(o -> Validate.isTrue(!notAllowedList.contains(o),String.format("语句中不能含有字符%s",o)));
      //最后一个词不能跟';'
      Validate.isTrue(!words.get(words.size()-1).contains(";"),"最后一个词后不能接分号");
    }
  }

  @Override
  public JSONArray findByDataSourceCode(String dataSourceCode) {
    //1 先查询统计视图相关信息
    List<Object[]> jsonList;
    if(StringUtils.isBlank(dataSourceCode)) {
      jsonList = dataViewEntityRepository.findByDataSourceCode();
    }else {
      jsonList = dataViewEntityRepository.findByDataSourceCode(dataSourceCode);
    }
    JSONArray jsonViewArr = new JSONArray();
    if(!CollectionUtils.isEmpty(jsonList)) {
      jsonList.forEach(e -> {
        JSONObject jsonObject = new JSONObject();
        //这里的下标值，是按照查询语句的列的顺序来的
        jsonObject.put("id", e[0].toString());
        jsonObject.put("code", e[1].toString());
        jsonObject.put("title", e[2].toString());
        jsonObject.put("datasourcegroup", e[3].toString());
        jsonObject.put("outCount", e[4] == null ? 0 : ((BigInteger)e[4]).intValue());
        jsonObject.put("displayCount", e[5] == null ? 0 : ((BigInteger)e[5]).intValue());
        jsonViewArr.add(jsonObject);
      });
    }

    //2 查询所有分组信息
    JSONArray groupArr = new JSONArray();
    List<DataViewGroupEntity> groupList = dataViewGroupService.findByDataSourceCode(dataSourceCode);
    Validate.notNull(groupList, "根据数据源code值未能获取到视图分组信息，请检查!!");
    groupList.forEach(e -> {
      JSONObject jsonObject = new JSONObject();
      JSONArray viewArr = new JSONArray();
      jsonObject.put("id", e.getId());
      jsonObject.put("title", e.getGroupName());
      jsonObject.put("targetTable", e.getTargetTable());
      if(!jsonViewArr.isEmpty()) {
        jsonViewArr.stream().filter(view -> StringUtils.equals(e.getId(), ((JSONObject)view).getString("datasourcegroup"))).forEach(viewArr::add);
      }
      jsonObject.put("children", viewArr);
      groupArr.add(jsonObject);
    });
    return groupArr;
  }

  @Override
  public List<DataViewEntity> findByDataSource(String dataSourceId) {
    if(StringUtils.isBlank(dataSourceId)) {
      return Collections.emptyList();
    }
    return this.dataViewEntityRepository.findByDataSource(dataSourceId);
  }

  @Override
  public List<DataViewEntity> findByDataViewGroup(String dataViewGroupId) {
    if(StringUtils.isBlank(dataViewGroupId)) {
      return Collections.emptyList();
    }
    return this.dataViewEntityRepository.findByDataViewGroup(dataViewGroupId);
  }

  @Override
  public DataViewEntity findByFields(String fieldsId) {
    if(StringUtils.isBlank(fieldsId)) {
      return null;
    }
    return this.dataViewEntityRepository.findByFields(fieldsId);
  }

  @Override
  public DataViewEntity findByFilters(String filtersId) {
    if(StringUtils.isBlank(filtersId)) {
      return null;
    }
    return this.dataViewEntityRepository.findByFilters(filtersId);
  }

  @Override
  public DataViewEntity findBySystemFilters(String systemFiltersId) {
    if(StringUtils.isBlank(systemFiltersId)) {
      return null;
    }
    return this.dataViewEntityRepository.findBySystemFilters(systemFiltersId);
  }

  @Override
  public DataViewEntity findDetailsByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.dataViewEntityRepository.findDetailsByCode(code);
  }

  @Override
  public DataViewEntity findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.dataViewEntityRepository.findByCode(code);
  }

  @Override
  public List<DataViewEntity> findDetailsByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return Lists.newArrayList();
    }
    return this.dataViewEntityRepository.findDetailsByIds(ids);
  }

  @Override
  public DataViewEntity findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }

    Optional<DataViewEntity> op = this.dataViewEntityRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length <= 0) {
      return 0;
    }
    return this.dataViewEntityRepository.countByIds(ids);
  }

  @Override
  @Transactional
  public void moveGroup(String dataViewId, String dataViewGroupId) {
    Validate.notBlank(dataViewId,"所选视图ID不能为空！");
    Validate.notBlank(dataViewGroupId,"所选新分组ID不能为空！");
    DataViewEntity dataViewEntity = dataViewEntityRepository.findById(dataViewId).orElse(null);
    Validate.notNull(dataViewEntity,"未找到所选视图信息，请检查！");
    DataViewGroupEntity group = dataViewGroupService.findById(dataViewGroupId);
    Validate.notNull(group,"未找到所选分组信息，请检查！");
    dataViewEntity.setDataViewGroup(group);
    dataViewEntityRepository.saveAndFlush(dataViewEntity);
  }

  @Override
  public long countByDataViewGroupId(String groupId) {
    return dataViewEntityRepository.countByDataViewGroupId(groupId);
  }

  @Override
  public List<DataViewEntity> findAllByConditions(DataViewEntity dataView) {
    if(dataView == null) {
      return dataViewEntityRepository.findAll();
    }
    Example<DataViewEntity> example = Example.of(dataView);
    return dataViewEntityRepository.findAll(example);
  }

  @SuppressWarnings("unchecked")
  @Override
  @Transactional(REQUIRES_NEW)
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    importModel.appendLine("开始导入数据");
    ZipEntry dataViewEntry = zipFile.getEntry(DATA_VIEW_FILENAME);
    if(dataViewEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
    }
    if(dataViewEntry != null) {
      try (InputStream is = zipFile.getInputStream(dataViewEntry);
           ObjectInputStream ois = new ObjectInputStream(is)) {
        List<DataViewEntity> dataViews = (List<DataViewEntity>) ois.readObject();
        List<DataViewAuthEntity> auths = (List<DataViewAuthEntity>) ois.readObject();
        if(CollectionUtils.isEmpty(dataViews)) {
          importModel.appendLine("导入的数据为空");
        } else {
          this.importData(dataViews, auths, importModel);
        }
      } catch (IOException | ClassNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        importModel.append("读取业务数据失败：").appendLine(e.getMessage());
      }
    }
  }

  @Override
  public Set<String> findFieldsByCodeAndAuthCode(String code, String authCode) {
    if(StringUtils.isBlank(code)) {
      return Sets.newHashSet();
    }
    DataViewEntity dataView = dataViewEntityRepository.findByCode(code);
    if(dataView == null) {
      return Sets.newHashSet();
    }
    Set<DataViewFieldEntity> fields = dataView.getFields();
    if(CollectionUtils.isEmpty(fields)) {
      Lists.newArrayList();
    }
    Set<String> fieldNames = fields.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet());
    if(StringUtils.isBlank(authCode)) {
      return fieldNames;
    }
    DataViewAuthEntity auth = dataViewAuthService.findDetailsByDataViewCodeAndCode(code, authCode);
    if(auth == null || CollectionUtils.isEmpty(auth.getVerticalAuths())) {
      return fieldNames;
    }
    return SQLAuthVerticalAnalysis.getVerticalField(fields, auth.getVerticalAuths());
  }

  /**
   * 处理导入数据
   * @param dataViews
   * @param dataViews
   * @param importModel
   */
  private void importData(List<DataViewEntity> dataViews, List<DataViewAuthEntity> auths, MigrateImportModel importModel) {
    importModel.setTotalCount(dataViews.size());
    Map<String, List<DataViewAuthEntity>> authMap = this.getAuthMapByDataViewKey(auths);
    for (int i = 0; i < dataViews.size(); i++) {
      DataViewEntity dataView = dataViews.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(dataView, importModel, authMap);
    }
  }

  /**
   * 将数据权限根据数据权限分组
   * @param auths
   * @return
   */
  private Map<String, List<DataViewAuthEntity>> getAuthMapByDataViewKey(List<DataViewAuthEntity> auths) {
    Map<String, List<DataViewAuthEntity>> authMap = new HashMap<>();
    if(!CollectionUtils.isEmpty(auths)) {
      for (DataViewAuthEntity auth : auths) {
        String dataViewCode = auth.getDataView().getCode();
        List<DataViewAuthEntity> dataViewAuths = authMap.get(dataViewCode);
        if(dataViewAuths == null) {
          dataViewAuths = new ArrayList<>();
        }
        dataViewAuths.add(auth);
        authMap.put(dataViewCode, dataViewAuths);
      }
    }
    return authMap;
  }

  /**
   * 处理单个数据
   * @param dataView
   * @param importModel
   */
  private void importData(DataViewEntity dataView, MigrateImportModel importModel, Map<String, List<DataViewAuthEntity>> authMap) {
    importModel.append("开始处理数据：").appendLine(dataView.getName());
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    DataViewEntity dbDataView = dataViewEntityRepository.findByCode(dataView.getCode());
    this.handleImportDataId(dataView);
    if(dbDataView != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("数据视图已存在，跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbDataView != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("数据视图已存在，进行更新覆盖");
      this.handleUpdateData(dataView, dbDataView, importModel);
      dataViewAuthService.importAuths(authMap.get(dataView.getCode()), importModel);
      return;
    }
    if(dbDataView != null && ImportExecuteModeEnum.ADD == executeMode) {
      this.handleCreateDataOnExist(dataView, importModel);
      dataViewAuthService.importAuths(authMap.get(dataView.getCode()), importModel);
      return;
    }
    if(dbDataView == null) {
      this.handleCreateData(dataView, importModel);
      dataViewAuthService.importAuths(authMap.get(dataView.getCode()), importModel);
      return;
    }
    importModel.appendLine("暂不支持的执行方式");
    importModel.addSkipCount();
  }

  /**
   * 处理系统已存在的新增数据
   * @param dataView
   * @param importModel
   * @return
   */
  private void handleCreateDataOnExist(DataViewEntity dataView, MigrateImportModel importModel) {
    importModel.appendLine("数据视图已存在，执行已存在新增导入");
    String newCode;
    int count = 1;
    long codeCount;
    do {
      newCode = StringUtils.join(dataView.getCode(), "_", count);
      count++;
      codeCount = dataViewEntityRepository.countByCode(newCode);
    } while (codeCount > 0);
    importModel.append("生成新的数据视图：").appendLine(newCode);
    importModel.appendLine("导入模版数据");
    dataView.setCode(newCode);
    this.handleCreateData(dataView, importModel);
  }

  /**
   * 处理新增数据
   * @param dataView
   * @param importModel
   */
  private void handleCreateData(DataViewEntity dataView, MigrateImportModel importModel) {
    importModel.appendLine("开始新增数据视图");
    DataSourceEntity dataSource = dataView.getDataSource();
    if(dataSource != null) {
      DataSourceEntity dbDataSource = dataSourceService.findByCode(dataSource.getCode());
      if(dbDataSource == null) {
        importModel.appendLine("导入数据视图关联的数据源");
        dbDataSource = dynamicDataSourceManager.create(dataSource);
      }
      dataView.setDataSource(dbDataSource);
    }
    DataViewGroupEntity group = dataView.getDataViewGroup();
    if(group != null) {
      DataViewGroupEntity dbGroup = this.handleDataViewGroup(group, importModel);
      dataView.setDataViewGroup(dbGroup);
    }
    importModel.appendLine("导入数据视图");
    this.create(dataView, false);
    importModel.addCreateCount();
  }

  /**
   * 处理数据视图的分组，并返回持久化的分组信息
   * @param group
   * @param importModel
   * @return
   */
  private DataViewGroupEntity handleDataViewGroup(DataViewGroupEntity group, MigrateImportModel importModel) {
    DataViewGroupEntity dbGroup = dataViewGroupService.findByCode(group.getCode());
    if(dbGroup != null) {
      return dbGroup;
    }
    DataViewGroupEntity parent = group.getParent();
    if(parent != null) {
      DataViewGroupEntity dbParent = this.handleDataViewGroup(parent, importModel);
      group.setParent(dbParent);
    }
    importModel.append("导入数据视图分组：").appendLine(group.getGroupName());
    return dataViewGroupService.create(group, false);
  }

  /**
   * 处理更新的数据
   * @param dataView
   * @param dbDataView
   * @param importModel
   */
  private void handleUpdateData(DataViewEntity dataView, DataViewEntity dbDataView, MigrateImportModel importModel) {
    DataSourceEntity dataSource = dataView.getDataSource();
    DataSourceEntity dbDataSource = dbDataView.getDataSource();
    if((dataSource == null && dbDataSource != null) || (dataSource != null && dbDataSource == null)) {
      throw new IllegalArgumentException("导入的数据视图与现有数据视图分别属于主数据源和第三方数据源，取消导入");
    }
    dataView.setId(dbDataView.getId());
    this.update(dataView);
    importModel.appendLine("更新成功");
    importModel.addUpdateCount();
  }

  /**
   * 处理导入数据的ID
   *
   * @param dataView
   */
  private void handleImportDataId(DataViewEntity dataView) {
    dataView.setId(null);
    DataViewGroupEntity group = dataView.getDataViewGroup();
    if(group != null) {
      do {
        group.setId(null);
      } while ((group = group.getParent()) != null);
    }
    Set<DataViewFilterEntity> filters = dataView.getFilters();
    if(!CollectionUtils.isEmpty(filters)) {
      filters.forEach(f -> f.setId(null));
    }
    Set<DataViewFieldEntity> fields = dataView.getFields();
    if(!CollectionUtils.isEmpty(fields)) {
      fields.forEach(f -> f.setId(null));
    }
    Set<DataViewSystemEntity> systemFilters = dataView.getSystemFilters();
    if(!CollectionUtils.isEmpty(systemFilters)) {
      systemFilters.forEach(f -> f.setId(null));
    }
    DataSourceEntity dataSource = dataView.getDataSource();
    if(dataSource != null) {
      dataSource.setId(null);
    }
  }


}