package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.common.constant.UnBusinessTableConst;
import com.bizunited.platform.core.common.enums.DataSourceTableType;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataSourceTableEntity;
import com.bizunited.platform.core.repository.DataSourceTableRepository;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import com.bizunited.platform.core.service.DataSourceTableService;
import com.bizunited.platform.core.service.dataview.DataSourceService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * @Author: Paul Chan
 * @Date: 2019-04-15 11:07
 * @Description: 数据表业务模型的服务层接口实现
 */
@Service("DataSourceTableServiceImpl")
public class DataSourceTableServiceImpl implements DataSourceTableService {
  @Autowired
  private DataSourceTableRepository dataSourceTableRepository;
  @Autowired
  private DynamicDataSourceManager dynamicDataSourceManager;
  @Autowired
  private DataSourceService dataSourceService;

  @Override
  public Page<DataSourceTableEntity> findByConditions(Pageable pageable, Integer tableType, String dataSourceId, Integer status, String tableName) {
    return this.dataSourceTableRepository.findByConditions(pageable, tableType, dataSourceId, status, tableName);
  }

  @Override
  public Set<String> findByDataSource(DataSourceEntity dataSource) {
    if(dataSource == null){
      return dataSourceTableRepository.queryMainDataSourceTables();
    }
    SessionFactory sessionFactory = dynamicDataSourceManager.getCurrentSessionFactory(dataSource.getCode());
    if(sessionFactory == null){
      throw new NullPointerException("query DataSourceTables is null");
    }
    return dataSourceTableRepository.queryDataSourceTables(sessionFactory);
  }

  @Override
  @Transactional
  public void refresh() {
    initDataSourceTableWithDataSource(null);
    List<DataSourceEntity> dataSources = this.dataSourceService.findByStatus(1);
    if(CollectionUtils.isEmpty(dataSources)){
      return;
    }
    for (DataSourceEntity dataSource : dataSources) {
      this.initDataSourceTableWithDataSource(dataSource);
    }
  }

  /**
   * 根据数据源初始化数据表数据
   * @param dataSource
   */
  private void initDataSourceTableWithDataSource(DataSourceEntity dataSource){
    List<DataSourceTableEntity> oldTables;
    if(dataSource != null){
      oldTables = dataSourceTableRepository.findByDataSource(dataSource.getId());
    } else {
      oldTables = dataSourceTableRepository.findMainDataSourceTables();
    }
    Set<String> oldTableSet = new HashSet<>(128);
    Set<String> tables = this.findByDataSource(dataSource);
    Validate.isTrue(!CollectionUtils.isEmpty(tables) , dataSource == null ? "获取主数据源的表数据失败" : StringUtils.join("获取数据源[", dataSource.getCode(),"]的表数据失败"));
    
    // 通过以下代码我们可以知道相对于已存在的数据表oldTables，哪些数据表被新增了，哪些被删除了，哪些被修改了
    Map<String, DataSourceTableEntity> oldTablesMap = new HashMap<>(128);
    oldTables.forEach(oldTable -> {
      oldTableSet.add(oldTable.getTableName());
      oldTablesMap.put(oldTable.getTableName(), oldTable);
    });
    Set<String> newTables = Sets.difference(tables, oldTableSet);
    Set<String> deletedTables = Sets.difference(oldTableSet, tables);
    Set<String> isTables = Sets.intersection(oldTableSet, tables);
    //保存新增的表
    for (String tableName : newTables) {
      DataSourceTableEntity entity = this.initDataSourceTableEntity(dataSource, tableName);
      dataSourceTableRepository.saveAndFlush(entity);
    }
    //禁用已删除的表
    for (String tableName : deletedTables) {
      DataSourceTableEntity dataSourceTable = oldTablesMap.get(tableName);
      dataSourceTable.settStatus(0);
      dataSourceTableRepository.saveAndFlush(dataSourceTable);
    }
    // 更新旧数据
    for (String table : isTables) {
      DataSourceTableEntity dataSourceTable = oldTablesMap.get(table);
      boolean update = false;
      if(dataSourceTable.gettStatus() == 0){
        dataSourceTable.settStatus(1);
        update = true;
      }
      Integer tableType = dataSourceTable.getTableType();
      if(dataSourceTable.getDataSource() == null && UnBusinessTableConst.isUnBusinessTable(dataSourceTable.getTableName())){
        if(DataSourceTableType.BUSINESS_TABLE.getType().equals(tableType)){
          dataSourceTable.setTableType(DataSourceTableType.UN_BUSINESS_TABLE.getType());
          if(StringUtils.isBlank(dataSourceTable.getTableDesc())){
            dataSourceTable.setTableDesc("只读的数据表");
          }
          update = true;
        }
      }else if(DataSourceTableType.UN_BUSINESS_TABLE.getType().equals(tableType)){
        dataSourceTable.setTableType(DataSourceTableType.BUSINESS_TABLE.getType());
        update = true;
      }
      if(update){
        dataSourceTable.setModifyTime(new Date());
        dataSourceTableRepository.saveAndFlush(dataSourceTable);
      }
    }
  }

  /**
   * 检验并初始化各种entity的状态属性——是业务表还是非业务表
   * @param dataSource
   * @param tableName
   * @return
   */
  private DataSourceTableEntity initDataSourceTableEntity(DataSourceEntity dataSource, String tableName){
    DataSourceTableEntity entity = new DataSourceTableEntity();
    Date now = new Date();
    entity.setCreateTime(now);
    entity.setModifyTime(now);
    entity.setTableName(tableName);
    entity.settStatus(1);
    if(dataSource == null && UnBusinessTableConst.isUnBusinessTable(tableName)){
      entity.setTableType(DataSourceTableType.UN_BUSINESS_TABLE.getType());
      entity.setTableDesc("只读的数据表");
    }else{
      entity.setTableType(DataSourceTableType.BUSINESS_TABLE.getType());
    }
    entity.setDataSource(dataSource);
    return entity;
  }

  @Override
  @Transactional
  public DataSourceTableEntity update(DataSourceTableEntity dataSourceTableEntity) {
    String id = dataSourceTableEntity.getId();
    String tableDesc = dataSourceTableEntity.getTableDesc();
    String dataSourceTableId = null;
    if(dataSourceTableEntity.getDataSource() != null){
      dataSourceTableId = dataSourceTableEntity.getDataSource().getId();
    }
    Validate.notEmpty(dataSourceTableEntity.getId(), "主键ID不能为空");
    Validate.isTrue(StringUtils.isNotBlank(tableDesc) || StringUtils.isNotBlank(dataSourceTableId), "没有需要更新的数据");
    Optional<DataSourceTableEntity> op = this.dataSourceTableRepository.findById(id);
    DataSourceTableEntity dataSourceTable = op.orElse(null);
    Validate.notNull(dataSourceTable, "更新的对象不存在");

    if(StringUtils.isNotBlank(tableDesc)){
      dataSourceTable.setTableDesc(tableDesc);
    }
    if(StringUtils.isNotBlank(dataSourceTableId)){
      DataSourceEntity dataSource = dataSourceService.findById(dataSourceTableId);
      Validate.notNull(dataSource, "数据源对象不存在");
      dataSourceTable.setDataSource(dataSource);
    }
    dataSourceTable.setModifyTime(new Date());
    this.dataSourceTableRepository.save(dataSourceTable);
    return dataSourceTable;
  }
  
  @Override
  @Transactional
  public void updateTableDesc(DataSourceEntity dataSource, String tableName, String tableDesc) {
    Validate.notNull(tableName, "表名不能为空");
    Validate.notNull(tableDesc, "表说明不能为空");
    DataSourceTableEntity dataSourceTable;
    if(dataSource == null){
      dataSourceTable = dataSourceTableRepository.findMainDataSourceTableByTableName(tableName);
    } else {
      dataSourceTable = dataSourceTableRepository.findByTableNameAndDataSource(tableName, dataSource.getId());
    }
    Validate.notNull(dataSourceTable, "数据表对象不存在");
    dataSourceTable.setTableDesc(tableDesc);
    dataSourceTableRepository.save(dataSourceTable);
  }
}
