package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ChineseCharUtils;
import com.bizunited.platform.core.entity.EnvironmentVariableEntity;
import com.bizunited.platform.core.repository.EnvironmentVariableRepository;
import com.bizunited.platform.core.service.EnvironmentVariableService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.Constants.LINE_SEPARATOR;
import static com.bizunited.platform.common.constant.MigrateDataConstants.ENV_FILENAME;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * EnvironmentVariableServiceImpl 参数
 *
 * @description:
 * @author: yanwe
 * @date: 15/May/2019 14:34
 */
@Service("EnvironmentVariableServiceImpl")
public class EnvironmentVariableServiceImpl implements EnvironmentVariableService {

  private static final Logger LOGGER = LoggerFactory.getLogger(EnvironmentVariableServiceImpl.class);
  @Autowired private EnvironmentVariableRepository environmentVariableRepository;

  @Override
  public List<EnvironmentVariableEntity> findAll() {
    return environmentVariableRepository.findAll();
  }

  @Override
  public Page<EnvironmentVariableEntity> findByConditions(String paramCode, String paramKey, String paramValue, String paramType, Integer paramStatus, Pageable pageable) {
    Map<String,Object> conditions = new HashMap<>();
    if(StringUtils.isNotEmpty(paramCode)){
      conditions.put("paramCode",paramCode);
    }
    if(StringUtils.isNotEmpty(paramKey)){
      conditions.put("paramKey",paramKey);
    }
    if(StringUtils.isNotEmpty(paramValue)){
      conditions.put("paramValue",paramValue);
    }
    if(StringUtils.isNotEmpty(paramType)){
      conditions.put("paramType",paramType);
    }
    if(null != paramStatus){
      conditions.put("paramStatus",paramStatus);
    }
    if(pageable ==null){
      pageable = PageRequest.of(0,50);
    }
    return environmentVariableRepository.queryPage(pageable,conditions);
  }

  @Override
  public EnvironmentVariableEntity findByKey(String key) {
    if(StringUtils.isBlank(key)) {
      return null;
    }
    return environmentVariableRepository.findByParamKey(key);
  }

  @Override
  public EnvironmentVariableEntity findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return environmentVariableRepository.findByParamCode(code);
  }

  @Override
  public Set<EnvironmentVariableEntity> findDetailsByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return Sets.newHashSet();
    }
    return this.environmentVariableRepository.findDetailsByIds(ids);
  }

  @Override
  @Transactional
  public EnvironmentVariableEntity enable(String id) {
    Validate.notBlank(id,"启用时必须传入ID");
    Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(id);
    EnvironmentVariableEntity entity = op.orElse(null);
    Validate.notNull(entity,"未找到该参数！");

    entity.setParamStatus(1);
    return environmentVariableRepository.save(entity);
  }

  @Override
  @Transactional
  public EnvironmentVariableEntity disable(String id) {
    Validate.notBlank(id,"禁用时必须传入ID");
    Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(id);
    EnvironmentVariableEntity entity = op.orElse(null);
    Validate.notNull(entity,"未找到该参数！");

    entity.setParamStatus(0);
    return environmentVariableRepository.save(entity);
  }

  @Override
  @Transactional
  public EnvironmentVariableEntity save(EnvironmentVariableEntity entity) {
    Validate.notNull(entity, "参数不能为空！");
    Validate.notBlank(entity.getParamKey(), "参数的键不能为空！");
    Validate.notBlank(entity.getParamValue(),"参数的值不能为Null！");
    Validate.notBlank(entity.getParamCode(),"参数的编码不能为Null！");
    Validate.isTrue(!ChineseCharUtils.hasChinese(entity.getParamKey()),"参数键不能含有中文！");
    Validate.notBlank(entity.getParamType(),"参数类型不能为空！");
    Validate.notNull(entity.getParamStatus(),"参数状态不能为空！");
    //获取当前登录账户
    UserVo user = SecurityUtils.getCurrentUser();
    if (StringUtils.isEmpty(entity.getId())) {
      // 新增
      long existByKey = environmentVariableRepository.countByParamKey(entity.getParamKey());
      long existByCode = environmentVariableRepository.countByParamCode(entity.getParamCode());
      Validate.isTrue(existByCode < 1, "保存参数的参数名称已有重复，请检查！");
      Validate.isTrue(existByKey < 1, "保存参数的键已有重复，请检查！");
      entity.setCreateDate(new Date());
      entity.setModifyDate(new Date());
      entity.setCreateUser(user.getAccount());
      entity.setModifyUser(user.getAccount());
      return environmentVariableRepository.save(entity);
    } else {
      // 修改
      Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(entity.getId());
      EnvironmentVariableEntity exist = op.orElse(null);
      Validate.notNull(exist,"根据ID未获取到该参数！");

      //验证修改后的名称，是否有重复
      EnvironmentVariableEntity existCode = environmentVariableRepository.findByParamCode(entity.getParamCode());
      Boolean notSameCode = null == existCode || exist.getParamCode().equals(existCode.getParamCode());
      Validate.isTrue(notSameCode,"修改后的参数编码已有重复，请检查！");
      //验证修改后的键，是否有重复
      EnvironmentVariableEntity existKey = environmentVariableRepository.findByParamKey(entity.getParamKey());
      Boolean notSameKey = null == existKey || exist.getParamKey().equals(existKey.getParamKey());
      Validate.isTrue(notSameKey,"修改后的参数键已有重复，请检查！");
      // 其余的属性变更
      exist.setParamCode(entity.getParamCode());
      exist.setParamKey(entity.getParamKey());
      exist.setParamValue(entity.getParamValue());
      exist.setParamType(entity.getParamType());
      exist.setParamDesc(entity.getParamDesc());
      exist.setModifyUser(user.getAccount());
      exist.setModifyDate(new Date());
      return environmentVariableRepository.save(exist);
    }
  }

  @Override
  @Transactional(rollbackOn = Exception.class)
  public void delete(String id) {
    Validate.notBlank(id, "删除参数ID不能为空！");
    Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(id);
    EnvironmentVariableEntity exist = op.orElse(null);
    Validate.notNull(exist,"未找到参数，请检查！");
    environmentVariableRepository.delete(exist);
  }

  @Override
  public long countByIds(String[] ids) {
    if(ids == null || ids.length <= 0) {
      return 0;
    }
    return this.environmentVariableRepository.countByIds(ids);
  }

  /**
   * 导入全局变量
   * @param importModel
   */
  @Override
  @Transactional(REQUIRES_NEW)
  @SuppressWarnings("unchecked")
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    StringBuilder executeLog = new StringBuilder();
    importModel.appendLine("开始导入数据");
    ZipEntry evnEntry = zipFile.getEntry(ENV_FILENAME);
    if(evnEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
    }
    if(evnEntry != null) {
      try (InputStream is = zipFile.getInputStream(evnEntry)) {
        ObjectInputStream ois = new ObjectInputStream(is);
        Set<EnvironmentVariableEntity> environmentVariablesSet = (Set<EnvironmentVariableEntity>) ois.readObject();
        List<EnvironmentVariableEntity> environmentVariables = Lists.newArrayList(environmentVariablesSet);
        if(CollectionUtils.isEmpty(environmentVariables)) {
          importModel.appendLine("导入的数据为空");
        } else {
          this.importData(environmentVariables, importModel);
        }
      } catch (IOException | ClassNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        executeLog.append("读取业务数据失败：").append(e.getMessage()).append(LINE_SEPARATOR);
      }
    }
  }

  /**
   * 不分页查询
   * @param paramCode
   * @param paramKey
   * @param paramType
   * @param paramStatus
   * @return
   */
  @Override
  public List<EnvironmentVariableEntity> findAllByConditions(String paramCode, String paramKey, String paramType, Integer paramStatus) {
    Map<String,Object> conditions = new HashMap<>();
    if(StringUtils.isNotEmpty(paramCode)){
      conditions.put("paramCode",paramCode);
    }
    if(StringUtils.isNotEmpty(paramKey)){
      conditions.put("paramKey",paramKey);
    }
    if(StringUtils.isNotEmpty(paramType)){
      conditions.put("paramType",paramType);
    }
    if(null != paramStatus){
      conditions.put("paramStatus",paramStatus);
    }
    return environmentVariableRepository.findAllByConditions(conditions);
  }

  /**
   * 处理导入数据
   * @param environmentVariables
   * @param importModel
   */
  private void importData(List<EnvironmentVariableEntity> environmentVariables, MigrateImportModel importModel) {
    importModel.setTotalCount(environmentVariables.size());
    for (int i = 0; i < environmentVariables.size(); i++) {
      EnvironmentVariableEntity environmentVariable = environmentVariables.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(environmentVariable, importModel);
    }
  }

  /**
   * 处理单个数据
   * @param environmentVariable
   * @param importModel
   */
  private void importData(EnvironmentVariableEntity environmentVariable, MigrateImportModel importModel) {
    importModel.append("开始处理数据：").appendLine(environmentVariable.getParamCode());
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    EnvironmentVariableEntity dbEnvironmentVariable = environmentVariableRepository.findByParamCode(environmentVariable.getParamCode());
    environmentVariable.setId(null);
    if(dbEnvironmentVariable != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("全局参数已存在，跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbEnvironmentVariable != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("全局参数已存在，进行更新覆盖");
      this.handleUpdateData(environmentVariable, dbEnvironmentVariable, importModel);
      return;
    }
    if(dbEnvironmentVariable == null) {
      this.handleCreateData(environmentVariable, importModel);
    }
  }

  /**
   * 新增全局变量
   * @param environmentVariable
   * @param importModel
   */
  private void handleCreateData(EnvironmentVariableEntity environmentVariable, MigrateImportModel importModel) {
    this.save(environmentVariable);
    importModel.appendLine("导入全局变量");
    importModel.addCreateCount();
  }

  /**
   * 更新全局变量
   * @param environmentVariable
   * @param dbEnvironmentVariable
   * @param importModel
   */
  private void handleUpdateData(EnvironmentVariableEntity environmentVariable, EnvironmentVariableEntity dbEnvironmentVariable, MigrateImportModel importModel) {
    dbEnvironmentVariable.setParamDesc(environmentVariable.getParamDesc());
    dbEnvironmentVariable.setParamKey(environmentVariable.getParamKey());
    dbEnvironmentVariable.setParamType(environmentVariable.getParamType());
    dbEnvironmentVariable.setParamValue(environmentVariable.getParamValue());
    dbEnvironmentVariable.setParamStatus(environmentVariable.getParamStatus());
    this.save(dbEnvironmentVariable);
    importModel.appendLine("更新成功");
    importModel.addUpdateCount();
  }
}
