package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.repository.script.ScriptRepository;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.script.persistent.PeristentGroovyClassService;
import com.bizunited.platform.core.service.script.persistent.SimplePersistentGroovyServiceFactory;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import groovy.lang.Binding;
import groovy.lang.GroovyClassLoader;
import groovy.lang.GroovyShell;
import groovy.lang.Script;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.codehaus.groovy.control.CompilationFailedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;

import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;

/**
 * groovy动态脚本模块的服务实现在这里
 * @description:
 * @author: yanwe yinwnejie
 * @date: 27/May/2019 15:50
 */
@Service("ScriptServiceImpl")
public class ScriptServiceImpl implements ScriptService {

  private static final Logger LOGGER = LoggerFactory.getLogger(ScriptServiceImpl.class);
  @Autowired 
  private ApplicationContext applicationContext;
  @Autowired 
  private VenusFileService venusFileService;
  @Autowired 
  private ScriptRepository scriptRepository;
  @Autowired(required = false)
  private RedisMutexService redisMutexService;
  @Autowired 
  private RoleService roleService;
  @Autowired 
  private PositionService positionService;
  @Autowired 
  private OrganizationService organizationService;
  @Autowired 
  private UserService userService;
  @Autowired 
  private SimplePersistentGroovyServiceFactory simplePersistentGroovyServiceFactory;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 针对当前script的hashcode分离的不同可重入锁
   */
  private static Map<Integer, ReentrantLock> scriptExecuteLocks = new ConcurrentHashMap<>();
  
  @Override
  public Page<ScriptEntity> findByConditions(Pageable pageable, String name, String language) {
    Validate.notNull(pageable, "分页信息不能为空");
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (StringUtils.isNotBlank(language)) {
      conditions.put("language", language);
    }
    conditions.put(PROJECT_NAME, platformContext.getAppName());
    return scriptRepository.queryPage(pageable, conditions);
  }

  @Override
  public ScriptEntity findById(String scriptId) {
    if (StringUtils.isBlank(scriptId)) {
      return null;
    }
    Optional<ScriptEntity> scriptEntity = scriptRepository.findById(scriptId);
    return scriptEntity.isPresent() ? scriptEntity.get() : null;
  }

  @Override
  public ScriptEntity findByName(String name) {
    if (StringUtils.isBlank(name)) {
      return null;
    }
    return scriptRepository.findByName(name);
  }

  @Override
  public String findContentByName(String scriptName) {
    if (StringUtils.isBlank(scriptName)) {
      return null;
    }
    ScriptEntity currentScript = this.findByName(scriptName);
    if(currentScript == null) {
      return null;
    }

    return this.findContentById(currentScript.getId());
  }

  @Override
  public String findContentById(String scriptId) {
    if (StringUtils.isBlank(scriptId)) {
      return null;
    }
    Optional<ScriptEntity> scriptEntity = scriptRepository.findById(scriptId);
    if (!scriptEntity.isPresent()) {
      return null;
    }
    byte[] content = venusFileService.readFileContent(scriptEntity.get().getFileCode(), scriptEntity.get().getFileName());
    return new String(content, StandardCharsets.UTF_8);
  }

  @Override
  @Transactional
  public ScriptEntity update(ScriptEntity scriptEntity, String scriptContent) {

    Validate.notNull(scriptEntity, "脚本实体不能为空！");
    Validate.notBlank(scriptContent, "脚本内容不能为空！");
    Validate.notNull(scriptEntity.getLanguage(), "脚本语言不能为空");
    Validate.notBlank(scriptEntity.getName(), "脚本名称不能为空！");
    Validate.isTrue(scriptEntity.getName().length() <= 255, "脚本名称不能超过255个字符！");
    Validate.notBlank(scriptEntity.getId(), "脚本ID不能为空");
    Optional<ScriptEntity> op = scriptRepository.findById(scriptEntity.getId());
    ScriptEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity, "查询脚本不能为空");

    // 获取账号信息
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext, "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication, "未发现任何用户登录信息!!");
    UserVo modifyUser = userService.findByAccount(authentication.getName());
    Validate.notNull(modifyUser, "未找到当前用户信息");

    // 保存脚本到文件路径
    String[] fileinfo = this.saveScriptContent(scriptContent);
    // 保存文件路径
    existEntity.setFileName(fileinfo[0]);
    existEntity.setFileCode(fileinfo[1]);
    // 最后修改时间
    existEntity.setModifyTime(new Date());
    // 最后修改人
    existEntity.setModifyAccount(modifyUser.getAccount());
    // 脚本语言
    existEntity.setLanguage(scriptEntity.getLanguage());
    // 脚本名称
    existEntity.setName(scriptEntity.getName());
    //描述信息
    existEntity.setDescription(scriptEntity.getDescription());

    try{
      return scriptRepository.save(existEntity);
    } finally {
      //更新脚本内容后，需要移除之前脚本在缓冲中数据
      PeristentGroovyClassService peristentGroovyClassService = simplePersistentGroovyServiceFactory.createPeristentGroovyClassService();
      peristentGroovyClassService.delete(existEntity.getId());
    }

  }
  
  /**
   * 重复的代码，进行私有方法封装，返回的一个数组，第一个值是文件的重命名名称、第二个值是文件相对路径
   * @param scriptContent
   * @return
   */
  private String[] saveScriptContent(String scriptContent) {
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String uuid = UUID.randomUUID().toString();
    String fileRename = uuid + ".txt";
    String relativePath = StringUtils.join("/groovyScript/", folderName, "/", (new Random().nextInt(100) % 10));
    byte[] scriptContentByte = scriptContent.getBytes(StandardCharsets.UTF_8);
    venusFileService.saveFile(relativePath, fileRename, fileRename, scriptContentByte);
    
    return new String[] {fileRename , relativePath};
  }

  @Override
  @Transactional
  public ScriptEntity create(ScriptEntity scriptEntity, String scriptContent) {
    Validate.notNull(scriptEntity, "脚本实体不能为空！");
    Validate.notBlank(scriptContent, "脚本内容不能为空！");
    Validate.notNull(scriptEntity.getLanguage(), "脚本语言不能为空");
    Validate.notBlank(scriptEntity.getName(), "脚本名称不能为空！");
    ScriptEntity existEntity = scriptRepository.findByName(scriptEntity.getName());
    Validate.isTrue(null == existEntity, "脚本名称重复，请检查！");
    // 获取账号信息
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext, "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication, "未发现任何用户登录信息!!");
    UserVo creator = userService.findByAccount(authentication.getName());
    Validate.notNull(creator, "未找到当前用户信息");
    // 保存脚本到文件路径
    String[] fileinfo = this.saveScriptContent(scriptContent);
    // 保存文件路径
    scriptEntity.setFileName(fileinfo[0]);
    scriptEntity.setFileCode(fileinfo[1]);
    // 创建者
    scriptEntity.setCreateAccount(creator.getAccount());
    scriptEntity.setCreateTime(new Date());
    // 更新时间
    scriptEntity.setModifyTime(new Date());
    scriptEntity.setModifyAccount(creator.getAccount());
    scriptEntity.setProjectName(platformContext.getAppName());
    return scriptRepository.save(scriptEntity);
  }

  @Override
  @NebulaServiceMethod(name = "scriptService.invoke" , desc = "执行一个或者多个groovy脚本执行器的调用")
  public Map<String, Object> invoke(@ServiceMethodParam(name = "scriptIds") String scriptIds[], Map<String, Object> params) throws InvokeProxyException {
    Validate.notNull(scriptIds , "至少需要传入一个脚本内容编号");
    Validate.isTrue(scriptIds.length >= 0 , "至少需要传入一个脚本内容编号");
    Map<String, Object> currentParams = params;
    if(currentParams == null) {
      currentParams = new HashMap<>();
    }
    
    /*
     * 1、验证所有scriptId对应脚本的存在性和正确性
     * 2、正确性全部验证成功后，再开始性质
     * */
    // 验证脚本
    List<ScriptEntity> scriptEntities = new ArrayList<>();
    for (String scriptId : scriptIds) {
      Validate.notBlank(scriptId, "脚本ID不能为空！");
      Optional<ScriptEntity> op = scriptRepository.findById(scriptId);
      ScriptEntity scriptEntity = op.orElse(null);
      Validate.notNull(scriptEntity, "[%s]对应的脚本不能为空" , scriptId);
      scriptEntities.add(scriptEntity);
    }
    
    // 开始执行
    for (ScriptEntity scriptItem : scriptEntities) {
      Map<String, Object> result = this.invokeOne(scriptItem, currentParams);
      // 执行当前脚本完毕后，再将结果写入下一个脚本的入参
      if (!CollectionUtils.isEmpty(result)) {
        currentParams.putAll(result);
      }
    }
    return currentParams;
  }
  
  /**
   * 单个的脚本调用在这里完成
   * @param scriptId
   * @param params
   * @return
   * @throws InvokeProxyException
   */
  @SuppressWarnings("unchecked")
  private Map<String, Object> invokeOne(ScriptEntity scriptItem, Map<String, Object> params) throws InvokeProxyException {
    /*
     * 1、取得已准备号的scriptShell。首先从cache中取得（使用的是ecache每10秒刷新一次），如果没有再从磁盘上取得。
     * 从磁盘取得的脚本信息，将按照不同的groovyClassloader进行编译
     * 2、以下信息需要作为script的全局变量进行加载：
     *    a、有params传入的各种外部参数
     *    b、spring的上下文，applicationContext，变量名ctx
     *    c、经常会使用的service、例如userService。
     * 3、开始执行，并输出script中的所有全局变量——以Map的方式输出。 
     * 系统会按照script的id的hashcode为依据，为每一次运行加锁，以避免groovy脚本对象共享引起的线程安全性问题
     * 可指定一个变量，例如returnKey，作为整个调用过程的返回值。
     */
    // 1、===
    // 尝试从内存中获取，如果没有再从磁盘读取
    PeristentGroovyClassService peristentGroovyClassService = simplePersistentGroovyServiceFactory.createPeristentGroovyClassService();
    Script groovyScript = (Script) peristentGroovyClassService.findByClassName(scriptItem.getId());
    Integer idHashcode = scriptItem.getId().hashCode();
    // 如果没有，则从磁盘中获取
    if (groovyScript == null) {
      synchronized (ScriptService.class) {
        groovyScript = (Script) peristentGroovyClassService.findByClassName(scriptItem.getId());
        if(groovyScript == null) {
          byte[] scriptContent = venusFileService.readFileContent(scriptItem.getFileCode(), scriptItem.getFileName());
          String groovyStr = new String(scriptContent);
          LOGGER.info("正在进行groovy脚本编译，脚本id = " + idHashcode);
          GroovyClassLoader groovyClassLoader = new GroovyClassLoader(this.applicationContext.getClassLoader());
          GroovyShell shell = new GroovyShell(groovyClassLoader);
          try {
            groovyScript = shell.parse(groovyStr);
            peristentGroovyClassService.save(scriptItem.getId(), groovyScript);
            if(scriptExecuteLocks.get(idHashcode) == null) {
              scriptExecuteLocks.put(idHashcode, new ReentrantLock());
            }
          } catch (CompilationFailedException e) {
            LOGGER.error(e.getMessage(), e);
            throw new IllegalArgumentException("编译groovy脚本时发生错误，请检查！");
          }
        }
      }
    }
    // 2、===
    Map<String, Object> result = new HashMap<>();
    ReentrantLock reentrantLock = null;
    try {
      reentrantLock = scriptExecuteLocks.get(idHashcode);
      reentrantLock.lock();
      Binding binding = bindingScriptVars(params);
      groovyScript.setBinding(binding);
      groovyScript.run();
      // 3、===
      // 将其他参数写入输出
      Map<String,Object> bindingParams= binding.getVariables();
      if(!CollectionUtils.isEmpty(bindingParams)){
        for(String key: bindingParams.keySet()){
          result.put(key,binding.getVariable(key));
        }
      }
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }finally {
      if(reentrantLock != null) {
        reentrantLock.unlock();
      }
    }
    return result;
  }

  /**
   * 在执行脚本前，为脚本内容绑定入参值
   * @param params
   * @return
   */
  private Binding bindingScriptVars(Map<String, Object> params) {
    Binding binding = new Binding();
    // 绑定添加默认方法参数
    binding.setVariable("userService", userService);
    binding.setVariable("roleService", roleService);
    binding.setVariable("positionService", positionService);
    binding.setVariable("organizationService", organizationService);
    if(redisMutexService != null) {
      binding.setVariable("redisMutexService", redisMutexService);
    }
    binding.setVariable("ctx", applicationContext);
    // 绑定传入参数
    if (!CollectionUtils.isEmpty(params)) {
      for (Map.Entry<String,Object> item : params.entrySet()) {
        binding.setVariable(item.getKey(), item.getValue());
      }
    }
    return binding;
  }
  
  @SuppressWarnings("unchecked")
  @Override
  public Object invoke(String scriptContent, Map<String, Object> params) {
    // 执行过程和另一个invoke类似，请参考
    // 唯一不同的是，这个的缓存key，将以scriptContent的base64编码作为key进行保存
    Validate.notBlank(scriptContent , "必须要有需要执行的脚本内容");
    Map<String, Object> currentParams = params;
    if(currentParams == null) {
      currentParams = new HashMap<>();
    }
    
    byte[] contxts = scriptContent.getBytes(Charset.forName("UTF-8"));
    String conextBase64 = Base64.getEncoder().encodeToString(contxts);
    Integer idHashcode = conextBase64.hashCode();
    PeristentGroovyClassService peristentGroovyClassService = simplePersistentGroovyServiceFactory.createPeristentGroovyClassService();
    Script groovyScript = (Script) peristentGroovyClassService.findByClassName(conextBase64);
    if(groovyScript == null) {
      synchronized (ScriptService.class) {
        groovyScript = (Script) peristentGroovyClassService.findByClassName(conextBase64);
        if(groovyScript == null) {
          GroovyClassLoader groovyClassLoader = new GroovyClassLoader(this.applicationContext.getClassLoader());
          GroovyShell shell = new GroovyShell(groovyClassLoader);
          try {
            LOGGER.info("正在进行groovy脚本编译，脚本id = " + conextBase64);
            groovyScript = shell.parse(scriptContent);
            peristentGroovyClassService.save(conextBase64 , groovyScript);
            if(scriptExecuteLocks.get(idHashcode) == null) {
              scriptExecuteLocks.put(idHashcode, new ReentrantLock());
            }
          } catch (CompilationFailedException e) {
            LOGGER.error(e.getMessage(), e);
            throw new IllegalArgumentException("编译groovy脚本时发生错误，请检查！");
          }
        }
      }
    }

    // 绑定参数后，开始执行
    Object returnValue = null;
    ReentrantLock reentrantLock = null;
    try {
      reentrantLock = scriptExecuteLocks.get(idHashcode);
      reentrantLock.lock();
      Binding binding = bindingScriptVars(currentParams);
      groovyScript.setBinding(binding);
      returnValue = groovyScript.run();
      // 将出参情况，重新绑回currentParams中
      Map<String, Object> variables = (Map<String, Object>)binding.getVariables();
      if(variables != null) {
        Set<Entry<String, Object>> entrySet = variables.entrySet();
        for (Entry<String, Object> entry : entrySet) {
          currentParams.put(entry.getKey(), entry.getValue());
        }
      }
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }finally {
      if(reentrantLock != null) {
        reentrantLock.unlock();
      }
    }
    return returnValue;
  }
}
