package com.bizunited.platform.core.service.invoke;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

import com.alibaba.fastjson.JSONObject;

/**
 * 链式调用的入参信息主要分为三个部分:</br>
 * 第一个部分：由K-V结构构成的普通参数</p>
 * 第二个部分：需要直接进行json对象化的json结构。</p>
 * 第三个部分：对象化的，按照添加顺序进行存储的入参</p>
 * </p>
 * 
 * 如果用使用场景进行说明，可以描述如下:</br>
 * 1、当在http使用场景下进行相关调用时，采用request parameter进行的传参，可以写入到InvokeParams的第一部分</p>
 * 2、当在http使用场景下进行相关调用时，采用request body进行的传参可以写入到InvokeParams的第二部分</br>
 * 当然是不是说第2种场景下写入到InvokeParams的第一部分就是错误的呢？也不是，如果在这种场景下写入到InvokeParams的第一部分，就需要技术人员自行决定一个K-V结构的key信息</p>
 * 3、当传入的参数不是json结构，也不能进行序列化，那么可以尝试将这些值对象顺序放入到InvokeParams的第三部分，并在使用时，按序取出</p>
 * 
 * 另外需要注意：</br>
 * a、以上InvokeParams的三个部分的存值都是可选的</br>
 * b、在链式调用工作过程中，如果这些值传递的是引用，则有可能出现变换
 * 
 * TODO 注释要进行修改
 * 
 * @author yinwenjie
 */
public class InvokeParams implements Serializable {

  private static final long serialVersionUID = 6217415824929398935L;
  /**
   * 用来进行入参描述的第一部分值的描述
   */
  private Map<String, Object> invokeParams = new HashMap<>();
  /**
   * 用来进行入参描述的第二部分值的描述
   */
  private JSONObject jsonData = new JSONObject();
  /**
   * 用来进行入参描述的第三部分值的描述
   */
  private List<Object> variables = new ArrayList<>(16);
  
  public InvokeParams(Map<String, Object> invokeParams) {
    Validate.notNull(invokeParams , "初始化调用参数时，K-V结构的入参信息不能为空");
    this.invokeParams.putAll(invokeParams);
  }
  
  public InvokeParams(JSONObject jsonData) {
    Validate.notNull(jsonData , "初始化调用参数时，json结构的入参信息不能为空");
    this.jsonData.putAll(jsonData);
  }

  public InvokeParams(List<Object> variables) {
    Validate.notNull(variables , "初始化调用参数时，非序列化参数variables结构的入参信息不能为空");
    this.variables = variables;
  }
  
  public InvokeParams() {
    
  }
  /**
   * 注意，如果存在原始值，原始的值将被覆盖
   * @param key key值信息
   * @param value 
   */
  public void putInvokeParam(String key , Object value) {
    Validate.notBlank(key , "添加调用链参数时，必须传入key值");
    Validate.notBlank(key , "添加调用链参数时，必须传入value值");
    this.invokeParams.put(key, value);
  }
  /**
   * 注释未写
   * @param source
   */
  public void putInvokeParams(Map<String, Object>  source) {
    if(source == null) {
      return;
    }
    this.invokeParams.putAll(source);
  }
  /**
   * @param key
   * @return
   */
  public Map<String, Object> getInvokeParams() {
    return this.invokeParams;
  }
  /**
   * @param key
   * @return
   */
  public Object getInvokeParam(String key) {
    if(StringUtils.isBlank(key)) {
      return null;
    }
    return this.invokeParams.get(key);
  }
  
  public void clearInvokeParam() {
    this.invokeParams.clear();
  }
  /**
   * 设置新的jsonObject结构信息，这些信息将会被全部替换原有的json结构
   * @param jsonObject
   */
  public void setJsonData(JSONObject jsonData) {
    Validate.notNull(jsonData , "添加调用链json数据时，发现错误的json对象传入");
    this.jsonData = jsonData;
  }
  /**
   * 设置新的jsonObject结构信息，这些信息将会被合并到原有的json结构中
   * @param jsonObject
   */
  public void addJsonData(JSONObject jsonData) {
    Validate.notNull(jsonData , "添加调用链json数据时，发现错误的json对象传入");
    this.jsonData.putAll(jsonData);
  }
  /**
   * TODO 注释未写
   */
  public void clearJsonData() {
    this.jsonData.clear();
  }
  /**
   * TODO 注释未写
   * @return
   */
  public JSONObject getJsonData() {
    return this.jsonData;
  }
  
  public void setVariables(Object ...variables) {
    Validate.notNull(variables , "新设定的变量列表不能为null");
    this.variables.clear();
    this.variables.addAll(Arrays.asList(variables));
  }
  
  public void addVariable(Object ...variables) {
    Validate.notNull(variables , "新设定的变量列表不能为null");
    for (Object variable : variables) {
      this.variables.add(variable);
    }
  }
  
  public Object[] getVariables() {
    return this.variables.toArray();
  }
}