package com.bizunited.platform.core.service.log;

import com.bizunited.platform.core.entity.log.LoggerInfoEntity;
import com.bizunited.platform.core.repository.log.LoggerInfoEntityRepository;
import com.bizunited.platform.core.service.LoggerInfoEntityService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.*;

/**
 * LoggerInfoEntity业务模型的服务层接口实现
 * @author yinwenjie
 */
@Service
public class LoggerInfoEntityServiceImpl implements LoggerInfoEntityService { 
  @Autowired
  private LoggerInfoEntityRepository loggerInfoEntityRepository;
  @Transactional
  @Override
  public LoggerInfoEntity create(LoggerInfoEntity loggerInfoEntity) {
    LoggerInfoEntity current = this.createForm(loggerInfoEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public LoggerInfoEntity createForm(LoggerInfoEntity loggerInfoEntity) {
    //设置创建时间与账号，更新时间与账号
    Date date = new Date();
    SecurityUtils.getUserAccount();
    loggerInfoEntity.setCreateTime(date);
    loggerInfoEntity.setCreateAccount(SecurityUtils.getUserAccount());
    loggerInfoEntity.setModifyTime(date);
    loggerInfoEntity.setModifyAccount(SecurityUtils.getUserAccount());
    this.createValidation(loggerInfoEntity);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.loggerInfoEntityRepository.save(loggerInfoEntity);
    // 返回最终处理的结果，里面带有详细的关联信息
    return loggerInfoEntity;
  }
  /**
   * 在创建一个新的LoggerInfoEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(LoggerInfoEntity loggerInfoEntity) {
    Validate.notNull(loggerInfoEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(loggerInfoEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    loggerInfoEntity.setId(null);
    Validate.notBlank(loggerInfoEntity.getModule(), "添加信息时，日志所属模块不能为空！");
    Validate.notBlank(loggerInfoEntity.getOpId(), "添加信息时，日志对应的业务数据唯一标识不能为空！");
    Validate.notBlank(loggerInfoEntity.getType(), "添加信息时，日志操作类型（同模板中的设定）不能为空！");
    Validate.notNull(loggerInfoEntity.getOpTime(), "添加信息时，日志事件操作时间不能为空！");
    Validate.notBlank(loggerInfoEntity.getAccount(), "添加信息时，操作账号信息（可能匿名）不能为空！");
    Validate.notBlank(loggerInfoEntity.getMsg(), "添加信息时，日志信息描述（来源于模板的信息格式设定）不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(loggerInfoEntity.getModule() == null || loggerInfoEntity.getModule().length() < 255 , "日志所属模块,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getOpId() == null || loggerInfoEntity.getOpId().length() < 255 , "日志对应的业务数据唯一标识,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getType() == null || loggerInfoEntity.getType().length() < 255 , "日志操作类型（同模板中的设定）,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getAccount() == null || loggerInfoEntity.getAccount().length() < 255 , "操作账号信息（可能匿名）,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getMsg() == null || loggerInfoEntity.getMsg().length() < 255 , "日志信息描述（来源于模板的信息格式设定）,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getRelativePath() == null || loggerInfoEntity.getRelativePath().length() < 255 , "日志文件存储相对位置,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getFileName() == null || loggerInfoEntity.getFileName().length() < 255 , "日志文件文件名,在进行添加时填入值超过了限定长度(255)，请检查!");
  }
  @Transactional
  @Override
  public LoggerInfoEntity update(LoggerInfoEntity loggerInfoEntity) {
    LoggerInfoEntity current = this.updateForm(loggerInfoEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public LoggerInfoEntity updateForm(LoggerInfoEntity loggerInfoEntity) {
    this.updateValidation(loggerInfoEntity);
    // ===================基本信息
    String currentId = loggerInfoEntity.getId();
    Optional<LoggerInfoEntity> op_currentLoggerInfoEntity = this.loggerInfoEntityRepository.findById(currentId);
    LoggerInfoEntity currentLoggerInfoEntity = op_currentLoggerInfoEntity.orElse(null);
    currentLoggerInfoEntity = Validate.notNull(currentLoggerInfoEntity ,"未发现指定的原始模型对象信");
    //更新创建时间与账号
    loggerInfoEntity.setModifyTime(new Date());
    loggerInfoEntity.setModifyAccount(SecurityUtils.getUserAccount());
    // 开始重新赋值——一般属性
    currentLoggerInfoEntity.setModule(loggerInfoEntity.getModule());
    currentLoggerInfoEntity.setOpId(loggerInfoEntity.getOpId());
    currentLoggerInfoEntity.setType(loggerInfoEntity.getType());
    currentLoggerInfoEntity.setOpTime(loggerInfoEntity.getOpTime());
    currentLoggerInfoEntity.setAccount(loggerInfoEntity.getAccount());
    currentLoggerInfoEntity.setMsg(loggerInfoEntity.getMsg());
    currentLoggerInfoEntity.setRelativePath(loggerInfoEntity.getRelativePath());
    currentLoggerInfoEntity.setFileName(loggerInfoEntity.getFileName());
    
    this.loggerInfoEntityRepository.saveAndFlush(currentLoggerInfoEntity);
    return currentLoggerInfoEntity;
  }
  /**
   * 在更新一个已有的LoggerInfoEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(LoggerInfoEntity loggerInfoEntity) {
    Validate.isTrue(!StringUtils.isBlank(loggerInfoEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(loggerInfoEntity.getModule(), "修改信息时，日志所属模块不能为空！");
    Validate.notBlank(loggerInfoEntity.getOpId(), "修改信息时，日志对应的业务数据唯一标识不能为空！");
    Validate.notBlank(loggerInfoEntity.getType(), "修改信息时，日志操作类型（同模板中的设定）不能为空！");
    Validate.notNull(loggerInfoEntity.getOpTime(), "修改信息时，日志事件操作时间不能为空！");
    Validate.notBlank(loggerInfoEntity.getAccount(), "修改信息时，操作账号信息（可能匿名）不能为空！");
    Validate.notBlank(loggerInfoEntity.getMsg(), "修改信息时，日志信息描述（来源于模板的信息格式设定）不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(loggerInfoEntity.getModule() == null || loggerInfoEntity.getModule().length() < 255 , "日志所属模块,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getOpId() == null || loggerInfoEntity.getOpId().length() < 255 , "日志对应的业务数据唯一标识,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getType() == null || loggerInfoEntity.getType().length() < 255 , "日志操作类型（同模板中的设定）,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getAccount() == null || loggerInfoEntity.getAccount().length() < 255 , "操作账号信息（可能匿名）,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getMsg() == null || loggerInfoEntity.getMsg().length() < 255 , "日志信息描述（来源于模板的信息格式设定）,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getRelativePath() == null || loggerInfoEntity.getRelativePath().length() < 255 , "日志文件存储相对位置,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerInfoEntity.getFileName() == null || loggerInfoEntity.getFileName().length() < 255 , "日志文件文件名,在进行修改时填入值超过了限定长度(255)，请检查!");
  } 
  @Override
  public LoggerInfoEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.loggerInfoEntityRepository.findDetailsById(id);
  }
  @Override
  public LoggerInfoEntity findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    
    Optional<LoggerInfoEntity> op = loggerInfoEntityRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    LoggerInfoEntity current = this.findById(id);
    if(current != null) { 
      this.loggerInfoEntityRepository.delete(current);
    }
  }
  @Override
  public Page<LoggerInfoEntity> findByOpidAndModule(String opId, String module, Pageable pageable) {
    Validate.notNull(pageable, "查询分页信息不能为空!");
    Map<String,Object> conditions=new HashMap<>(2);
    if(StringUtils.isNotBlank(opId)){
      conditions.put("opId",opId);
    }
    if(StringUtils.isNotBlank(module)){
      conditions.put("module",module);
    }
    return this.loggerInfoEntityRepository.findByConditions(pageable,conditions);
  }
} 
