package com.bizunited.platform.core.service.migrate.internal;

import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.enums.MigrateDataTypeEnum;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.FileUtils;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.MigrateImportDetailEntity;
import com.bizunited.platform.core.entity.MigrateImportEntity;
import com.bizunited.platform.core.repository.migrate.MigrateImportDetailRepository;
import com.bizunited.platform.core.repository.migrate.MigrateImportRepository;
import com.bizunited.platform.core.service.CodeRuleService;
import com.bizunited.platform.core.service.EnvironmentVariableService;
import com.bizunited.platform.core.service.RemoteServiceAddressService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.migrate.MigrateImportService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.bizunited.platform.venus.common.service.image.FileUpdateService;
import com.bizunited.platform.venus.common.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.MigrateDataConstants.ZIP_FILE_SUBFIX;

/**
 * MigrateImportServiceImpl
 *
 * @description: 数据信息迁入的查询，分析，执行。包括基础配置信息与表单信息
 * @author: yanwe
 * @date: 03/Sep/2019 15:47
 */
@Service("MigrateImportService")
public class MigrateImportServiceImpl implements MigrateImportService {

  private static final Logger LOGGER = LoggerFactory.getLogger(MigrateImportServiceImpl.class);

  @Value("${venus.file.fileRoot}")
  private String fileRoot;

  @Autowired
  private MigrateImportRepository migrateImportRepository;
  @Autowired
  private MigrateImportDetailRepository migrateImportDetailRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private FileUpdateService fileUpdateService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private DataViewService dataViewService;
  @Autowired
  private RemoteServiceAddressService remoteServiceAddressService;
  @Autowired
  private EnvironmentVariableService environmentVariableService;
  @Autowired
  private CodeRuleService codeRuleService;
  @Autowired
  private PlatformContext platformContext;

  @Override
  @Transactional
  public MigrateImportEntity upload(MultipartFile file, Integer dataType) {
    Validate.notNull(file, "上传文件不能为空！");
    Validate.notNull(dataType, "数据类型不能为空");
    MigrateDataTypeEnum dataTypeEnum = MigrateDataTypeEnum.valueOfType(dataType);
    Validate.notNull(dataTypeEnum, "不支持的数据类型：%d", dataType);
    UserVo user = SecurityUtils.getCurrentUser();
    MigrateImportEntity migrateImport = new MigrateImportEntity();
    migrateImport.setCreateTime(new Date());
    migrateImport.setExecuted(false);
    migrateImport.setDataType(dataType);
    migrateImport.setCreator(user.getAccount());
    migrateImport.setOriginalFileName(file.getOriginalFilename());
    MultipartFile[] files = new MultipartFile[] {file};
    List<OrdinaryFileVo> ordinaryFileEntities = fileUpdateService.fileUpload("migrate", user.getAccount(), null, files);
    Validate.notEmpty(files, "保存上传文件失败，请检查！");
    OrdinaryFileVo fileEntity = ordinaryFileEntities.get(0);
    Validate.notNull(fileEntity,"上次文件保存返回结果不能为空！");
    migrateImport.setFileName(fileEntity.getFileName());
    migrateImport.setRelativeLocal(fileEntity.getRelativeLocal());
    migrateImport.setProjectName(platformContext.getAppName());
    return this.create(migrateImport);
  }

  @Override
  @Transactional
  public MigrateImportEntity create(MigrateImportEntity migrateImport) {
    Validate.notNull(migrateImport.getCreateTime(), "保存记录时，导入时间不能为空！");
    Validate.notBlank(migrateImport.getRelativeLocal(), "迁入文件在本地路径不能为空！");
    Validate.notBlank(migrateImport.getFileName(), "迁入文件重命名后的文件名字不能为空！");
    Validate.notNull(migrateImport.getCreator(), "迁入文件上传人不能为空！");
    Validate.notNull(migrateImport.getCreateTime(), "迁入文件上传时间不能为空！");
    Validate.notNull(migrateImport.getExecuted(), "迁入文件任务是否被执行不能为空！");
    return migrateImportRepository.save(migrateImport);
  }

  @Override
  public List<MigrateImportEntity> findDetailsByDataType(Integer dataType) {
    if(dataType == null) {
      return Lists.newArrayList();
    }
    List<MigrateImportEntity> imports = null;
    String projectName = platformContext.getAppName();
    if(StringUtils.isNotBlank(projectName)){
      imports = migrateImportRepository.findByDataTypeAndProjectName(dataType, projectName);
    }else {
      imports = migrateImportRepository.findByDataTypeAndBlankProjectName(dataType);
    }
    if(CollectionUtils.isEmpty(imports)) {
      return Lists.newArrayList();
    }
    return this.loadUsers(imports);
  }

  /**
   * 加载用户信息
   * @param imports
   * @return
   */
  private List<MigrateImportEntity> loadUsers(List<MigrateImportEntity> imports) {
    Map<String, UserVo> tmp = new HashMap<>();
    for (MigrateImportEntity anImport : imports) {
      String creator = anImport.getCreator();
      UserVo createUser = tmp.get(creator);
      if(createUser == null) {
        createUser = userService.findByAccount(creator);
        tmp.put(creator, createUser);
      }
      anImport.setCreateUser(createUser);
      String executor = anImport.getExecutor();
      if(StringUtils.isNotBlank(executor)) {
        UserVo executeUser = tmp.get(executor);
        if(executeUser == null) {
          executeUser = userService.findByAccount(executor);
          tmp.put(executor, executeUser);
        }
        anImport.setExecuteUser(executeUser);
      }
    }
    return imports;
  }

  @Override
  public MigrateImportEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return migrateImportRepository.findDetailsById(id);
  }

  @Override
  @Transactional
  public MigrateImportEntity execute(String id, Integer mode) {
    /*
     * 1、验证入参
     * 2、获取导入的文件内容
     * 3、将文件内容存到本地的临时文件中，并加载压缩文件
     * 4、根据数据类型调用对应的导入方法
     * 5、删除临时文件
     * 6、保存明细信息并返回
     *
     * */
    // 1、
    Validate.notBlank(id, "导入记录ID不能为空");
    ImportExecuteModeEnum executeMode;
    if(mode == null) {
      executeMode = ImportExecuteModeEnum.SKIP;
    } else {
      executeMode = ImportExecuteModeEnum.valueOfMode(mode);
      Validate.notNull(executeMode, "不支持的执行模式：%s", mode);
    }
    UserVo user = SecurityUtils.getCurrentUser();
    // 2
    MigrateImportEntity migrateImport = migrateImportRepository.findDetailsById(id);
    Validate.notNull(migrateImport, "未找到导入的记录");
    Validate.isTrue(!migrateImport.getExecuted(), "该导入已经执行过了！！");
    MigrateDataTypeEnum dataType = MigrateDataTypeEnum.valueOfType(migrateImport.getDataType());
    Validate.notNull(dataType, "不支持导入的数据类型:%s", migrateImport.getDataType());
    byte[] bytes = venusFileService.readFileContent(migrateImport.getRelativeLocal(), migrateImport.getFileName());
    MigrateImportDetailEntity importDetail = migrateImport.getDetail();
    if(importDetail == null) {
      importDetail = new MigrateImportDetailEntity();
    }
    Validate.isTrue(bytes != null && bytes.length > 0, "导入的文件内容为空");
    // 3
    File tmpFile = FileUtils.writeLocalFile(bytes, fileRoot, ZIP_FILE_SUBFIX);
    int executeResult = NormalStatusEnum.ENABLE.getStatus();
    try (ZipFile zipFile = new ZipFile(tmpFile)) {
      MigrateImportModel importModel = new MigrateImportModel(zipFile, executeMode);
      importModel.appendLine("执行模式：%s", executeMode.getDesc());
      // 4
      try {
        switch (dataType) {
          case DATA_VIEW:
            dataViewService.importData(importModel);
            break;
          case REMOTE_SERVICE:
            remoteServiceAddressService.importData(importModel);
            break;
          case ENVIRONMENT_VARIABLE:
            environmentVariableService.importData(importModel);
            break;
          case CODE_RULE:
            codeRuleService.importData(importModel);
            break;
          default:
            importModel.setError(true);
            importModel.appendLine("导入错误：不支持的数据类型【%s】", dataType.name());
            break;
        }
        importModel.appendLine("导入完成！");
      } catch (RuntimeException e) {
        LOGGER.error(e.getMessage(), e);
        importModel.setError(true);
        importModel.append("导入错误：").appendLine(e.getMessage());
      } finally {
        importDetail.setCreateCount(importModel.getCreateCount());
        importDetail.setIgnoreCount(importModel.getSkipCount());
        importDetail.setTotalCount(importModel.getTotalCount());
        importDetail.setUpdateCount(importModel.getUpdateCount());
        importDetail.setExecuteLog(importModel.getExecuteLog().toString());
        if(importModel.isError()) {
          executeResult = NormalStatusEnum.DISABLE.getStatus();
        }
      }
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("读取压缩文件出错：%s", e.getMessage()));
    } finally {
      // 5
      tmpFile.delete();
    }
    // 6
    migrateImportDetailRepository.save(importDetail);
    migrateImport.setDetail(importDetail);
    migrateImport.setExecuted(true);
    migrateImport.setExecutor(user.getAccount());
    migrateImport.setExecuteTime(new Date());
    migrateImport.setExecuteResult(executeResult);
    migrateImportRepository.save(migrateImport);
    return migrateImport;
  }

}
