package com.bizunited.platform.core.service.scheduler;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import com.bizunited.platform.core.entity.DynamicTaskSchedulerLogEntity;
import com.bizunited.platform.core.repository.DynamicTaskSchedulerLogRepository;
import com.bizunited.platform.core.service.DynamicTaskSchedulerLogService;
import com.google.common.collect.Sets;

/**
 * DynamicTaskSchedulerLogEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DynamicTaskSchedulerLogServiceImpl")
public class DynamicTaskSchedulerLogServiceImpl implements DynamicTaskSchedulerLogService { 
  @Autowired
  private DynamicTaskSchedulerLogRepository dynamicTaskSchedulerLogRepository;
  @Transactional
  @Override
  public DynamicTaskSchedulerLogEntity create(DynamicTaskSchedulerLogEntity log) { 
    return this.createForm(log);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
  }
  @Transactional
  @Override
  public DynamicTaskSchedulerLogEntity createForm(DynamicTaskSchedulerLogEntity log) { 
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    this.createValidation(log);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.dynamicTaskSchedulerLogRepository.save(log);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return log;
  }
  /**
   * 在创建一个新的DynamicTaskSchedulerLogEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(DynamicTaskSchedulerLogEntity log) { 
    Validate.notNull(log , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(log.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    log.setId(null);
    Validate.notNull(log.getStartTime(), "执行开始时间不能为空！");
    Validate.notNull(log.getEndTime(), "执行结束时间不能为空！"); 
    Validate.notNull(log.getSuccess(), "执行是否成功不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(log.getErrorContents() == null || log.getErrorContents().length() < 512 , "可能的错误日志,填入值超过了限定长度(512)，请检查!");
  }
  @Transactional
  @Override
  public DynamicTaskSchedulerLogEntity update(DynamicTaskSchedulerLogEntity dynamicTaskSchedulerLogEntity) { 
    return this.updateForm(dynamicTaskSchedulerLogEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
  }
  @Transactional
  @Override
  public DynamicTaskSchedulerLogEntity updateForm(DynamicTaskSchedulerLogEntity log) { 
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(log);
    // ===================基本信息
    String currentId = log.getId();
    Optional<DynamicTaskSchedulerLogEntity> op = this.dynamicTaskSchedulerLogRepository.findById(currentId);
    DynamicTaskSchedulerLogEntity currentDynamicTaskSchedulerLogEntity = op.orElse(null);
    Validate.notNull(currentDynamicTaskSchedulerLogEntity ,"未发现指定的原始模型对象信");

    // 开始重新赋值——一般属性
    currentDynamicTaskSchedulerLogEntity.setStartTime(log.getStartTime());
    currentDynamicTaskSchedulerLogEntity.setEndTime(log.getEndTime());
    currentDynamicTaskSchedulerLogEntity.setSuccess(log.getSuccess());
    currentDynamicTaskSchedulerLogEntity.setErrorContents(log.getErrorContents());
    currentDynamicTaskSchedulerLogEntity.setDynamicTaskScheduler(log.getDynamicTaskScheduler());
    
    this.dynamicTaskSchedulerLogRepository.saveAndFlush(currentDynamicTaskSchedulerLogEntity);
    return currentDynamicTaskSchedulerLogEntity;
  }
  /**
   * 在更新一个已有的DynamicTaskSchedulerLogEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(DynamicTaskSchedulerLogEntity dynamicTaskSchedulerLogEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dynamicTaskSchedulerLogEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notNull(dynamicTaskSchedulerLogEntity.getStartTime(), "执行开始时间不能为空！");
    Validate.notNull(dynamicTaskSchedulerLogEntity.getEndTime(), "执行结束时间不能为空！");
    Validate.notNull(dynamicTaskSchedulerLogEntity.getSuccess(), "执行是否成功不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dynamicTaskSchedulerLogEntity.getErrorContents() == null || dynamicTaskSchedulerLogEntity.getErrorContents().length() < 512 , "可能的错误日志,填入值超过了限定长度(512)，请检查!");
    
    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  } 
  @Override
  public Set<DynamicTaskSchedulerLogEntity> findDetailsByDynamicTaskScheduler(String dynamicTaskScheduler) { 
    if(StringUtils.isBlank(dynamicTaskScheduler)) { 
      return Sets.newHashSet();
    }
    return this.dynamicTaskSchedulerLogRepository.findDetailsByDynamicTaskScheduler(dynamicTaskScheduler);
  }
  @Override
  public DynamicTaskSchedulerLogEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dynamicTaskSchedulerLogRepository.findDetailsById(id);
  }
  @Override
  public DynamicTaskSchedulerLogEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<DynamicTaskSchedulerLogEntity> op = dynamicTaskSchedulerLogRepository.findById(id);
    return op.orElse(null);
  }
 @Override
 public Page<DynamicTaskSchedulerLogEntity> findByConditions(Pageable pageable, String taskId, Boolean success, String logtime) {
   Map<String,Object> conditions = new HashMap<>();
   Validate.notBlank(taskId , "查询时，任务id编号必须传入!!");
   conditions.put("taskId", taskId);
   if(success != null){
     conditions.put("success",success);
   }
   if(StringUtils.isNotBlank(logtime)){
     conditions.put("logtime",logtime);
   }
   if(pageable ==null){
     pageable = PageRequest.of(0,50);
   }
   
   return this.dynamicTaskSchedulerLogRepository.queryPage(pageable, conditions);
 }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DynamicTaskSchedulerLogEntity> current = this.dynamicTaskSchedulerLogRepository.findById(id);
    if(current.isPresent()) { 
      this.dynamicTaskSchedulerLogRepository.delete(current.get());
    }
  }
  @Override
  @Transactional
  public void deleteByDynamicTaskSchedulerId(String dynamicTaskSchedulerId) {
    Validate.notBlank(dynamicTaskSchedulerId , "进行删除时，必须给定动态任务编号信息!!");
    this.dynamicTaskSchedulerLogRepository.deleteByDynamicTaskSchedulerId(dynamicTaskSchedulerId);
  }
} 