package com.bizunited.platform.core.service.serviceable.aspect;

import java.lang.reflect.Method;

import org.apache.commons.lang3.Validate;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.core.service.invoke.InvokeProxy;
import com.bizunited.platform.core.service.invoke.InvokeProxyBuilder;
import com.bizunited.platform.core.service.serviceable.handle.ServiceMethodExecutionHandle;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodInfo;

/**
 * NebulaServiceMethod注解的原生切面逻辑，注意的关键点是，各服务源可能存在多层级调用
 * @author yinwenjie
 */
@Aspect
@Component
public class AnnotationNebulaServiceMethodAspect extends AbstractNebulaServiceMethodExecution {
  
  /**
   * 以任何使用了NebulaServiceMethod注解的方法为切面关注点
   */
  @Pointcut(value = "execution(@com.bizunited.platform.core.annotations.NebulaServiceMethod public * *(..))") 
  public void aspectHandle() { 
    
  } 
  
  @SuppressWarnings("unchecked")
  @Around(value="aspectHandle()")  
  public Object around(ProceedingJoinPoint point) {
    MethodSignature methodSignature = (MethodSignature) point.getSignature();
    Method currentMethod = methodSignature.getMethod();
    Object[] args = point.getArgs();
    
    /* 
     * 处理过程为：
     * 1、如果当前执行的method没有被AOP拦截过，则根据调用场景构建InvokeProxy，完成request Handle-response Handle的调用过程
     * 2、如果当前执行的method已经被AOP拦截过，那么就直接进行调用，
     * 调用过程无论是否成功，都会移除这个方法——从当前线程的NebulaServiceMethodExecutionInfo信息中
     * */
    // 如果条件成立，说明是1、需要建立代理器
    Object result = null;
    // 1、========
    if(this.isExecutionInfoEmpty() || !this.hasExecutionInfo(currentMethod)) {
      try {
        this.pushExecutionInfo(currentMethod);
        NebulaServiceMethod nebulaServiceMethod = currentMethod.getAnnotation(NebulaServiceMethod.class);
        String servicableMethodName = nebulaServiceMethod.name();
        ServicableMethodInfo servicableMethodInfo = super.findByServicableMethodName(servicableMethodName);
        Validate.notNull(servicableMethodInfo , "未发现名为servicableMethodName的服务源，请检查!!");
        
        // 生成代理器
        InvokeProxyBuilder invokeProxyBuilder = this.buildInvokeProxy(servicableMethodInfo);
        invokeProxyBuilder.addInvokeRequestTypeFilter(ServiceMethodExecutionHandle.class);
        InvokeProxy invokeProxy = invokeProxyBuilder.build();
        // 准备入参信息
        InvokeParams invokeParams = new InvokeParams();
        if(args != null && args.length != 0) {
          invokeParams.setVariables(args);
        }
        invokeParams.putInvokeParam(ServiceMethodExecutionHandle.SERVER_NAME_PARAM, servicableMethodName);
        // 开始调用
        result = invokeProxy.invoke(invokeParams);
        return result;
      } catch (RuntimeException | Error ex) {
        throw ex;
      } catch (Throwable thr) {
        Rethrower.rethrow(thr);
      } finally {
        this.popExecutionInfo();
      }
    } 
    // 如果条件成立，则是2
    else if(!this.isExecutionInfoEmpty() && this.hasExecutionInfo(currentMethod)) {
      try {
        return point.proceed(point.getArgs());
      } catch (RuntimeException | Error ex) {
        throw ex;
      } catch (Throwable thr) {
        Rethrower.rethrow(thr);
      }
    }
    
    // 不可能执行到这里
    return null;
  }
  
  /**
   * 重新抛出异常
   */
  private static class Rethrower {
    public static void rethrow(final Throwable exception) {
      class CheckedExceptionRethrower<T extends Throwable> {
        @SuppressWarnings("unchecked")
        private void rethrow(Throwable exception) throws T {
          throw (T) exception;
        }
      }
      new CheckedExceptionRethrower<RuntimeException>().rethrow(exception);
    }
  }
}
