package com.bizunited.platform.core.service.serviceable.template;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.core.service.invoke.InvokeProxyContext;

import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.security.Principal;
import java.time.temporal.Temporal;
import java.util.Date;
import java.util.Map;

/**
 * 处理调用时传来的data-json结构——该对象结构对应的json结构是存在于invokeParams属性中的一个K-V结构
 * 所以该解析器需要使用ServiceMethodParam注解
 * @author yinwenjie
 */
@Component
public class ObjectDataValueParamMappingTemplate extends AbstractObjectValueMappingTemplate {
  
  @Override
  public boolean match(Class<?> targetClass, Method targetMethod, Parameter parameter, int index) {
    /*
     * 满足当前模板的具体处理条件是：
     * 1、下层实际入参不是一个数组
     * 2、下层实际入参有ServiceMethodParam注解
     * 3、并不是诸如InvokeProxyContext、Principal、Pageable这样的保留类
     * 4、不是java中的八大基础类型
     * */
    // 1、========
    Class<?> parameterType = parameter.getType();
    boolean isArray = parameterType.isArray();
    if(isArray) {
      return false;
    }
    
    // 2、========
    ServiceMethodParam serviceMethodParam = parameter.getAnnotation(ServiceMethodParam.class);
    if(serviceMethodParam == null) {
      return false;
    }
    // 被支持的目标类型不能是JSON类型
    if(JSON.class.isAssignableFrom(parameterType)) {
      return false;
    }
    
    // 3、========
    if(parameterType == InvokeProxyContext.class || parameterType == InvokeParams.class || Principal.class.isAssignableFrom(parameterType)
        || Pageable.class.isAssignableFrom(parameterType) || Date.class.isAssignableFrom(parameterType) 
        || Number.class.isAssignableFrom(parameterType) || Temporal.class.isAssignableFrom(parameterType)
        || CharSequence.class.isAssignableFrom(parameterType)) {
      return false;
    }
    
    // 4、========
    if(parameterType.isPrimitive()) {
      return false;
    }
    
    return true;
  }

  @SuppressWarnings("unchecked")
  @Override
  public Object mapping(Class<?> targetClass, Method targetMethod, Parameter parameter, int index, String parameterName , InvokeParams invokeParams) {
    Object values = invokeParams.getInvokeParam(parameterName);
    if(values == null) {
      return null;
    }
    if(!(values instanceof Map)) {
      return null;
    }
    Map<String, Object> mapping = (Map<String, Object>)values;
    JSONObject jsonObject = new JSONObject(mapping);
    Class<?> parameterClass = parameter.getType();
    return this.mappingObject(jsonObject, parameterClass);
  }
}