package com.bizunited.platform.core.service.serviceable.template;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;

import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;

/**
 * 参数类型转换和值映射处理模板。有效减少if-else判定分支</p>
 * 
 * 在end-point层发生调用时，会传入不同的参数key-value信息，以及json-object信息，这些信息需要根据join-point层下层具体的实际方法进行参数值转换
 * 也就是说字符串的“1”可能需要转换成int或者char。不同的类型转换需要由不同的转换模板类进行支持。</p>
 * 
 * 注意，这个end-point链接层规定暂不支持多维数组，所以多维数组的情况不会出现在具体模板调用过程中，开发者也无需进行检查。
 * 只需要将精力集中在对单个对象和一维数组对象的转换处理上</p>
 * 
 * @see java.lang.Boolean#TYPE
 * @see java.lang.Character#TYPE
 * @see java.lang.Byte#TYPE
 * @see java.lang.Short#TYPE
 * @see java.lang.Integer#TYPE
 * @see java.lang.Long#TYPE
 * @see java.lang.Float#TYPE
 * @see java.lang.Double#TYPE
 * @see java.lang.Void#TYPE 
 * 
 * </p>
 * also see：</br>
 * boolean    Z </br>
 * byte       B </br>
 * char       C </br>
 * class or interface   L classname </br>
 * double     D </br>
 * float      F </br>
 * int        I </br>
 * long       J </br>
 * short      S </br>
 * 
 * @author yinwenjie
 */
public interface ValueMappingTemplate {
  /**
   * 该方法用于检测当前给定对象类型，是否匹配某个具体模板的处理过程。
   * 
   * @param targetComponentType 当前连接点的具体类型
   * @param targetMethod 当前连接点方法
   * @param parameter 当前连接点方法参数信息
   * @param index 入参的索引位置
   * @return 如果当前这个具体的参数值处理模板支持parameterType这个类型的值映射处理过程，则返回true，其它情况返回false
   */
  boolean match(Class<?> targetClass , Method targetMethod , Parameter parameterType , int index);
  
  /**
   * 使用该方法完成具体的参数值映射
   * @param targetComponentType 当前连接点的具体类型
   * @param targetMethod 当前连接点方法
   * @param parameter 当前连接点方法参数信息
   * @param index 入参的索引位置
   * @param context 链式调用的上下文对象，所有入参信息都在这里
   * @param parameterName 对于那些使用了ServiceMethodParam注解进行标识的参数信息，其ServiceMethodParam注解的name信息在这里表示
   * @return 如果映射成功，则返回相应结果，如果映射失败（且允许这样的失败），则返回null
   */
  Object mapping(Class<?> targetClass , Method targetMethod , Parameter parameterType , int index , String parameterName , InvokeParams invokeParams);
  
  /**
   * 该默认方法无需下层实现类进行实现，主要判定当前参数是否含有ServiceMethodParam注解进行标识
   * @param parameter 当前连接点方法参数信息
   * @return
   */
  default boolean checkServiceMethodParamAnnotation(Parameter parameter) {
    return parameter.getAnnotation(ServiceMethodParam.class) != null;
  }
}