package com.bizunited.platform.rbac.server.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RolePositionMappingService;
import com.bizunited.platform.user.common.vo.PositionVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

@RestController
@RequestMapping("/v1/nebula/rolePositionMapping")
public class RolePositionMappingController extends BaseController {

  @Autowired
  private RolePositionMappingService rolePositionMappingService;
  /**
   * 配置的那些可以忽略方法级检查的，具有超级管理员性质的角色名
   */
  @Value("${rbac.ignoreMethodCheckRoles:ADMIN}")
  private String[] ignoreMethodCheckRoles;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RolePositionMappingController.class);

  /**
   * 将指定角色绑定一个或多个岗位
   *
   * @param roleId
   * @param positionCodes
   * @return
   */
  @ApiOperation(value = "将指定角色绑定一个或多个岗位")
  @PatchMapping("bindPositions")
  public ResponseModel bindPositions(
      @RequestParam @ApiParam("角色id") String roleId,
      @RequestParam @ApiParam("当前要绑定的一个或多个岗位编码") String[] positionCodes) {
    try {
      rolePositionMappingService.bindPositions(roleId, positionCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 解除指定角色和一个或多个岗位的绑定关系
   * @param roleId
   * @param positionCodes
   * @return
   */
  @ApiOperation(value = "解除指定角色和一个或多个岗位的绑定关系")
  @PatchMapping("unbindPositions")
  public ResponseModel unbindPositions(
      @RequestParam @ApiParam("角色id") String roleId,
      @RequestParam @ApiParam("当前要解绑的一个或多个岗位编码") String[] positionCodes) {
    try {
      rolePositionMappingService.unbindPositions(roleId, positionCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定角色重新绑定一个或多个岗位
   *
   * @param roleId
   * @param positionCodes
   * @return
   */
  @ApiOperation(value = "将指定角色重新绑定一个或多个岗位")
  @PatchMapping("rebindPositions")
  public ResponseModel rebindPositions(
      @RequestParam @ApiParam("角色id") String roleId,
      @RequestParam(required = false) @ApiParam("当前要绑定的一个或多个岗位编码") String[] positionCodes) {
    try {
      rolePositionMappingService.rebindPositions(roleId, positionCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定岗位绑定一个或多个角色
   *
   * @param positionCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定岗位绑定一个或多个角色")
  @PatchMapping("bindPositionRoles")
  public ResponseModel bindPositionRoles(
      @RequestParam @ApiParam("岗位编码") String positionCode,
      @RequestParam @ApiParam("当前要绑定的一个或多个角色id") String[] roleIds) {
    try {
      rolePositionMappingService.bindPositionRoles(positionCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定岗位解除一个或多个角色的绑定
   *
   * @param positionCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定岗位解除一个或多个角色的绑定")
  @PatchMapping("unbindPositionRoles")
  public ResponseModel unbindPositionRoles(
      @RequestParam @ApiParam("岗位编码") String positionCode,
      @RequestParam @ApiParam("当前要解绑定的一个或多个角色id") String[] roleIds) {
    try {
      rolePositionMappingService.unbindPositionRoles(positionCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定岗位重新绑定一个或多个角色
   * @param positionCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定岗位重新绑定一个或多个角色")
  @PatchMapping("rebindPositionRoles")
  public ResponseModel rebindPositionRoles(
      @RequestParam @ApiParam("岗位编码") String positionCode,
      @RequestParam(required = false) @ApiParam("当前要绑定的一个或多个角色id") String[] roleIds) {
    try {
      rolePositionMappingService.rebindPositionRoles(positionCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色ID查询直接关联的岗位
   * @param roleId
   * @return
   */
  @ApiOperation(value = "根据角色ID查询直接关联的岗位")
  @GetMapping("findPositionsByRoleId")
  public ResponseModel findByRoleId(@ApiParam(value = "指定的角色编号") @RequestParam String roleId) {
    try {
      Set<PositionVo> result = rolePositionMappingService.findByRoleId(roleId);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
