package com.bizunited.platform.rbac.server.starter.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.Constants;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.enums.CompetenceTypeEnum;
import com.bizunited.platform.rbac.server.service.ButtonService;
import com.bizunited.platform.rbac.server.service.CompetenceService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.starter.entity.ButtonEntity;
import com.bizunited.platform.rbac.server.starter.entity.CompetenceEntity;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.repository.CompetenceRepository;
import com.bizunited.platform.rbac.server.starter.repository.RoleRepository;
import com.bizunited.platform.rbac.server.vo.ButtonVo;
import com.bizunited.platform.rbac.server.vo.CompetenceVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;
import org.springframework.util.PathMatcher;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 功能信息变化不大，且经常需要查询，所以需要将查询性质的操作放入到缓存中<br>
 * 功能在服务中的描述信息，主要为了和角色进行绑定产生权限控制
 * @author yinwenjie
 */
public class CompetenceServiceImpl implements CompetenceService {
  /**
   * 权限repository自动注入.
   */
  @Autowired
  private CompetenceRepository competenceRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private RoleService roleService;
  @Autowired
  private UserService userService;
  @Autowired
  private ButtonService buttonService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  /**
   * 可忽略方法级判定的角色名
   */
  @Value("${rbac.ignoreMethodCheckRoles}")
  private String[] ignoreMethodCheckRoles; 
  @Autowired
  private RequestMappingHandlerMapping requestMappingHandlerMapping;
  private static final String MESS_RESOURCE = "resource";
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#create(com.bizunited.platform.rbac.server.vo.CompetenceVo)
   */
  @Override 
  @Transactional
  public CompetenceVo create(CompetenceVo comp) {
    // 验证
    this.validCompetence(comp);
    if(comp.getViewItem().equals(NormalStatusEnum.ENABLE.getStatus())) {
      String code = comp.getCode();
      Validate.notNull(code, "菜单编码不能为空");
      long count = competenceRepository.countByCode(comp.getCode());
      Validate.isTrue(count == 0, "菜单编码不能重复");
    }
    comp.setId(null);
    // 验证是否已经存在
    comp.setMethods(comp.getMethods().toUpperCase());
    comp.setTstatus(NormalStatusEnum.ENABLE.getStatus());
    comp.setCreateDate(new Date());
    comp.setModifyDate(new Date());
    //添加默认排序索引
    if(comp.getSortIndex() == null){
      comp.setSortIndex(Constants.DEFAULT_COMPETENCE_SORT_INDEX);
    }
    // 转换entity后保存
    CompetenceEntity competenceEntity = this.nebulaToolkitService.copyObjectByWhiteList(comp, CompetenceEntity.class, HashSet.class, ArrayList.class, "parent");
    competenceRepository.save(competenceEntity);
    comp.setId(competenceEntity.getId());
    return comp;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#createAll(java.util.List)
   */
  @Override
  @Transactional
  public List<CompetenceVo> createAll(List<CompetenceVo> comps) {
    Validate.isTrue(comps != null && !comps.isEmpty() , "批量添加功能时，必须进行批量传参");
    for (CompetenceVo competenceVo : comps) {
      this.create(competenceVo);
    }
    return comps;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#update(com.bizunited.platform.rbac.server.vo.CompetenceVo)
   */
  @Override
  @Transactional
  public CompetenceVo update(CompetenceVo comp) {
    Validate.notNull(comp , "必须传入需要修改的功能信息!!");
    String compId = comp.getId();
    Validate.notBlank(compId , "错误的功能信息编号，请检查!!");
    this.validCompetence(comp);
    Optional<CompetenceEntity> op = this.competenceRepository.findById(compId);
    // 验证对象
    Validate.isTrue(op.isPresent(), "id为%s的功能对象在数据库中不存在！" , compId);
    CompetenceEntity currentComp = op.get();
    //更新验证功能备注是否重复
    if (comp.getViewItem().equals(NormalStatusEnum.ENABLE.getStatus())){
      long count = competenceRepository.countByCommentAndViewItemExceptId(currentComp.getId(), comp.getComment(), comp.getViewItem());
      Validate.isTrue(count == 0, "功能备注名重复，请重新输入功能名!!");
    }
    currentComp.setResource(comp.getResource());
    currentComp.setMethods(comp.getMethods());
    currentComp.setComment(comp.getComment());
    currentComp.setDescription(comp.getDescription());
    currentComp.setType(comp.getType());
    currentComp.setTag(comp.getTag());
    //排序索引
    if(comp.getSortIndex() == null){
      currentComp.setSortIndex(Constants.DEFAULT_COMPETENCE_SORT_INDEX);
    }else {
      currentComp.setSortIndex(comp.getSortIndex());
    }
    currentComp.setIcon(comp.getIcon());
    currentComp.setModifyDate(new Date());
    // 验证通过后methods统一大写存储
    currentComp.setMethods(currentComp.getMethods().toUpperCase());
    CompetenceVo parent = comp.getParent();
    CompetenceEntity oldParent = currentComp.getParent();
    // 当传入的父级ID不为null，并且旧父级为null或者新父级不等于旧父级时，则更新父级
    if(parent != null && (oldParent == null || !oldParent.getId().equals(parent.getId()))){
      CompetenceEntity entity = competenceRepository.findById(parent.getId()).orElse(null);
      Validate.notNull(entity, "未找到上级菜单");
      Validate.isTrue(!parent.getId().equals(currentComp.getId()), "不能将自身设置为上级菜单");
      Set<String> compStack = new HashSet<>();
      compStack.add(currentComp.getId());
      this.validCompetenceCircular(entity, compStack);
      currentComp.setParent(entity);
    } else if(parent == null) {
      currentComp.setParent(null);
    }
    competenceRepository.saveAndFlush(currentComp);
    return comp;
  }

  /**
   *判断是否形成循环依赖
   * @param competence
   * @param compStack
   */
  private void validCompetenceCircular(CompetenceEntity competence, Set<String> compStack) {
    Validate.notNull(competence, "必须传入菜单信息");
    CompetenceEntity parent = competence.getParent();
    if(parent == null) {
      return;
    }
    Validate.isTrue(!compStack.contains(parent.getId()), "形成循环依赖，更新失败，请检查！");
    compStack.add(parent.getId());
    this.validCompetenceCircular(parent, compStack);
  }

  /**
   * 验证权限对象和字段.
   * @param comp 权限对象
   */
  private void validCompetence(CompetenceVo comp) {
    // 验证对象是否存在
    Validate.notNull(comp, "功能对象不能为空！");
    // 权限串
    String resource = comp.getResource();
    Validate.notNull(resource, "功能URL串不能为空! 如果没有功能的url，请传递空字符串!");
    // 涉及的方法描述（POST/GET……）
    String method = comp.getMethods();
    Validate.notBlank(method, "方法描述不能为空！");
    // 备注
    String comment = comp.getComment();
    Validate.notBlank(comment, "功能备注不能为空！");
    //功能名
    Validate.notNull(comp.getViewItem(), "功能项是否显示在菜单树中必传");
    if(comp.getParent() != null) {
      Validate.notBlank(comp.getParent().getId(), "设置上级菜单必须传入上级菜单id");
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#updateStatus(java.lang.String, java.lang.Boolean)
   */
  @Override
  @Transactional
  public CompetenceVo updateStatus(String id, Boolean flag) {
    if (StringUtils.isEmpty(id) || flag == null) {
      throw new IllegalArgumentException("updateStatus操作时参数错误！");
    }
    
    CompetenceEntity competence = this.recursiveUpdateStatus(id, flag);
    return this.nebulaToolkitService.copyObjectByWhiteList(competence, CompetenceVo.class, HashSet.class, ArrayList.class);
  }
  
  /**
   * 递归更新子级功能信息的状态
   * @param parentCompetenceId
   * @param flag
   */
  private CompetenceEntity recursiveUpdateStatus(String parentCompetenceId, Boolean flag) {
    Optional<CompetenceEntity> op = this.competenceRepository.findById(parentCompetenceId);
    Validate.isTrue(op.isPresent(), "对象不存在！");
    CompetenceEntity competenceEntity = op.get();
    if (Boolean.TRUE.equals(flag)) {
      competenceEntity.setTstatus(1);
    } else {
      competenceEntity.setTstatus(0);
    }
    competenceEntity.setModifyDate(new Date());
    competenceRepository.save(competenceEntity);
    // 子级功能信息
    List<CompetenceEntity> childList = this.competenceRepository.findByParentId(parentCompetenceId);
    for(int index = 0 ; childList != null && index < childList.size() ; index++) {
      this.recursiveUpdateStatus(childList.get(index).getId(), flag);
    }
    
    return competenceEntity;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#bindCompetence(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindCompetence(String roleId, String[] competenceIds) {
    Validate.notBlank(roleId , "进行功能和角色绑定时，必须指定当前角色编号!!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    Validate.isTrue(competenceIds != null && competenceIds.length > 0 , "进行功能和角色绑定时，必须至少指定一个功能编号!!");
    for (String competenceId : competenceIds) {
      Optional<CompetenceEntity> opCompetence = this.competenceRepository.findById(competenceId);
      Validate.isTrue(opCompetence.isPresent() , "没有发现指定的功能信息[%s]，请检查!!" , competenceId);
      Validate.isTrue(CompetenceTypeEnum.COMPETENCE_VIEW_ITEM_ONE.getType().equals(opCompetence.get().getViewItem()), "只能绑定菜单");
      int count = this.competenceRepository.countByRoleIdAndCompetenceId(roleId, competenceId);
      if(count == 0) {
        this.competenceRepository.bindCompetence(roleId, competenceId);
      }
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#unbindCompetence(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unbindCompetence(String roleId, String[] competenceIds) {
    Validate.notBlank(roleId , "进行功能和角色解绑时，必须指定当前角色编号!!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    Validate.isTrue(competenceIds != null && competenceIds.length > 0 , "进行功能和角色解绑时，必须至少指定一个功能编号!!");

    for (String competenceId : competenceIds) {
      Optional<CompetenceEntity> optional = this.competenceRepository.findById(competenceId);
      Validate.isTrue(optional.isPresent() , "没有发现指定的功能信息[%s]，请检查!!" , competenceId);
      this.competenceRepository.unbindCompetence(roleId, competenceId);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#deleteById(java.lang.String)
   */
  @Override
  @Transactional
  public void deleteById(String competenceId) {
    Validate.notBlank(competenceId , "删除功能时，指定的功能编号必须传递!!");
    Optional<CompetenceEntity> op = this.competenceRepository.findById(competenceId);
    Validate.isTrue(op.isPresent() , "未发现指定功能编号的功能，请检查参数!!");
    CompetenceEntity currentCompetence = op.get();
    Set<RoleEntity> roles = currentCompetence.getRoles();
    Validate.isTrue(roles == null || roles.isEmpty() , "指定的功能已经绑定角色，不能进行删除!!");
    Set<ButtonEntity> buttons = currentCompetence.getButtons();
    for(ButtonEntity button : buttons){
      buttonService.deleteById(button.getId());
    }
    List<CompetenceEntity> childs = this.competenceRepository.findByParentId(competenceId);
    Validate.isTrue(childs == null || childs.isEmpty() , "指定的功能存在子级功能，不能进行删除!!");
    
    this.competenceRepository.delete(currentCompetence);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findByViewItem(java.lang.Boolean)
   */
  @Override
  public JSONArray findByViewItem(Boolean viewItem) {
    if(viewItem == null) {
      return null;
    }
    List<CompetenceEntity> dbResults;
    if(Boolean.TRUE.equals(viewItem)) {
      dbResults = this.competenceRepository.findByViewItem(1);
    } else {
      dbResults = this.competenceRepository.findByViewItem(0);
    }
    if(dbResults == null || dbResults.isEmpty()) {
      return null;
    }
    return this.buildParent(dbResults);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findByViewItemAndStatus(java.lang.Boolean, java.lang.Integer)
   */
  @Override
  public JSONArray findByViewItemAndStatus(Boolean viewItem, Integer tstatus) {
    if(viewItem == null || tstatus == null) {
      return null;
    }
    List<CompetenceEntity> dbResults = null;
    if(Boolean.TRUE.equals(viewItem)) {
      dbResults = this.competenceRepository.findByViewItemAndStatus(1, tstatus);
    } else {
      dbResults = this.competenceRepository.findByViewItemAndStatus(0, tstatus);
    }
    if(dbResults == null || dbResults.isEmpty()) {
      return null;
    }
    return this.buildParent(dbResults);
  }
  
  // 由于其中的parent属性存在基于持久层对象的级联关系，所以需要手动去掉
  private JSONArray buildParent(List<CompetenceEntity> dbResults) {
    JSONArray jsonResults = new JSONArray();
    for (CompetenceEntity competenceItem : dbResults) {
      JSONObject jsonObject = new JSONObject();
      jsonObject.put("comment", competenceItem.getComment());
      jsonObject.put("description", competenceItem.getDescription());
      jsonObject.put(MESS_RESOURCE, competenceItem.getResource());
      jsonObject.put("createDate", competenceItem.getCreateDate());
      jsonObject.put("id", competenceItem.getId());
      jsonObject.put("methods", competenceItem.getMethods());
      jsonObject.put("modifyDate", competenceItem.getModifyDate());
      jsonObject.put("sortIndex", competenceItem.getSortIndex());
      jsonObject.put("tstatus", competenceItem.getTstatus());
      jsonObject.put("viewItem", competenceItem.getViewItem());
      jsonObject.put("extractUri", competenceItem.getExtractUri());
      jsonObject.put("icon", competenceItem.getIcon());
      jsonObject.put("tag", competenceItem.getTag());
      jsonObject.put("type", competenceItem.getType());
      jsonObject.put("code", competenceItem.getCode());
      
      CompetenceEntity parent = competenceItem.getParent();
      if(parent != null) {
        JSONObject parentObject = new JSONObject();
        parentObject.put("id", parent.getId());
        jsonObject.put("parent", parentObject);
      } else {
        jsonObject.put("parent", null);
      }
      jsonResults.add(jsonObject);
    }
    
    return jsonResults;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findByViewItemAndRoleNamesAndStatus(java.lang.Boolean, java.lang.String[], java.lang.Integer)
   */
  @Override
  public JSONArray findByViewItemAndRoleNamesAndStatus(Boolean viewItem , String[] roleNames , Integer status) {
    Validate.isTrue(roleNames != null && roleNames.length > 0, "要进行判定的角色信息必须传入");
    // 对当前用户的角色名称与yml配置的管理员角色信息取交集，判定是否具有管理员角色
    Sets.SetView<String> intersections = Sets.intersection(Sets.newHashSet(ignoreMethodCheckRoles), Sets.newHashSet(roleNames));
    List<CompetenceEntity> competences = null;
    // 如果求得的交集不为空，直接返回null。null代表该用户拥有所有的菜单权限
    if(!CollectionUtils.isEmpty(intersections)){
      if (status != null){
        competences = this.competenceRepository.findByViewItemAndStatus(Boolean.TRUE.equals(viewItem) ? 1:0, status);
      }else {
        competences = this.competenceRepository.findByViewItem(Boolean.TRUE.equals(viewItem) ? 1:0);
      }
      return this.buildParent(competences);
    }

    if(status != null) {
      //获取该用户当前角色下的功能状态正常的功能集
      competences = this.competenceRepository.findByViewItemAndRoleNamesAndStatus(Boolean.TRUE.equals(viewItem) ? 1:0 , roleNames, status);
    } else {
      //查询该用户当前角色下的所有功能集
      competences = this.competenceRepository.findByViewItemAndRoleNames(Boolean.TRUE.equals(viewItem) ? 1:0 , roleNames);
    }
    //这里可能会存在当前用户还未分配功能权限的情况，如果在这里没有发现功能集competences有值，那么这里就会返回一个空的功能数组
    if(CollectionUtils.isEmpty(competences)){
      return new JSONArray();
    }
    return this.buildParent(competences);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findByResource(java.lang.String, java.lang.Integer)
   */
  @Override
  public List<CompetenceVo> findByResource(String resource , Integer tstatus) {
    if(StringUtils.isBlank(resource)) {
      return Lists.newArrayList();
    }
    
    List<CompetenceEntity> currentCompetences = null;
    if(tstatus != null) {
      currentCompetences = this.competenceRepository.findByResourceAndTstatus(resource, tstatus);
    } else {
      currentCompetences = this.competenceRepository.findByResource(resource);
    }
    Collection<CompetenceVo> competenceVos = this.nebulaToolkitService.copyCollectionByWhiteList(currentCompetences, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newLinkedList(competenceVos);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findByUrlResource(java.lang.String[], java.security.Principal)
   */
  @Override
  public JSONArray findByUrlResource(String[] resources, Principal userPrincipal) {
    /*
     * 处理方式为：
     * 1、首先判定当前用户的角色信息
     * 2、如果当前用户的角色有一个属于“不需要进行功能级限制”的角色（也就是属于ignoreMethodCheckRoles）
     * 那么传递的resources将全部被标识为“可以访问”
     * 3、如果提交2不成立，则取出这个用户下所有的功能信息，并进行访问路径判定
     * */
    // 1、=======
    UserVo currentUser = userService.findByAccount(userPrincipal.getName());
    JSONArray allCompetences = new JSONArray();
    Validate.isTrue(resources != null && resources.length > 0 , "至少需要指定一个resources参数信息，请检查!!");
    Validate.notNull(currentUser , "未找到指定的用户信息，请检查!!");
    List<RoleVo> roles = this.roleService.findAllByUserId(currentUser.getId(), 0);
    Validate.isTrue(roles != null && !roles.isEmpty() , "未找到任何角色属性信心，请检查!!");
    Set<String> roleNames = roles.stream().map(RoleVo::getRoleName).collect(Collectors.toSet());
    
    // 2、======
    Set<String> ignoreRoles = Sets.newHashSet(ignoreMethodCheckRoles);
    JSONArray results = new JSONArray();
    // 如果条件成立，则无需再进行功能查询了，以上给定的路径都可用
    if(ignoreRoles != null && !Sets.intersection(roleNames, ignoreRoles).isEmpty()) {
      for (String resource : resources) {
        JSONObject node = new JSONObject();
        node.put(MESS_RESOURCE, resource);
        node.put("flag", true);
        results.add(node);
      }
      return results;
    }
    
    // 3、======
    JSONArray noViewItemCompetences = this.findByViewItemAndRoleNamesAndStatus(false, roleNames.toArray(new String[]{}), null);
    JSONArray viewItemCompetences = this.findByViewItemAndRoleNamesAndStatus(true, roleNames.toArray(new String[]{}), null);
    if(noViewItemCompetences != null) {
      allCompetences.addAll(noViewItemCompetences);
    }
    if(viewItemCompetences != null) {
      allCompetences.addAll(viewItemCompetences);
    }
    // 如果条件成立，说明没有任何匹配路径
    if(CollectionUtils.isEmpty(allCompetences)) {
      for (String resource : resources) {
        JSONObject node = new JSONObject();
        node.put(MESS_RESOURCE, resource);
        node.put("flag", false);
        results.add(node);
      }
      return results;
    }
    // 使用MVC提供的路径匹配工具，进行URL路径匹配
    PathMatcher pathMatcher = requestMappingHandlerMapping.getPathMatcher();
    Map<String , Boolean> resourceMapping = new LinkedHashMap<>();
    for(String resource : resources) {
      boolean matcher = false;
      for(int index = 0 ; index < allCompetences.size() ; index++) {
        JSONObject competenceItem = allCompetences.getJSONObject(index);
        Integer extractUri = competenceItem.getInteger("extractUri");
        Integer tstatus = competenceItem.getInteger("tstatus");
        String currentResource = competenceItem.getString(MESS_RESOURCE);
        if(tstatus != 1) {
          continue;
        }
        
        // 如果条件成立，说明是动态路径，那么动过pathMatcher去判断
        if(extractUri == 1) {
          try {
            Map<String, String> pathMatcherMapping = pathMatcher.extractUriTemplateVariables(currentResource, resource);
            if(pathMatcherMapping != null) {
              matcher = true;
              break;
            }
          } catch(Exception e) {
            // 吃掉
          }
        } 
        // 否则只需要比对完整的url路径
        else if(extractUri == 0 && StringUtils.equals(currentResource, resource)) {
          matcher = true;
          break;
        }
      }
      resourceMapping.put(resource, matcher);
    }
    
    // 构建一个前端要求的对象形式
    Set<String> keys = resourceMapping.keySet();
    for (String resource : keys) {
      JSONObject node = new JSONObject();
      node.put(MESS_RESOURCE, resource);
      node.put("flag", resourceMapping.get(resource));
      results.add(node);
    }
    return results;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findByConditions(java.lang.String, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<CompetenceVo> findByConditions(CompetenceVo competence, Pageable pageable) {
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    
    // Page中的内容必须自己转换
    Page<CompetenceEntity> competenceEntityPage = this.competenceRepository.findByConditions(pageable, competence);
    List<CompetenceEntity> competenceEntitys = competenceEntityPage.getContent();
    Page<CompetenceVo> competenceVoPage = null;
    if(!CollectionUtils.isEmpty(competenceEntitys)) {
      Collection<CompetenceVo> competenceVos = this.nebulaToolkitService.copyCollectionByWhiteList(competenceEntitys, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class);
      competenceVoPage = new PageImpl<>(new ArrayList<>(competenceVos), pageable, competenceEntityPage.getTotalElements());
    } else {
      competenceVoPage = new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    return competenceVoPage;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findById(java.lang.String)
   */
  @Override
  public CompetenceVo findById(String competenceId) {
    if(StringUtils.isBlank(competenceId)) {
      return null;
    }
    Optional<CompetenceEntity> op = competenceRepository.findById(competenceId);
    if(!op.isPresent()) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(op.get(), CompetenceVo.class, LinkedHashSet.class, ArrayList.class);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.CompetenceService#findAll()
   */
  @Override
  public List<CompetenceVo> findAll() {
    List<CompetenceEntity> competences = this.competenceRepository.findAll();
    if(competences == null || competences.isEmpty()) {
      return Lists.newArrayList();
    }
    Collection<CompetenceVo> competenceVos = this.nebulaToolkitService.copyCollectionByWhiteList(competences, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class, "parent");
    return Lists.newArrayList(competenceVos);
  }

  /**
   * 根据按钮id查询绑定的接口信息
   * @param buttonId
   * @return
   */
  @Override
  public Set<CompetenceVo> findByButtonId(String buttonId){
    if(StringUtils.isBlank(buttonId)){
      return Sets.newHashSet();
    }
    Set<CompetenceEntity> competences = this.competenceRepository.findByButtonId(buttonId);
    Collection<CompetenceVo> competenceVos = this.nebulaToolkitService.copyCollectionByWhiteList(competences, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newHashSet(competenceVos);
  }

  /**
   * 根据用户id查询用户绑定的接口信息
   * @param userId
   * @return
   */
  public Set<CompetenceVo> findByUserId(String userId){
    if(StringUtils.isBlank(userId)){
      return Sets.newHashSet();
    }
    Set<CompetenceVo> competenceVos = new HashSet<>();
    Set<ButtonVo> buttonVos = this.buttonService.findByUserId(userId);
    for(ButtonVo button : buttonVos){
      Set<CompetenceVo> competences = this.findByButtonId(button.getId());
      if(!CollectionUtils.isEmpty(competences)){
        competenceVos.addAll(competences);
      }
    }
    return competenceVos;
  }

  @Override
  public CompetenceVo findByCommentAndViewItemAndParent(String comment, int viewItem, String parentId) {
    if(StringUtils.isAnyBlank(comment, parentId)) {
      return null;
    }
    CompetenceEntity competence = competenceRepository.findByCommentAndViewItemAndParent(comment, viewItem, parentId);
    return competence == null ? null : nebulaToolkitService.copyObjectByWhiteList(competence, CompetenceVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public CompetenceVo findByCommentAndViewItemAndNullParent(String comment, int viewItem) {
    if(StringUtils.isBlank(comment)) {
      return null;
    }
    CompetenceEntity competence = competenceRepository.findByCommentAndViewItemAndNullParent(comment, viewItem);
    return competence == null ? null : nebulaToolkitService.copyObjectByWhiteList(competence, CompetenceVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public CompetenceVo findByResourceAndMethods(String resource, String methods) {
    if(StringUtils.isAnyBlank(resource, methods)) {
      return null;
    }
    CompetenceEntity competence = competenceRepository.findByResourceAndMethods(resource, methods);
    return competence == null ? null : nebulaToolkitService.copyObjectByWhiteList(competence, CompetenceVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public long count() {
    return competenceRepository.count();
  }

  /**
   * 菜单条件查询（不分页,只针对菜单）
   * @param competence
   * @return
   */
  @Override
  public List<CompetenceVo> findAllByConditions(CompetenceVo competence) {
    List<CompetenceEntity> competences = competenceRepository.findAllByConditions(competence);
    if(CollectionUtils.isEmpty(competences)){
      return Lists.newArrayList();
    }
    Collection<CompetenceVo> competenceVos = this.nebulaToolkitService.copyCollectionByWhiteList(competences, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class, "parent");
    return Lists.newArrayList(competenceVos);
  }

  @Override
  public long countByViewItem(int viewItem) {
    return competenceRepository.countByViewItem(viewItem);
  }
}