package com.bizunited.platform.rbac.server.starter.controller;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

@RestController
@RequestMapping("/v1/nebula/roles")
public class RoleController extends BaseController {

  @Autowired
  private RoleService roleService;
  /**
   * 配置的那些可以忽略方法级检查的，具有超级管理员性质的角色名
   */
  @Value("${rbac.ignoreMethodCheckRoles:ADMIN}")
  private String[] ignoreMethodCheckRoles;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RoleController.class);

  /**
   * 添加一个角色信息
   * 
   * @param role
   * @return 创建后的角色基本信息将会被返回
   */
  @ApiOperation(value = "添加一个角色信息")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel addRole(@RequestBody @ApiParam(value = "role" , name="新建的角色对象") RoleVo role) {
    try {
      if (role == null) {
        throw new IllegalArgumentException("role info not be null!");
      }
      RoleVo currentRole = roleService.create(role);
      return this.buildHttpResultW(currentRole);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 修改一个指定的角色信息，注意配置在roles.deleteDeny属性的信息不能进行修改操作<br>
   * 且指定的一个角色只能修改角色的comment信息
   *
   * @param role 指定的修改信息
   */
  @ApiOperation(value = "修改一个指定的角色信息，注意配置在roles.sDeny属性的信息不能进行修改操作。且指定的一个角色只能修改角色的comment信息")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel updateRole(@RequestBody @ApiParam(value = "role" , name="修改的角色对象") RoleVo role) {
    try {
      RoleVo currentRole = this.roleService.update(role);
      return this.buildHttpResultW(currentRole);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 重新启用某一个指定的角色信息
   * 
   * @param roleId
   * @return
   */
  @ApiOperation(value = "重新启用某一个指定的角色信息")
  @RequestMapping(value = "/enable/{roleId}", method = RequestMethod.PATCH)
  public ResponseModel enableRole(@PathVariable("roleId") String roleId) {
    try {
      this.roleService.enable(roleId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 禁用某一个指定的角色信息（相当于删除）<br>
   * 只是系统中不能真正的删除某一个角色，只能是将这个角色作废掉或者恢复正常状态
   * 
   * @param roleId
   * @return
   */
  @ApiOperation(value = "禁用某一个指定的角色信息（相当于删除）<br>" + "只是系统中不能真正的删除某一个角色，只能是将这个角色作废掉或者恢复正常状态")
  @RequestMapping(value = "/disable/{roleId}", method = RequestMethod.PATCH)
  public ResponseModel disableRole(@PathVariable("roleId") String roleId) {
    try {
      this.roleService.disable(roleId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据组织机构编码查询绑定的角色
   * @param orgCode
   * @return
   */
  @GetMapping("findByOrgCode")
  @ApiOperation("根据组织机构编码查询绑定的角色")
  public ResponseModel findByOrgCode(@RequestParam @ApiParam("组织机构编码") String orgCode) {
    try {
      List<RoleVo> roles = roleService.findByOrgCode(orgCode);
      return this.buildHttpResultW(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  /**
   * 根据岗位编号查询直接关联的角色
   * @param positionCode
   * @return
   */
  @ApiOperation(value = "根据岗位编号查询直接关联的角色")
  @GetMapping("/findByPositionCode")
  public ResponseModel findByPositionCode(@ApiParam(value = "岗位编号") @RequestParam String positionCode) {
    try {
      Set<RoleVo> result = roleService.findByPositionCode(positionCode);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  /**
   * 根据用户组编码查询绑定的角色
   * @param userGroupCode
   * @return
   */
  @ApiOperation("根据用户组编码查询绑定的角色")
  @GetMapping("/findByGroupCode")
  public ResponseModel findByGroupCode(@ApiParam(name = "userGroupCode", value = "用户组编码") @RequestParam("userGroupCode") String userGroupCode){
    try {
      List<RoleVo> roles = this.roleService.findByGroupCode(userGroupCode);
      return this.buildHttpResult(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 该接口方法用于查询符合指定状态的角色信息，只返回角色的基本信息，没有任何的关联信息但是包括了可能存在的修改者信息
   * 
   * @param status
   * @return
   */
  @ApiOperation(value = "该接口方法用于查询符合指定状态的角色信息，只返回角色的基本信息，没有任何的关联信息但是包括了可能存在的修改者信息。")
  @RequestMapping(value = "/findByStatus", method = RequestMethod.GET)
  public ResponseModel findByStatus(Integer status) {
    try {
      List<RoleVo> roles = this.roleService.findByStatus(status);
      return this.buildHttpResultW(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询一个指定的角色信息，只查询这个角色的基本信息
   */
  @ApiOperation(value = "查询一个指定的角色信息，只查询这个角色的基本信息")
  @RequestMapping(value = "/find/{roleId}", method = RequestMethod.GET)
  public ResponseModel findById(@PathVariable("roleId") String roleId) {
    try {
      RoleVo currentRole = this.roleService.findById(roleId);
      return this.buildHttpResultW(currentRole);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取指定用户已绑定的角色信息
   * @param userId 用户id
   * @return List<RoleEntity>
   */
  @ApiOperation(value = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等",
      notes = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等," +
          "因为有主副岗位之分，在查询用户的角色用于权限验证时，岗位绑定的角色只会查询主岗位绑定的角色，用户角色维护时，则返回用户所有岗位绑定的角色")
  @RequestMapping(value = "/findAllByUserId", method = {RequestMethod.GET})
  public ResponseModel findAllByUserId(@ApiParam(value = "指定的人员数据编号") @RequestParam String userId,
                                    @RequestParam(defaultValue = "0") @ApiParam(value = "查询类型，不传默认为0，0：只是查询用户的角色用于权限验证，1：查询用户的角色用于角色维护") Integer type) {
    try {
      List<RoleVo> nowList = roleService.findAllByUserId(userId, type);
      return this.buildHttpResult(nowList);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据用户ID查询用户直接关联的角色
   * @param userId 
   * @return
   */
  @ApiOperation(value = "根据用户ID查询用户直接关联的角色，不包括角色的任何关联信息")
  @GetMapping("/findByUserId")
  public ResponseModel findByUserId(@ApiParam(value = "指定的人员数据编号") @RequestParam String userId) {
    try {
      Set<RoleVo> roleSet = roleService.findByUserId(userId);
      return this.buildHttpResultW(roleSet , new String[] {});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据用户account账户信息查询用户直接关联的角色
   * @param userId
   * @return
   */
  @ApiOperation(value = "根据用户account账户信息查询用户直接关联的角色，不包括角色的任何关联信息")
  @GetMapping("/findByUserAccount")
  public ResponseModel findByUserAccount(@ApiParam(value = "指定的人员account账户信息") @RequestParam String userId) {
    try {
      Set<RoleVo> roleSet = roleService.findByUserId(userId);
      return this.buildHttpResultW(roleSet, new String[] {});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "该方法给定一批角色名，系统会返回这批角色名中，是否有一个或者多个角色拥有管理员性质的角色。而判定具有超级管理员性质角色的用户会被返回",
      notes = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等")
  @RequestMapping(value = "/findByIgnoreRoles", method = {RequestMethod.GET})
  public ResponseModel findByIgnoreRoles(@ApiParam(value = "当前需要验证的是否具有本地超级管理员") @RequestParam(name="roleNames") String[] roleNames) {
    try{
      Validate.isTrue(roleNames != null && roleNames.length > 0 , "要进行判定的角色信息必须传入");
      SetView<String> intersections = Sets.intersection(Sets.newHashSet(ignoreMethodCheckRoles), Sets.newHashSet(roleNames));
      
      if(intersections == null || intersections.isEmpty()) {
        return this.buildHttpResult();
      }
      // 组装成json返回
      JSONArray results = new JSONArray();
      for (String item : intersections) {
        results.add(item);
      }
      return this.buildHttpResult(results);
    } catch(RuntimeException e) {
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 该接口方法用于查询所有角色信息，无论它们的状态如何。<br>
   * 只返回角色的基本信息，没有任何的关联信息但是包括了可能的修改者信息"
   * 
   */
  @ApiOperation(value = "该接口方法用于查询所有角色信息，无论它们的状态如何(只推荐的管理端使用该接口，在业务端不推荐)。<br>" + "只返回角色的基本信息，没有任何的关联信息但是包括了可能的修改者信息")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      List<RoleVo> roles = this.roleService.findAll();
      return this.buildHttpResultW(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询多个角色信息")
  @RequestMapping(value = "/findByIds", method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids) {
    try {
      Set<RoleVo> roles = this.roleService.findByIds(ids);
      return this.buildHttpResultW(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 角色管理-查询角色树状关系
   * @return
   */
  @ApiOperation(value = "查询角色树状关系")
  @GetMapping(value = "/findRoleTree")
  public ResponseModel findRoleTree() {
    try {
      Set<RoleVo> roles = this.roleService.findRoleTree();
      return this.buildHttpResult(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色名称，或则角色编码，或则角色编码查询角色信息 （角色名称支持模糊查询）
   * @param roleName
   * @return
   */
  @ApiOperation(value = "根据角色名称，或则角色编码，或则角色编码查询角色信息 （角色名称支持模糊查询）")
  @GetMapping(value = "/findByConditions")
  public ResponseModel findByConditions(@ApiParam(value = "角色名称") @RequestParam(name="roleName", required = false) String roleName,
                                      @ApiParam(value = "角色编码") @RequestParam(name="roleCode", required = false) String roleCode,
                                      @ApiParam(value = "角色状态") @RequestParam(name="tstatus", required = false) Integer tstatus) {
    try {
      Map<String, Object> conditions = new HashMap<>();
      if (StringUtils.isNotBlank(roleName)) {
        conditions.put("roleName", roleName);
      }
      if (StringUtils.isNotBlank(roleCode)) {
        conditions.put("roleCode", roleCode);
      }
      if (tstatus != null) {
        conditions.put("tstatus", tstatus);
      }
      List<RoleVo> roles = this.roleService.findByConditions(conditions);
      return this.buildHttpResultW(roles, "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色编码查询
   * @param roleCode
   * @return
   */
  @ApiOperation(value = "根据角色编码查询")
  @GetMapping(value = "/findByCode")
  public ResponseModel findByCode(String roleCode) {
    try {
      RoleVo role = this.roleService.findByCode(roleCode);
      return this.buildHttpResultW(role);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色编码删除（只能删除无绑定关系的角色）
   * @param roleCodes
   * @return
   */
  @ApiOperation(value = "根据角色编码删除（只能删除无绑定关系的角色）")
  @GetMapping(value = "/deleteByCodes")
  public ResponseModel deleteByCodes(
          @ApiParam(value = "角色编码数组", required = true) @RequestParam(name="roleCodes", required = true)String[] roleCodes) {
    try {
      roleService.deleteByCodes(roleCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  @ApiOperation(value = "根据当前人信息，判断当前人是否具有管理员角色")
  @GetMapping(value = "/hasAdminByCurrentUser")
  public ResponseModel hasAdminByCurrentUser(){
    try {
      Boolean result = roleService.hasAdminByCurrentUser();
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
