package com.bizunited.platform.rbac.server.starter.service.internal;

import java.util.List;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.rbac.server.service.RbacSettingService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.RoleUserGroupMappingService;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.entity.RoleUserGroupMappingEntity;
import com.bizunited.platform.rbac.server.starter.repository.RoleUserGroupMappingRepository;
import com.bizunited.platform.rbac.server.vo.RbacSettingVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.userGroup.UserGroupService;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import com.google.common.collect.Lists;

public class RoleUserGroupMappingServiceImpl implements RoleUserGroupMappingService {
  @Autowired
  private RbacSettingService rbacSettingService;
  @Autowired
  private RoleUserGroupMappingRepository roleUserGroupMappingRepository;
  @Autowired
  private UserGroupService userGroupService;
  @Autowired
  private RoleService roleService;
  
  @Override
  @Transactional
  public void bindUserGroups(String roleId, String[] userGroupCodes) {
    Validate.notBlank(roleId, "进行用户组-角色绑定时，必须指定当前的角色信息!!");
    Validate.isTrue(userGroupCodes != null && userGroupCodes.length > 0, "进行用户组-角色绑定时,至少传入一个组织机构编号信息!!");
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到角色信息");
    RbacSettingVo rbacSettingVo = rbacSettingService.findRbacSetting();
    for (String code : userGroupCodes) {
      UserGroupVo userGroupVo = userGroupService.findByCode(code);
      Validate.notNull(userGroupCodes, "未找到用户组：%s", code);
      this.validateBindRole(rbacSettingVo, role, userGroupVo);
      RoleUserGroupMappingEntity entity = new RoleUserGroupMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      entity.setRole(currentRole);
      entity.setGroupCode(code);
      roleUserGroupMappingRepository.save(entity);
    }
  }

  @Override
  @Transactional
  public void unbindUserGroups(String roleId, String[] userGroupCodes) {
    Validate.notBlank(roleId, "进行用户组-角色解绑时，必须指定当前的角色信息!!");
    Validate.isTrue(userGroupCodes != null && userGroupCodes.length > 0, "进行用户组-角色解绑时,至少传入一个用户组编号信息!!");
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到角色信息");
    for (String code : userGroupCodes) {
      RoleUserGroupMappingEntity entity = roleUserGroupMappingRepository.findByRoleAndGroupCode(role.getRoleCode(), code);
      if(entity != null) {
        roleUserGroupMappingRepository.delete(entity);
      }
    }
  }

  /**
   * 给指定id的角色重新进行用户组绑定，按照最新给定的用户组进行绑定
   * @param roleId
   * @param userGroupCodes
   */
  @Override
  @Transactional
  public void reBindUserGroup(String roleId, String[] userGroupCodes) {
    Validate.notBlank(roleId, "角色绑定用户组时，角色编码不能为空");
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到对应角色信息，请检查");
    String currentUserGroupCodes[] = userGroupCodes;
    if(currentUserGroupCodes == null){
      currentUserGroupCodes = new String[]{};
    }

    // 解绑所有绑定
    this.roleUserGroupMappingRepository.deleteByRoleCode(role.getRoleCode());
    

    // 重新绑定
    RbacSettingVo rbacSettingVo = rbacSettingService.findRbacSetting();
    for(String groupCode : currentUserGroupCodes){
      UserGroupVo userGroupVo = userGroupService.findByCode(groupCode);
      Validate.notNull(userGroupVo, "未找到用户组：%s", groupCode);
      this.validateBindRole(rbacSettingVo, role, userGroupVo);
      RoleUserGroupMappingEntity mapping = new RoleUserGroupMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setGroupCode(groupCode);
      roleUserGroupMappingRepository.save(mapping);
    }
  }
  
  /**
   * 验证绑定角色和分组
   * @param rbacSettingVo
   * @param role
   * @param userGroupVo
   */
  private void validateBindRole(RbacSettingVo rbacSettingVo, RoleVo role, UserGroupVo userGroupVo) {
    long count = roleUserGroupMappingRepository.countByRoleAndGroupCode(role.getId(), userGroupVo.getGroupCode());
    Validate.isTrue(count == 0, "在批量绑定时指定用户组[%s]已经绑定了，不能重复绑定，请检查!!", userGroupVo.getGroupName());
    int bindRelation = rbacSettingVo.getRoleGroupRelation();
    //如果用户和组织绑定关系为多对多,多对一，一对多，一对一分别控制判断
    if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
      count = roleUserGroupMappingRepository.countByRoleAndGroupCode(role.getRoleCode(), userGroupVo.getGroupCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和用户组多对多，指定组织机构[%s]已经绑定了，不能重复绑定，请检查!!", userGroupVo.getGroupName());
    }
    if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
      count = roleUserGroupMappingRepository.countByRole(role.getRoleCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和用户组多对一，指定角色[%s]已经绑定了用户组，不能继续绑定，请检查!!", role.getRoleName());
    }
    if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
      count = roleUserGroupMappingRepository.countByUserGroupCode(userGroupVo.getGroupCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和用户组一对多，指定用户组[%s]角色，不能继续绑定，请检查!!", userGroupVo.getGroupName());
    }
    if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
      count = roleUserGroupMappingRepository.countByRole(role.getRoleCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和用户组一对一，指定角色[%s]已经绑定了用户组，不能继续绑定，请检查!!", role.getRoleName());
      long countRole = roleUserGroupMappingRepository.countByUserGroupCode(userGroupVo.getGroupCode());
      Validate.isTrue(countRole == 0, "在批量绑定时,角色和用户组一对一，指定用户组[%s]角色，不能继续绑定，请检查!!", userGroupVo.getGroupName());
    }

  }

  @Override
  @Transactional
  public void bindUserGroupRoles(String userGroupCode, String[] roleIds) {
    Validate.notBlank(userGroupCode, "进行用户组-角色绑定时，必须指定当前用户组信息!!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行用户组-角色绑定时,至少传入一个角色id信息!!");
    UserGroupVo userGroupVo = userGroupService.findByCode(userGroupCode);
    Validate.notNull(userGroupVo, "未找到用户组: %s", userGroupCode);
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    
    for (String roleId : roleIds) {
      RoleVo role = this.roleService.findById(roleId);
      Validate.notNull(role, "未找到角色信息：%s", roleId);
      this.validateBindRole(rbacSetting, role, userGroupVo);
      RoleUserGroupMappingEntity mapping = new RoleUserGroupMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(role.getId());
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setGroupCode(userGroupCode);
    }
  }

  @Override
  @Transactional
  public void unbindUserGroupRoles(String userGroupCode, String[] roleIds) {
    Validate.notBlank(userGroupCode, "进行用户组-角色解绑时，必须指定当前的用户组信息!!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行用户组-角色绑定时,至少传入一个角色id信息!!");
    for (String roleId : roleIds) {
      RoleVo role = this.roleService.findById(roleId);
      Validate.notNull(role, "未找到角色信息：%s", roleId);
      RoleUserGroupMappingEntity mapping = roleUserGroupMappingRepository.findByRoleAndGroupCode(role.getRoleCode(), userGroupCode);
      if (mapping != null) {
        roleUserGroupMappingRepository.delete(mapping);
      }
    }
  }

  /**
   * 给指定编码的用户组重新进行角色绑定，按照最新给定的角色进行绑定
   * @param userGroupCode
   * @param roleCodes
   */
  @Override
  @Transactional
  public void reBindUserGroupRoles(String userGroupCode, String[] roleCodes) {
    Validate.notBlank(userGroupCode, "用户组重新进行角色绑定，用户组编码不能为空");
    UserGroupVo userGroupVo = userGroupService.findByCode(userGroupCode);
    Validate.notNull(userGroupVo, "未找到对应用户组信息，请检查");
    String[] currentRoleCodes = roleCodes;
    if(currentRoleCodes == null){
      currentRoleCodes = new String[]{};
    }

    // 解除所有绑定
    this.roleUserGroupMappingRepository.deleteByGroupCode(userGroupVo.getGroupCode());
    
    // 重新绑定
    RbacSettingVo rbacSettingVo = rbacSettingService.findRbacSetting();
    for(String roleCode : currentRoleCodes){
      RoleVo role = roleService.findByCode(roleCode);
      Validate.notNull(role, "未找到编码为[%s]角色,请检查!!", roleCode);
      this.validateBindRole(rbacSettingVo, role, userGroupVo);
      RoleUserGroupMappingEntity mapping = new RoleUserGroupMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(role.getId());
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setGroupCode(userGroupCode);
      roleUserGroupMappingRepository.save(mapping);
    }
  }

  /**
   * 根据角色编码查询绑定用户组
   * @param roleCode
   * @return
   */
  @Override
  public List<UserGroupVo> findByRoleCode(String roleCode) {
    if(StringUtils.isBlank(roleCode)){
      return Lists.newArrayList();
    }
    List<String> groupCodes = roleUserGroupMappingRepository.findGroupCodesByRoleCode(roleCode);
    List<UserGroupVo> userGroups = userGroupService.findByGroupCodes(groupCodes);
    if(CollectionUtils.isEmpty(userGroups)){
      return Lists.newArrayList();
    }
    return userGroups;
  }
}