package com.bizunited.platform.rbac.server.starter.service.internal;

import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.rbac.server.service.RbacSettingService;
import com.bizunited.platform.rbac.server.service.RoleOrgMappingService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.entity.RoleOrgMappingEntity;
import com.bizunited.platform.rbac.server.starter.repository.RoleOrgMappingRepository;
import com.bizunited.platform.rbac.server.vo.RbacSettingVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public class RoleOrgMappingServiceImpl implements RoleOrgMappingService {
  @Autowired
  private RbacSettingService rbacSettingService;
  @Autowired
  private RoleOrgMappingRepository roleOrgMappingRepository;
  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private RoleService roleService;
  
  @Override
  @Transactional
  public void bindOrgs(String roleId, String[] orgCodes) {
    Validate.notBlank(roleId, "进行组织机构-角色绑定时，必须指定当前的角色信息!");
    Validate.notEmpty(orgCodes, "进行组织机构-角色绑定时,至少传入一个组织机构编号信息!");
    RoleVo role = this.roleService.findById(roleId);
    
    Validate.notNull(role, "未找到角色信息");
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    for (String orgCode : orgCodes) {
      OrganizationVo org = organizationService.findByCode(orgCode);
      Validate.notNull(org, "未找到组织机构：%s", orgCode);
      this.validateBindOrg(rbacSetting, role, org);
      RoleOrgMappingEntity mapping = new RoleOrgMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setOrgCode(orgCode);
      mapping.setProjectName(ApplicationContextUtils.getProjectName());
      roleOrgMappingRepository.save(mapping);
    }
  }

  @Override
  @Transactional
  public void rebindOrgs(String roleId, String[] orgCodes) {
    Validate.notBlank(roleId, "进行组织机构-角色绑定时，必须指定当前的角色信息!");
    String currentOrgCodes[] = orgCodes;
    if(currentOrgCodes == null) {
      currentOrgCodes = new String[0];
    }
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到角色信息");
    
    // 删除指定角色和组织机构原有的绑定
    this.roleOrgMappingRepository.deleteByRoleCodeAndProjectName(role.getRoleCode(), ApplicationContextUtils.getProjectName());
    
    // 重新进行绑定
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    for (String orgCode : currentOrgCodes) {
      OrganizationVo org = organizationService.findByCode(orgCode);
      Validate.notNull(org, "未找到组织机构：%s", orgCode);
      this.validateBindOrg(rbacSetting, role, org);
      RoleOrgMappingEntity mapping = new RoleOrgMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setOrgCode(orgCode);
      mapping.setProjectName(ApplicationContextUtils.getProjectName());
      roleOrgMappingRepository.save(mapping);
    }
  }


  @Override
  @Transactional
  public void unbindOrgs(String roleId, String[] orgCodes) {
    Validate.notBlank(roleId, "进行组织机构-角色解绑时，必须指定当前的角色信息!");
    Validate.isTrue(orgCodes != null && orgCodes.length > 0, "进行组织机构-角色解绑时,至少传入一个组织机构编号信息!");
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到角色信息:%s", roleId);
    for (String orgCode : orgCodes) {
      RoleOrgMappingEntity mapping = roleOrgMappingRepository.findByRoleAndOrgCodeAndProjectName(role.getId(), orgCode, ApplicationContextUtils.getProjectName());
      if(mapping != null) {
        roleOrgMappingRepository.delete(mapping);
      }
    }
  }
  
  /**
   * 验证角色与组织机构的绑定, 不仅验证组织机构是否存在，同时验证绑定后是否满足预设的关联关系
   * @param rbacSetting
   * @param role
   * @param org
   */
  private void validateBindOrg(RbacSettingVo rbacSetting, RoleVo role, OrganizationVo org) {
    Validate.notNull(org, "未找到对应组织机构");
    Validate.notNull(role, "未找到对应角色");
    Validate.notBlank(role.getId(), "未找到对应角色ID");
    String orgCode = org.getCode();
    long count = roleOrgMappingRepository.countByRoleAndOrgCodeAndProjectName(role.getRoleCode(), orgCode, ApplicationContextUtils.getProjectName());
    Validate.isTrue(count == 0, "在批量绑定时,指定组织机构[%s]已经绑定了，不能重复绑定，请检查!!", org.getOrgName());
    int bindRelation = rbacSetting.getRoleOrgRelation();
    //如果用户和组织绑定关系为多对多,多对一，一对多，一对一分别控制判断
    if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
      count = roleOrgMappingRepository.countByRoleAndOrgCodeAndProjectName(role.getRoleCode(), orgCode, ApplicationContextUtils.getProjectName());
      Validate.isTrue(count == 0, "在批量绑定时,角色和组织多对多，指定组织机构[%s]已经绑定了，不能重复绑定，请检查!", org.getOrgName());
    }
    if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
      count = roleOrgMappingRepository.countByRoleAndProjectName(role.getRoleCode(), ApplicationContextUtils.getProjectName());
      Validate.isTrue(count == 0, "在批量绑定时,角色和组织多对一，指定角色[%s]已经绑定了组织，不能继续绑定，请检查!", role.getRoleName());
    }
    if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
      count = roleOrgMappingRepository.countByOrgCodeAndProjectName(orgCode, ApplicationContextUtils.getProjectName());
      Validate.isTrue(count == 0, "在批量绑定时,角色和组织一对多，指定组织[%s]已绑定角色角色，不能继续绑定，请检查!", org.getOrgName());
    }
    if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
      long countOrganization = roleOrgMappingRepository.countByRoleAndProjectName(role.getRoleCode(), ApplicationContextUtils.getProjectName());
      Validate.isTrue(countOrganization == 0, "在批量绑定时,角色和组织一对一，指定角色[%s]已经绑定了组织，不能继续绑定，请检查!", role.getRoleName());
      long countRole = roleOrgMappingRepository.countByOrgCodeAndProjectName(orgCode, ApplicationContextUtils.getProjectName());
      Validate.isTrue(countRole == 0, "在批量绑定时,角色和组织一对一，指定组织[%s]与角色不能继续绑定，请检查!", org.getOrgName());
    }
  }

  @Override
  @Transactional
  public void bindOrgRoles(String orgCode, String[] roleIds) {
    Validate.notBlank(orgCode, "进行组织机构-角色绑定时，必须指定当前的组织机构信息!");
    Validate.notEmpty(roleIds, "进行组织机构-角色绑定时,至少传入一个角色ID信息!");
    OrganizationVo org = organizationService.findByCode(orgCode);
    Validate.notNull(org, "未找到组织机构：%s", orgCode);
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    for (String roleId : roleIds) {
      RoleVo role = this.roleService.findById(roleId);
      Validate.notNull("未找到角色：%s", roleId);
      this.validateBindOrg(rbacSetting, role, org);
      RoleOrgMappingEntity mapping = new RoleOrgMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setOrgCode(orgCode);
      roleOrgMappingRepository.save(mapping);
    }
  }

  @Override
  @Transactional
  public void rebindOrgRoles(String orgCode, String[] roleIds) {
    Validate.notBlank(orgCode, "进行组织机构-角色绑定时，必须指定当前的组织机构信息!");
    String currentRoleIds[] = roleIds;
    if(currentRoleIds == null) {
      currentRoleIds = new String[0];
    }
    OrganizationVo org = organizationService.findByCode(orgCode);
    Validate.notNull(org, "未找到组织机构：%s", orgCode);
    
    // 删除这个组织机构下原来所有的角色信息
    roleOrgMappingRepository.deleteByOrgCodeAndProjectName(orgCode, ApplicationContextUtils.getProjectName());
    
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    for (String roleId : currentRoleIds) {
      RoleVo role = this.roleService.findById(roleId);
      Validate.notNull("未找到角色：%s", roleId);
      this.validateBindOrg(rbacSetting, role, org);
      RoleOrgMappingEntity mapping = new RoleOrgMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setOrgCode(orgCode);
      mapping.setProjectName(ApplicationContextUtils.getProjectName());
      roleOrgMappingRepository.save(mapping);
    }
  }

  @Override
  @Transactional
  public void unbindOrgRoles(String orgCode, String[] roleIds) {
    Validate.notBlank(orgCode, "进行组织机构-角色解绑时，必须指定当前的组织机构信息!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行组织机构-角色解绑时,至少传入一个角色ID信息!");
    OrganizationVo org = organizationService.findByCode(orgCode);
    Validate.notNull(org, "未找到组织机构信息：%s", orgCode);
    Set<RoleOrgMappingEntity> mappings = roleOrgMappingRepository.findByOrgCodeAndRoleIds(orgCode, roleIds);
    if(!CollectionUtils.isEmpty(mappings)) {
      roleOrgMappingRepository.deleteAll(mappings);
    }
  }

  // TODO 进行优化处理
  @Override
  public List<OrganizationVo> findByRoleId(String roleId) {
    if(StringUtils.isBlank(roleId)) {
      return Lists.newArrayList();
    }
    RoleVo role = this.roleService.findById(roleId);
    if(role  == null) {
      return Lists.newArrayList();
    }
    Set<RoleOrgMappingEntity> roleOrgs = roleOrgMappingRepository.findByRoleAndProjectName(role.getRoleCode(), ApplicationContextUtils.getProjectName());
    if(CollectionUtils.isEmpty(roleOrgs)) {
      return Lists.newArrayList();
    }
    List<OrganizationVo> orgs = new ArrayList<>();
    for (RoleOrgMappingEntity roleOrg : roleOrgs) {
      OrganizationVo org = organizationService.findByCode(roleOrg.getOrgCode());
      if(org != null) {
        orgs.add(org);
      }
    }
    return orgs;
  }

}
