package com.bizunited.platform.rbac.server.starter.controller;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.CompetenceService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.vo.CompetenceVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.security.Principal;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 功能服务相关接口.
 * @author yinwenjie
 * @version V1.0
 */
@RestController
@RequestMapping(value="/v1/nebula/competences")
public class CompetenceController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(CompetenceController.class);
  @Autowired
  private CompetenceService competenceService;
  @Autowired
  private UserService userService;
  @Autowired
  private RoleService roleService;
  
  @ApiOperation(value = "创建新功能，创建新功能是其URL信息必须是唯一的")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(@RequestBody @ApiParam(name = "comp", value = "需要新增的功能对象信息 ") CompetenceVo comp) {
    try {
      CompetenceVo competence = this.competenceService.create(comp);
      return this.buildHttpResultW(competence);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "修改功能信息，只能修改对象中的基本信息（目前只能编辑url，功能中文说明）。")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(@RequestBody @ApiParam(name = "comp", value = "需要修改的功能对象信息") CompetenceVo comp) {
    try {
      CompetenceVo competence = this.competenceService.update(comp);
      return this.buildHttpResultW(competence);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "更改功能状态，以便实现功能的启用/禁用功能，注意，一旦父级功能的状态发生变更，那么其所有子级状态（多级）都会发生变化")
  @RequestMapping(value = "/updateStatus", method = RequestMethod.PATCH)
  public ResponseModel  updateStatus(@RequestParam(name="id") @ApiParam(name = "id", value = "功能信息id") String id
                                    , @RequestParam(name="flag") @ApiParam(name = "flag", value = "flag 标识（true：启用操作；false：禁用操作）") Boolean flag) {
    try {
      CompetenceVo competence = this.competenceService.updateStatus(id, flag);
      return this.buildHttpResultW(competence);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "建立指定角色和指定功能的绑定关系")
  @PatchMapping(value = "/bindCompetence")
  public ResponseModel bindCompetence(@RequestParam(name="roleId") @ApiParam(name = "roleId", value = "指定的角色编号信息") String roleId ,
                                      @RequestParam(name="competenceIds") @ApiParam(name = "competenceIds", value = "指定的功能编号信息(可一个是一个或者多个)") String[] competenceIds) {
    try {
      this.competenceService.bindCompetence(roleId, competenceIds);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "取消指定角色和指定功能的绑定关系")
  @RequestMapping(value = "/unbindCompetence", method = RequestMethod.PATCH)
  public ResponseModel unbindCompetence(@RequestParam(name="roleId") @ApiParam(name = "roleId", value = "指定的角色编号信息") String roleId ,
                                        @RequestParam(name="competenceIds") @ApiParam(name = "competenceIds", value = "指定的功能编号信息(可一个是一个或者多个)") String[] competenceIds) {
    try {
      this.competenceService.unbindCompetence(roleId, competenceIds);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "删除指定的功能" , notes="如果当前功能存在下级关联功能，则不能进行删除")
  @RequestMapping(value = "/{competenceId}", method = RequestMethod.DELETE)
  public ResponseModel deleteById(@PathVariable(name="competenceId") @ApiParam(name = "competenceId", value = "指定的功能编号") String competenceId) {
    try {
      this.competenceService.deleteById(competenceId);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "")
  @RequestMapping(value = "/findByUrlResource", method = RequestMethod.GET)
  public ResponseModel findByUrlResource(@RequestParam(name="resources") @ApiParam(name = "resources", value = "") String[] resources ,
                                         HttpServletRequest request) {    
    try {
      Principal userPrincipal = this.getPrincipal();
      JSONArray results = this.competenceService.findByUrlResource(resources, userPrincipal);
      return this.buildHttpResult(results);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "根据功能中的viewItem属性， 查询满足条件的所有功能信息（按照sortIndex进行排序）")
  @RequestMapping(value = "/findByViewItem", method = RequestMethod.GET)
  public ResponseModel findByViewItem(@RequestParam(name="viewItem") @ApiParam(name = "viewItem", value = "是否是功能菜单上的功能栏目1：需要显示的；0：不需要显示的") Boolean viewItem) {
    try {
      JSONArray results = this.competenceService.findByViewItem(viewItem);
      return this.buildHttpResult(results);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "根据功能中的viewItem属性、状态属性， 查询满足条件的所有功能信息（按照sortIndex进行排序）")
  @RequestMapping(value = "/findByViewItemAndStatus", method = RequestMethod.GET)
  public ResponseModel findByViewItemAndStatus(@RequestParam(name="viewItem") @ApiParam(name = "viewItem", value = "是否是功能菜单上的功能栏目1：需要显示的；0：不需要显示的") Boolean viewItem ,
                                           @RequestParam(name="tstatus") @ApiParam(name = "tstatus", value = "状态信息，1：正常；0：禁用") Integer tstatus) {
    try {
      JSONArray results = this.competenceService.findByViewItemAndStatus(viewItem, tstatus);
      return this.buildHttpResult(results);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = " 该方法按照当前操作者所拥有的一个或者多个角色信息的name信息，" +
      "1.对当前用户的角色名称与yml配置的管理员角色信息取交集，判定是否具有管理员角色,如果具有管理员权限，该接口直接返回约定的null" +
      "2.如果当前用户的角色没有包含管理员权限，那么就会查询当前用户所拥有的所有功能集" +
      "注：这里可能会存在当前用户还未分配功能权限的情况，如果在这里没有发现功能集competences有值，那么这里就会返回一个空的功能数组" +
      "3.查询指定父级功能下，能够被这些角色访问的viewItem为true的，处于正常状态的功能"
      , notes="如果没有传入parentId信息，则认为是针对根级功能做以上查询。并且只显示状态正确的那些功能")
  @RequestMapping(value = "/findByCurrentUser", method = RequestMethod.GET)
  public ResponseModel findByCurrentUser(@RequestParam(name="viewItem") @ApiParam(name = "viewItem", value = "是否是功能菜单上的功能栏目1：需要显示的；0：不需要显示的") Boolean viewItem) {
    try {
      UserVo currentUser = userService.findByAccount(this.getPrincipalAccount());
      Validate.notNull(currentUser , "未找到指定的用户信息，请检查!!");
      List<RoleVo> roles = this.roleService.findAllByUserId(currentUser.getId(), 0);
      Validate.notNull(roles, "当前用户未绑定任何角色属性信息，请检查!!");
      List<String> roleNames = roles.stream().map(RoleVo::getRoleName).collect(Collectors.toList());
      JSONArray competences = this.competenceService.findByViewItemAndRoleNamesAndStatus(viewItem, roleNames.toArray(new String[]{}), 1);
      return this.buildHttpResult(competences);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "查询指定的角色Name查询已绑定的功能信息，且这些功能状态符合查询的要求，" +
      "注：1.该接口会对当前用户的角色名称与yml配置的管理员角色信息取交集，判定是否具有管理员角色,如果具有管理员权限，该接口直接返回约定的null" +
      "2.如果当前用户的角色没有包含管理员权限，那么就会查询当前用户所拥有的所有功能集" +
      "注：这里可能会存在当前用户还未分配功能权限的情况，如果在这里没有发现功能集competences有值，那么这里就会返回一个空的功能数组" +
      "3.查询指定父级功能下，能够被这些角色访问的viewItem为true的，处于正常状态的功能"
  )
  @RequestMapping(value = "/findByRoleNamesAndStatus", method = RequestMethod.GET)
  public ResponseModel findByRoleNameAndStatus(@RequestParam(name="viewItem") @ApiParam(name = "viewItem", value = "是否是功能菜单上的功能栏目1：需要显示的；0：不需要显示的") Boolean viewItem ,
                                               @RequestParam(name="roleNames") @ApiParam(name = "roleNames", value = "指定的角色name（可以是多个），如是ADMIN、VISITOR") String[] roleNames ,
                                               @RequestParam(name="status" , required=false) @ApiParam(name = "status", value = "状态（1正常/0禁用），这个状态信息不用必须，如果不传入则表示无论功能处于何种状态，都满足查询要求" , required=false) Integer status) {
    try {
      JSONArray competences = this.competenceService.findByViewItemAndRoleNamesAndStatus(viewItem, roleNames, status);
      return this.buildHttpResult(competences);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "这里只准对功能信息中viewItem为false的信息进行分页查询处理（查询条件也只有功能名），且查询的结果，只有功能的基本信息")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(@RequestParam(name="comment" , required=false) @ApiParam(name = "comment", value = "可能存在的功能名（功能备注） " , required=false) String comment,
                                        @RequestParam(name="methods" , required=false) @ApiParam(name = "methods", value = "请求方式 " , required=false) String methods,
                                        @RequestParam(name="tstatus" , required=false) @ApiParam(name = "tstatus", value = "状态 " , required=false) Integer tstatus,
                                        @PageableDefault(value = 50) @ApiParam(name = "pageable", value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50") Pageable pageable) {
    try {
      CompetenceVo competence = new CompetenceVo();
      competence.setComment(comment);
      competence.setMethods(methods);
      competence.setTstatus(tstatus);
      Page<CompetenceVo> results = this.competenceService.findByConditions(competence, pageable);
      return this.buildHttpResultW(results);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据按钮id查询绑定的接口信息
   * @param buttonId
   * @return
   */
  @ApiOperation(value = "根据按钮id查询绑定的接口信息")
  @GetMapping("findByButtonId")
  public ResponseModel findByButtonId(@RequestParam(name = "buttonId") @ApiParam(name = "buttonId", value = "指定的按钮的编号信息") String buttonId ){
    try {
      Set<CompetenceVo> competenceVos = this.competenceService.findByButtonId(buttonId);
      return this.buildHttpResultW(competenceVos);
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据用户id查询用户绑定的接口信息
   * @param userId
   * @return
   */
  @ApiOperation(value = "根据用户id查询用户绑定的接口信息")
  @GetMapping("findByUserId")
  public ResponseModel findByUserId(@RequestParam(name = "userId") @ApiParam(name = "userId", value = "指定的用户的编号信息") String userId){
    try {
      Set<CompetenceVo> competenceVos = this.competenceService.findByUserId(userId);
      return this.buildHttpResultW(competenceVos);
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 菜单条件查询（不分页,只针对菜单）
   * @param comment
   * @param resource
   * @param tstatus
   * @return
   */
  @ApiOperation(value = "菜单条件查询")
  @GetMapping("findAllByConditions")
  public ResponseModel findAllByConditions(
          @RequestParam(name = "comment", required = false) @ApiParam(name = "comment", value = "菜单名称") String comment,
          @RequestParam(name = "resource", required = false) @ApiParam(name = "resource", value = "菜单类型") String resource,
          @RequestParam(name = "tstatus", required = false) @ApiParam(name = "tstatus", value = "状态") Integer tstatus){
    try {
      CompetenceVo competence = new CompetenceVo();
      competence.setComment(comment);
      competence.setResource(resource);
      competence.setTstatus(tstatus);
      List<CompetenceVo> competences = this.competenceService.findAllByConditions(competence);
      return this.buildHttpResult(competences);
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}