package com.bizunited.platform.rbac.server.starter.service.internal;

import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.rbac.server.service.RbacModelCodeGenerateService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.entity.RoleOrgMappingEntity;
import com.bizunited.platform.rbac.server.starter.entity.RolePositionMappingEntity;
import com.bizunited.platform.rbac.server.starter.repository.RoleOrgMappingRepository;
import com.bizunited.platform.rbac.server.starter.repository.RolePositionMappingRepository;
import com.bizunited.platform.rbac.server.starter.repository.RoleRepository;
import com.bizunited.platform.rbac.server.starter.repository.RoleUserGroupMappingRepository;
import com.bizunited.platform.rbac.server.starter.repository.RoleUserMappingRepository;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.service.userGroup.UserGroupService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.transaction.Transactional;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.util.CollectionUtils;

/**
 * @author yinwenjie
 */
public class RoleServiceImpl implements RoleService {

  /**
   * 角色repository自动注入.
   */
  @Autowired
  private RoleRepository roleRepository;


  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private UserService userService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private UserGroupService userGroupService;
  @Autowired
  private RolePositionMappingRepository rolePositionMappingRepository;
  @Autowired
  private RoleOrgMappingRepository roleOrgMappingRepository;
  @Autowired
  private RoleUserGroupMappingRepository roleUserGroupMappingRepository;
  @Autowired
  private RoleUserMappingRepository roleUserMappingRepository;
  @Autowired
  private RbacModelCodeGenerateService rbacModelCodeGenerateService;
  
  /**
   * 这个参数记录了那些可以忽略“功能”级别权限验证的角色
   */
  @Value("${rbac.ignoreMethodCheckRoles}")
  private String[] ignoreMethodCheckRoles; 
  /**
   * 配置的那些不允许被删除被作废的角色
   */
  @Value("${rbac.roles.deleteDeny}")
  private String[] deleteDenys;

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#create(com.bizunited.platform.rbac.server.vo.RoleVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name="RoleService.create" , desc="创建一个新的RoleVo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public RoleVo create(RoleVo role) {
    // 进入传入信息的检查
    this.validateRoleBeforeAdd(role);
    // 开始插入
    RoleEntity roleEntity = this.nebulaToolkitService.copyObjectByWhiteList(role, RoleEntity.class, HashSet.class, ArrayList.class, "parent");
    this.roleRepository.saveAndFlush(roleEntity);
    role.setId(roleEntity.getId());
    return role;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#update(com.bizunited.platform.rbac.server.vo.RoleVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name="RoleService.update" , desc="创建一个新的UserVo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public RoleVo update(RoleVo role) {
    /*
     * 修改角色的过程如下： 
     * 1、首先确定当前role是可以进行修改的role 
     * 2、只进行comment的修改，其它即使传递了也不进行修改
     */
    Validate.notNull(role, "角色信息不能为空");
    String roleId = role.getId();
    String updateComment = role.getComment();
    Validate.notEmpty(roleId, "角色不存在，请检查!");

    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "未找到该角色");
    RoleEntity currentRole = op.get();
    Validate.isTrue(!StringUtils.isAnyBlank(role.getRoleCode(),currentRole.getRoleCode()),"角色编码不能为空！");
    String dbRoleCode = currentRole.getRoleCode();
    if(!StringUtils.equals(dbRoleCode,role.getRoleCode())){
      RoleEntity roleEntity = roleRepository.findByRoleCode(role.getRoleCode());
      Validate.isTrue(roleEntity == null,"角色编码【%s】已经存在",role.getRoleCode());
    }
    // 1、========
    String currentName = currentRole.getRoleName();
    // 如果条件成立，说明这个角色信息不能被修改
    for (String deleteDeny : deleteDenys) {
      if (StringUtils.equals(currentName, deleteDeny)) {
        throw new AccessDeniedException("这个角色为系统特定默认角色，不允许修改！");
      }
    }

    // 2、========
    if(!StringUtils.isBlank(updateComment)){
      currentRole.setComment(updateComment);
    }

    if(role.getParent() != null){
      Validate.notBlank(role.getParent().getId(), "未找到该角色的父级角色的id，请检查！");
      RoleEntity parentRole = roleRepository.findById(role.getParent().getId()).orElse(null);
      Validate.notNull(parentRole, "未找到角色的父级角色");
      Validate.isTrue(!role.getId().equals(parentRole.getId()), "禁止将该角色本身设置为上级角色");
      currentRole.setParent(parentRole);
      Set<String> roleStack = new HashSet<>();
      roleStack.add(role.getParent().getId());
      this.handleCircular(currentRole, roleStack);
    }else {
      currentRole.setParent(null);
    }
    currentRole.setRoleName(role.getRoleName());
    currentRole.setRoleType(role.getRoleType());
    currentRole.setIsDeny(role.getIsDeny());
    //修改扩展字段
    currentRole.setExtend1(role.getExtend1());
    currentRole.setExtend2(role.getExtend2());
    currentRole.setExtend3(role.getExtend3());
    currentRole.setExtend4(role.getExtend4());
    currentRole.setExtend5(role.getExtend5());
    currentRole.setExtend6(role.getExtend6());
    currentRole.setExtend7(role.getExtend7());
    currentRole.setExtend8(role.getExtend8());
    currentRole.setExtend9(role.getExtend9());
    currentRole.setExtend10(role.getExtend10());
    this.roleRepository.saveAndFlush(currentRole);
    return role;
  }

  /**
   *判断是否形成循环依赖
   * @param parent
   * @param roleStack
   */
  private void handleCircular(RoleEntity parent, Set<String> roleStack) {
    if(CollectionUtils.isEmpty(parent.getChildren())) {
      return;
    }
    for (RoleEntity roleEntity : parent.getChildren()) {
      Validate.isTrue(!roleStack.contains(roleEntity.getId()), "形成循环依赖，更新失败，请检查！");
      roleStack.add(roleEntity.getId());
      this.handleCircular(roleEntity, roleStack);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#disable(java.lang.String)
   */
  @Override
  @Transactional
  public void disable(String[] roleIds) {
    /*
     * 注意：在配置文件中已经设定的那些不能操作的角色信息，是不允许禁用的
     */
    Validate.notEmpty(roleIds, "角色id不能为空!");
    for(String roleId : roleIds) {
      RoleEntity currentRole = this.roleRepository.findById(roleId).orElse(null);
      Validate.notNull(currentRole, "未找到指定的角色信息，请检查!!");
      // 如果条件成立，说明这个角色信息不能被删除（或者说作废）
      for (String deleteDeny : deleteDenys) {
        if (StringUtils.equals(currentRole.getRoleName(), deleteDeny)) {
          throw new AccessDeniedException("该角色不能禁用！");
        }
      }
      // 禁用角色
      currentRole.setTstatus(0);
      this.roleRepository.save(currentRole);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#enable(java.lang.String)
   */
  @Override
  @Transactional
  public void enable(String[] roleIds) {
    Validate.notEmpty(roleIds, "角色id不能为空!");
    for(String roleId : roleIds) {
      RoleEntity currentRole = this.roleRepository.findById(roleId).orElse(null);
      Validate.notNull(currentRole, "未找到指定的角色信息，请检查!!");
      // 启用角色
      currentRole.setTstatus(1);
      this.roleRepository.save(currentRole);
    }
  }
  /**
   * 该私有方法在新增一个role信息前，检查传入信息的证确定   * @param role
   */
  private void validateRoleBeforeAdd(RoleVo role) {
    Validate.notNull(role, "角色信息不能为空!");
    // 开始验证
    Validate.notBlank(role.getRoleName(), "角色名称不能为空!");
    if(StringUtils.isBlank(role.getRoleCode())) {
      role.setRoleCode(rbacModelCodeGenerateService.roleCodeGenerate());
    }
    // code必须验证
    RoleEntity oldRole = this.roleRepository.findByRoleCode(role.getRoleCode());
    Validate.isTrue(oldRole == null, "当前设定的角色编号（role code）已经被使用，请更换!");
    // 当前的创建时间和修改时间要一起写入
    Date currentTime = new Date();
    role.setCreateDate(currentTime);
    // 说明性信息
    if(StringUtils.isBlank(role.getComment())) {
      role.setComment("[未填写]");
    }
    // 当前角色必须是状态正常的
    Validate.isTrue(role.getTstatus() == null || role.getTstatus() == 1 ,"当前角色必须是状态正常的");
    role.setTstatus(1);
    if(role.getParent() != null){
      Validate.notBlank(role.getParent().getId(), "未找到该角色的父级角色的id，请检查！");
      RoleEntity parentRole = roleRepository.findById(role.getParent().getId()).orElse(null);
      Validate.notNull(parentRole, "未找到角色的父级角色");
    }
  }

  @Override
  public Set<RoleVo> findByUserId(String userId){
    if(StringUtils.isBlank(userId)) {
      return Sets.newHashSet();
    }
    UserVo userVo = userService.findByUserId(userId);
    if(userVo == null) {
      return Sets.newHashSet();
    }
    String  account = userVo.getAccount();
    // 进行账户查询
    return this.findByUserAccount(account);
  }

  @Override
  public Set<RoleVo> findByUserAccount(String account){
    if(StringUtils.isBlank(account)) {
      return Sets.newHashSet();
    }
    
    List<RoleEntity> roles = this.roleRepository.findByUserAccount(account);
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newHashSet(roleVos);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findAllByUserId(java.lang.String)
   */
  @Override
  public List<RoleVo> findAllByUserId(String userId, Integer type) {
    /*
     * 在表单引擎V1.2+，按照一个用户数据库编号，查询这个用户对应的角色信息，就没有那么简单了，
     * 因为一个用户的角色可以由三方面决定：
     * a、这个用户直接绑定的角色信息
     * b、这个用户所属的一个或者多个用户组绑定的角色信息
     * c、这个用户所属的一个或者多个组织机构（不包括父级）绑定的角色信息
     * d、这个用户所属的一个或多个岗位（不包括父级）绑定的角色信息
     * 
     * 注意：数据层并没有使用status进行过滤，这里需要自行进行过滤
     * */
    Map<String, String> sourceMap = new HashMap<>();
    if(StringUtils.isBlank(userId)) {
      return Collections.emptyList();
    }
    
    // a、====
    Set<RoleEntity> resultRoles = new HashSet<>();
    UserVo userVo = userService.findByUserId(userId);
    if(userVo == null) {
      return Lists.newArrayList();
    }
    List<RoleEntity> roles = this.roleRepository.findByUserAccount(userVo.getAccount());
    if(roles != null && !roles.isEmpty()) {
      this.setSource(Sets.newHashSet(roles), sourceMap, "user");
      resultRoles.addAll(roles);
    }
    
    // b、====
    Set<UserGroupVo> groups = this.userGroupService.findByUserId(userId);
    if(groups != null && !groups.isEmpty()) {
      for(UserGroupVo item : groups){
        Set<RoleEntity> userGroupRoles = roleRepository.findByUserGroupCode(item.getGroupCode());
        userGroupRoles = userGroupRoles.stream().filter(r -> r.getTstatus() == NormalStatusEnum.ENABLE.getStatus()).collect(Collectors.toSet());
        this.setSource(userGroupRoles, sourceMap, "group");
        resultRoles.addAll(userGroupRoles);
      }
    }
    
    // c、====
    Set<OrganizationVo> orgs = this.organizationService.findOrgByUserId(userId);
    if(!CollectionUtils.isEmpty(orgs)) {
      for(OrganizationVo item : orgs){
        Set<RoleEntity> orgRoles = roleRepository.findByOrgCode(item.getCode());
        orgRoles = orgRoles.stream().filter(r -> r.getTstatus() == NormalStatusEnum.ENABLE.getStatus()).collect(Collectors.toSet());
        this.setSource(orgRoles, sourceMap, "org");
        resultRoles.addAll(orgRoles);

      }
    }

    // d、==== 获取当前岗位的绑的角色
    Set<RoleEntity> positionRoles = new HashSet<>();
    if (type != null && type.equals(1)) {
      List<PositionVo> positions = positionService.findByUserId(userId);
      if (!CollectionUtils.isEmpty(positions)) {
        List<String> positionCodes = positions.stream().map(PositionVo::getCode).collect(Collectors.toList());
        positionRoles = roleRepository.findByPositions(positionCodes);
      }
    } else {//默认查询主岗位的角色
      PositionVo positionVo = positionService.findMainPositionByUserId(userId);
      if (positionVo != null) {
        positionRoles = roleRepository.findByPosition(positionVo.getCode());
      }
    }
    resultRoles.addAll(positionRoles);
    this.setSource(positionRoles, sourceMap, "position");

    // 过滤出唯一信息后返回
    if(resultRoles.isEmpty()) {
      return Collections.emptyList();
    }
    List<RoleEntity> roleEntitys = resultRoles.stream().filter(item -> item.getTstatus() == 1).collect(Collectors.toList());
    // 转换vo后进行返回
    List<RoleVo> roleVos = new ArrayList<>();
    for(RoleEntity entity : roleEntitys){
      RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, RoleVo.class, HashSet.class, ArrayList.class);
      String sourceName = sourceMap.get(roleVo.getId());
      roleVo.setSource(sourceName);
      roleVo.setParent(this.findParentRole(entity));
      roleVos.add(roleVo);
    }
    return roleVos.stream().sorted(Comparator.comparing(RoleVo::getCreateDate)).collect(Collectors.toList());
  }

  /**
   * 将获取的来源放进map中存储，key为role的id
   * 方便后期放入到roleVo中
   * @param roles
   * @param sourceMap
   * @param source
   */
  private void setSource(Set<RoleEntity> roles, Map<String, String> sourceMap, String source) {
    if(CollectionUtils.isEmpty(roles)){
      return;
    }
    for(RoleEntity role : roles){
      if(sourceMap.containsKey(role.getId())){
        String value = sourceMap.get(role.getId()) + "," + source;
        sourceMap.put(role.getId(), value);
      }else {
        sourceMap.put(role.getId(), source);
      }
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findByCompetenceId(java.lang.String)
   */
  @Override
  public List<RoleVo> findByCompetenceId(String competenceId) {
    if(StringUtils.isBlank(competenceId)) {
      return Lists.newArrayList();
    }
    List<RoleEntity> roles = this.roleRepository.findByCompetenceId(competenceId);
    if (roles == null || roles.isEmpty()) {
      return Lists.newArrayList();
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(roleVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findByStatus(java.lang.Integer)
   */
  @Override
  public List<RoleVo> findByStatus(Integer useStatus) {
    if(useStatus == null) {
      return Collections.emptyList();
    }
    List<RoleEntity> rolesList = this.roleRepository.findByTstatus(useStatus);
    if (rolesList == null || rolesList.isEmpty()) {
      return Collections.emptyList();
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(rolesList, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(roleVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findById(java.lang.String)
   */
  @Override
  @NebulaServiceMethod(name="RoleService.findById" , desc="根据id查询角色记录" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public RoleVo findById(@ServiceMethodParam(name = "id") String roleId) {
    Validate.notNull(roleId, "角色id不能为空!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    if (!op.isPresent()) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(op.get(), RoleVo.class, HashSet.class, ArrayList.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findAll()
   */
  @Override
  public List<RoleVo> findAll() {
    List<RoleEntity> rolesList = this.roleRepository.findAll(Sort.by(Sort.Order.desc("createDate")));
    // 如果条件成立说明系统该数据异常，这是直接抛出错误信息
    if (rolesList == null || rolesList.isEmpty()) {
      return Collections.emptyList();
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(rolesList, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(roleVos);
  }

  /**
   * 根据角色编码删除（只能删除无绑定关系的角色）
   * @param roleCodes
   */
  @Override
  @Transactional
  public void deleteByCodes(String[] roleCodes) {
    Validate.notEmpty(roleCodes, "删除时，至少有一个角色编码");
    for(String roleCode :roleCodes){
      //a.角色与组织机构
      long roleOrgMappings = roleOrgMappingRepository.countByRole(roleCode);
      Validate.isTrue(roleOrgMappings == 0L, "角色编码[%s]与组织机构存在关联关系，不能删除", roleCode);
      //b.角色与岗位
      long rolePositionMappings = rolePositionMappingRepository.countByRole(roleCode);
      Validate.isTrue(rolePositionMappings == 0L, "角色编码[%s]与岗位存在关联关系，不能删除", roleCode);
      //c。角色与用户组
      long roleUserGroupMappings = roleUserGroupMappingRepository.countByRole(roleCode);
      Validate.isTrue(roleUserGroupMappings == 0L, "角色编码[%s]与用户组存在关联关系，不能删除", roleCode);
      //d.角色与用户
      long roleUserMappings = roleUserMappingRepository.countByRole(roleCode);
      Validate.isTrue(roleUserMappings == 0L, "角色编码[%s]与用户存在关联关系，不能删除", roleCode);
      RoleEntity roleEntity = this.roleRepository.findByRoleCode(roleCode);
      Validate.notNull(roleEntity, "该角色不存在，无法删除");
      this.roleRepository.deleteByRoleCode(roleCode);
    }

  }

  @Override
  public Set<RoleVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<RoleEntity> roles = this.roleRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(roles)){
      return Sets.newHashSet();
    }
    Collection<RoleVo> rvos = this.nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(rvos);
  }

  /**
   * 查询角色树状关系
   * @return
   */
  @Override
  public Set<RoleVo> findRoleTree() {
    LinkedHashSet<RoleVo> roleSet = new LinkedHashSet<>();
    Set<RoleEntity> rootRoles = roleRepository.findByParentIsNull();
    if(CollectionUtils.isEmpty(rootRoles)){
      return Sets.newHashSet();
    }

    for(RoleEntity roleEntity : rootRoles){
      RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, RoleVo.class, HashSet.class, ArrayList.class);
      roleVo.setChildren(this.findChildrenRole(roleEntity));
      roleSet.add(roleVo);
    }
    return roleSet;
  }

  /**
   * 查询子角色
   * @param roleEntity
   * @return
   */
  public Set<RoleVo> findChildrenRole(RoleEntity roleEntity) {
    LinkedHashSet<RoleVo> roleVoSet = new LinkedHashSet<>();
    //  查询子级结构
    Set<RoleEntity> children = roleEntity.getChildren();
    if(CollectionUtils.isEmpty(children)){
      return Sets.newHashSet();
    }
    for(RoleEntity child : children){
      RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(child, RoleVo.class, HashSet.class, ArrayList.class);
      roleVo.setChildren(this.findChildrenRole(child));
      roleVoSet.add(roleVo);
    }
    return roleVoSet;
  }

  /**
   * 查询父角色
   * @param child
   * @return
   */
  private RoleVo findParentRole(RoleEntity child) {
    RoleEntity parent = child.getParent();
    if(parent == null){
      return null;
    }
    parent = roleRepository.findById(parent.getId()).orElse(null);
    if(parent == null){
      return null;
    }
    RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(parent, RoleVo.class, HashSet.class, ArrayList.class);
    roleVo.setParent(this.findParentRole(parent));
    return roleVo;
  }

  /**
   * 根据角色名称，或则角色编码，或则角色编码查询角色信息 （角色名称支持模糊查询）
   * @param invokeParams
   * @return
   */
  @Override
  @NebulaServiceMethod(name="RoleService.findByConditions" , desc="条件查询角色列表" , returnPropertiesFilter="parent" , scope= NebulaServiceMethod.ScopeType.READ)
  public Page<RoleVo> findByConditions(InvokeParams invokeParams, @ServiceMethodParam(name = "pageable") Pageable pageable) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> conditions = ObjectUtils.defaultIfNull(invokeParams.getInvokeParams(), Maps.newHashMap());
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    Page<RoleEntity> rolePage = this.roleRepository.findByConditions(conditions, pageable);
    if(CollectionUtils.isEmpty(rolePage.getContent())) {
      return Page.empty(pageable);
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(rolePage.getContent(), RoleEntity.class, RoleVo.class, HashSet.class, ArrayList.class, "parent");
    return new PageImpl<>(Lists.newArrayList(roleVos), pageable, rolePage.getTotalElements());
  }

  @Override
  public RoleVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    RoleEntity role = roleRepository.findByRoleCode(code);
    if(role == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(role, RoleVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Set<RoleVo> findByRoleNames(Set<String> roleNames) {
    Set<RoleEntity> roles = roleRepository.findByRoleNames(roleNames);
    if(CollectionUtils.isEmpty(roles)) {
      Sets.newHashSet();
    }
    Collection<RoleVo> collection = nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(collection);
  }

  @Override
  public List<RoleVo> findByOrgCode(String orgCode) {
    if(StringUtils.isBlank(orgCode)) {
      return Lists.newArrayList();
    }
    Set<RoleOrgMappingEntity> orgOrgs = roleOrgMappingRepository.findDetailsByOrgCode(orgCode);
    if(CollectionUtils.isEmpty(orgOrgs)) {
      return Lists.newArrayList();
    }
    List<RoleEntity> roles = orgOrgs.stream().map(RoleOrgMappingEntity::getRole).collect(Collectors.toList());
    Collection<RoleVo> collection = nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(collection);
  }

  /**
   * 根据岗位编号查询直接关联的角色
   * @param positionCode
   * @return
   */
  @Override
  public Set<RoleVo> findByPositionCode(String positionCode) {
    Validate.notBlank(positionCode, "传入的岗位编号不能为空！！");
    List<RolePositionMappingEntity> mappings = this.rolePositionMappingRepository.findByPositionCode(positionCode);
    Set<RoleVo> roleSet = new HashSet<>();
    for (RolePositionMappingEntity mapping : mappings) {
      RoleEntity role = mapping.getRole();
      if (role != null){
        RoleVo roleVo = nebulaToolkitService.copyObjectByWhiteList(role, RoleVo.class, HashSet.class, ArrayList.class);
        roleSet.add(roleVo);
      }
    }
    return roleSet;
  }
  
  /**
   * 根据用户组编码查询绑定的角色
   * @param userGroupCode
   * @return
   */
  @Override
  public List<RoleVo> findByGroupCode(String userGroupCode) {
    if(StringUtils.isBlank(userGroupCode)) {
      return Lists.newArrayList();
    }
    Set<RoleEntity> roles = roleRepository.findByUserGroupCode(userGroupCode);
    if(CollectionUtils.isEmpty(roles)){
      return Lists.newArrayList();
    }
    Collection<RoleVo> collection = nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(collection);
  }

  /**
   * 根据角色编码查询角色信息
   * @param roleCodes
   * @return
   */
  @Override
  public Set<RoleVo> findByRoleCodes(Set<String> roleCodes) {
    Set<RoleEntity> roles = roleRepository.findByRoleCodes(roleCodes);
    if(CollectionUtils.isEmpty(roles)) {
      Sets.newHashSet();
    }
    Collection<RoleVo> collection = nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(collection);
  }

  @Override
  public Page<RoleVo> findAll(Pageable pageable) {
    Page<RoleEntity> roleEntityPage = this.roleRepository.findAll(pageable);
    if(CollectionUtils.isEmpty(roleEntityPage.getContent())) {
      return Page.empty(pageable);
    }
    Collection<RoleVo> collection = nebulaToolkitService.copyCollectionByWhiteList(roleEntityPage.getContent(), RoleEntity.class,
            RoleVo.class, HashSet.class, ArrayList.class, "parent");
    return new PageImpl<RoleVo>(Lists.newArrayList(collection), pageable, roleEntityPage.getTotalElements());
  }

}