package com.bizunited.platform.rbac.server.starter.service.internal;


import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.rbac.server.service.RbacSettingService;
import com.bizunited.platform.rbac.server.service.RolePositionLevelMappingService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.entity.RolePositionLevelMappingEntity;
import com.bizunited.platform.rbac.server.starter.repository.RolePositionLevelMappingRepository;
import com.bizunited.platform.rbac.server.vo.RbacSettingVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.position.PositionLevelService;
import com.bizunited.platform.user.common.vo.PositionLevelVo;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author: weikangning
 * @Date: 2020/8/16 21:27
 * @Description:
 */
@Service
public class RolePositionLevelMappingServiceImpl implements RolePositionLevelMappingService {

  @Autowired
  private RolePositionLevelMappingRepository rolePositionLevelMappingRepository;
  @Autowired
  private PositionLevelService positionLevelService;
  @Autowired
  private RbacSettingService rbacSettingService;
  @Autowired
  private RoleService roleService;

  @Override
  @Transactional
  public void bindPositionLevelRoles(String positionLevelCode, String[] roleIds) {
    Validate.notBlank(positionLevelCode, "进行职位级别-角色绑定时，必须指定当前的职位级别信息!!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行职位级别-角色绑定时,至少传入一个角色id信息!!");
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    PositionLevelVo positionLevelVo = positionLevelService.findByCode(positionLevelCode);
    Validate.notNull(positionLevelVo, "未找到职位级别：%s", positionLevelCode);
    for (String roleId : roleIds) {
      RoleVo role = this.roleService.findById(roleId);
      Validate.notNull(role, "未找到角色信息：%s", roleId);
      this.validateBindPosition(rbacSetting, role, positionLevelVo);
      RolePositionLevelMappingEntity mapping = new RolePositionLevelMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setPositionLevelCode(positionLevelCode);
      rolePositionLevelMappingRepository.save(mapping);
    }
  }

  @Override
  @Transactional
  public void rebindPositionLevelRoles(String positionLevelCode, String[] roleIds) {
    Validate.notBlank(positionLevelCode, "传入的职位级别编号不能为空!!");
    PositionLevelVo positionLevelVo = this.positionLevelService.findByCode(positionLevelCode);
    Validate.notNull(positionLevelVo, "未找到职位级别：%s", positionLevelCode);
    //传入的角色id（去重）
    Set<String> currentRoleIds = new HashSet<>();
    if (roleIds != null) {
      for (String roleId : roleIds) {
        currentRoleIds.add(roleId);
      }
    }
    //解绑指定职位级别和所有角色的绑定关系
    this.rolePositionLevelMappingRepository.unbindPositionLevelCode(positionLevelCode);
    //绑定角色
    this.bindPositionLevelRoles(positionLevelCode, roleIds);
  }


  @Override
  @Transactional
  public void unbindPositionLevelRoles(String positionLevelCode, String[] roleIds) {
    Validate.notBlank(positionLevelCode, "进行职位级别-角色解绑时，必须指定当前的职位级别信息!!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行职位级别-角色解绑时,至少传入一个角色id信息!!");
    for (String roleId : roleIds) {
      RoleVo role = this.roleService.findById(roleId);
      Validate.notNull(role, "未找到角色信息：%s", roleId);
      RolePositionLevelMappingEntity mapping = rolePositionLevelMappingRepository.findByRoleAndPositionLevelCode(role.getRoleCode(), positionLevelCode);
      if (mapping != null) {
        rolePositionLevelMappingRepository.delete(mapping);
      }
    }
  }

  @Override
  @Transactional
  public void unbindPositionLevels(String[] positionLevelCodes) {
    Validate.notEmpty(positionLevelCodes,"进行职位级别解绑全部角色时,职位级别编码信息不能为空，请检查!!");
    for (String positionLevelCode : positionLevelCodes) {
      List<RolePositionLevelMappingEntity> list = rolePositionLevelMappingRepository.findBypositionLevelCode(positionLevelCode);
      for (RolePositionLevelMappingEntity rolePositionLevelMappingEntity : list) {
        rolePositionLevelMappingRepository.delete(rolePositionLevelMappingEntity);
      }
    }
  }

  /**
   * 验证绑定角色和职位级别
   *
   * @param role
   * @param positionLevelVo
   */
  private void validateBindPosition(RbacSettingVo rbacSetting, RoleVo role, PositionLevelVo positionLevelVo) {
    long count = rolePositionLevelMappingRepository.countByRoleAndPositionLevel(role.getRoleCode(), positionLevelVo.getCode());
    Validate.isTrue(count == 0, "在批量绑定时,指定职位级别[%s]已经绑定了，不能重复绑定，请检查!!", positionLevelVo.getName());
    //如果角色和职位级别绑定关系为多对一时判断
    if (rbacSetting.getRolePositionRelation() == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
      count = rolePositionLevelMappingRepository.countByRole(role.getRoleCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和职位级别多对一，指定角色[%s]已经绑定了职位级别，不能继续绑定，请检查!!", role.getRoleName());
    }
  }
}
