package com.bizunited.platform.rbac.server.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RolePositionLevelMappingService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.vo.PositionLevelSimpleVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;


/**
 * @author: weikangning
 * @Date: 2020/8/16 21:28
 * @Description: 职位级别
 */
@RestController
@RequestMapping("/v1/nebula/rolePositionLevelMapping")
public class RolePositionLevelMappingController extends BaseController {

  @Autowired
  private RolePositionLevelMappingService rolePositionLevelMappingService;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RolePositionMappingController.class);

  /**
   * 将指定职位级别绑定一个或多个角色
   *
   * @param positionLevelCode
   * @param roleCodes
   * @return
   */
  @ApiOperation(value = "将指定职位级别绑定一个或多个角色")
  @PatchMapping("bindPositionLevelRoles")
  public ResponseModel bindPositionLevelRoles(
          @RequestParam @ApiParam("职位级别编码") String positionLevelCode,
          @RequestParam @ApiParam("当前要绑定的一个或多个角色编码") String[] roleCodes) {
    try {
      rolePositionLevelMappingService.bindPositionLevelRoles(positionLevelCode, roleCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定职位级别重新绑定一个或多个角色
   *
   * @param positionLevelCode
   * @param roleCodes
   * @return
   */
  @ApiOperation(value = "将指定职位级别重新绑定一个或多个角色")
  @PatchMapping("rebindPositionLevelRoles")
  public ResponseModel rebindPositionLevelRoles(
          @RequestParam @ApiParam("职位级别编码") String positionLevelCode,
          @RequestParam(required = false) @ApiParam("当前要绑定的一个或多个角色编码") String[] roleCodes) {
    try {
      rolePositionLevelMappingService.rebindPositionLevelRoles(positionLevelCode, roleCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定职位级别解除一个或多个角色的绑定
   *
   * @param positionLevelCode
   * @param roleCodes
   * @return
   */
  @ApiOperation(value = "将指定职位级别解除一个或多个角色的绑定")
  @PatchMapping("unbindPositionLevelRoles")
  public ResponseModel unbindPositionLevelRoles(
          @RequestParam @ApiParam("职位级别编码") String positionLevelCode,
          @RequestParam @ApiParam("当前要解绑定的一个或多个角色编码") String[] roleCodes) {
    try {
      rolePositionLevelMappingService.unbindPositionLevelRoles(positionLevelCode, roleCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * @Description: 通过编码数组解绑解绑一个或多个职位级别的全部角色
   * @Param:
   * @Return:
   * @Author: weikangning
   * @Date: 2020/8/27 17:37
   */
  @ApiOperation(value = "解绑一个或多个职位级别的全部角色")
  @PatchMapping("unbindPositionLevels")
  public ResponseModel unbindPositionLevels(
          @RequestParam @ApiParam("职位级别编码数组") String[] positionLevelCodes) {
    try {
      rolePositionLevelMappingService.unbindPositionLevels(positionLevelCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 批量根据职级Id,删除职级、解绑角色
   * @param positionIds 职级Id列表
   */
  @ApiOperation(value = "批量根据职级编码,删除职级、解绑角色")
  @PostMapping("/deletePositionLevelsUnbindRole")
  public ResponseModel deletePositionLevelsUnbindRole(@RequestParam("positionIds") @ApiParam("职级Id列表") String[] positionIds) {
    try {
      this.rolePositionLevelMappingService.deletePositionLevelsUnbindRole(positionIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据职级Id,删除职级、解绑角色
   * @param positionId 职级id
   */
  @ApiOperation(value = "根据职级Id,删除职级、解绑角色")
  @PostMapping("/deletePositionLevelUnbindRole")
  public ResponseModel deletePositionLevelUnbindRole(@RequestParam("positionId") @ApiParam("角色Id") String positionId) {
    try {
      this.rolePositionLevelMappingService.deletePositionLevelUnbindRole(positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 分页查询
   * @param conditions
   * @param pageable
   * @return
   */
  @ApiOperation(value = "分页查询")
  @PostMapping("/findByConditions")
  public ResponseModel findByConditions(@RequestBody(required = false) Map<String,Object> conditions,
                                 @PageableDefault(value = 50)@ApiParam(name = "pageable", value = "分页信息") Pageable pageable) {
    try {
      Page<PositionLevelSimpleVo> result = this.rolePositionLevelMappingService.findByConditions(conditions, pageable);
      return this.buildHttpResult(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("条件查询")
  @GetMapping("/findByNameAndRoleName")
  public ResponseModel findByNameAndRoleName(@RequestParam(value = "职级名称", required = false) @ApiParam("职级名称") String name,
                                        @RequestParam(value = "角色名称", required = false) @ApiParam("角色名称") String roleName){
    try {
      List<PositionLevelSimpleVo> result = this.rolePositionLevelMappingService.findByNameAndRoleName(name, roleName);
      return this.buildHttpResult(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("根据职级编码查询角色和职级信息")
  @GetMapping("/findByPositionLevelCode")
  public ResponseModel findByPositionLevelCode(@RequestParam(name = "positionLevelCode") @ApiParam("职级编码") String positionLevelCode){
    try{
      List<PositionLevelSimpleVo> result = this.rolePositionLevelMappingService.findByPositionLevelCode(positionLevelCode);
      return this.buildHttpResult(result);
    } catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation("根据职级查询角色列表")
  @GetMapping("/findRolesByPositionLevelCode")
  public ResponseModel findRolesByPositionLevelCode(@RequestParam @ApiParam("职级编码") String positionLevelCode){
    try{
      List<RoleVo> roleVos = this.rolePositionLevelMappingService.findRolesByPositionLevelCode(positionLevelCode);
      return this.buildHttpResult(roleVos);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}
