package com.bizunited.platform.rbac.server.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RoleOrgMappingService;

import com.bizunited.platform.user.common.vo.OrganizationVo;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

@RestController
@RequestMapping("/v1/nebula/roleOrgMapping")
public class RoleOrgMappingController extends BaseController {

  @Autowired
  private RoleOrgMappingService roleOrgMappingService;
  /**
   * 配置的那些可以忽略方法级检查的，具有超级管理员性质的角色名
   */
  @Value("${rbac.ignoreMethodCheckRoles:ADMIN}")
  private String[] ignoreMethodCheckRoles;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RoleOrgMappingController.class);

  /**
   * 将指定角色绑定一个或多个组织机构
   * @param currentOrgId
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定角色绑定一个或多个组织机构")
  @PatchMapping("bindOrgs")
  public ResponseModel bindOrgs(@RequestParam @ApiParam("角色ID") String roleId,
                                @RequestParam @ApiParam("一个或多个组织机构编码") String[] orgCodes) {
    try {
      this.roleOrgMappingService.bindOrgs(roleId, orgCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定角色重新绑定一个或多个组织机构
   * @param currentOrgId
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定角色重新绑定一个或多个组织机构")
  @PatchMapping("rebindOrgs")
  public ResponseModel rebindOrgs(@RequestParam @ApiParam("角色ID") String roleId,
                                @RequestParam(required = false) @ApiParam("一个或多个组织机构编码") String[] orgCodes) {
    try {
      this.roleOrgMappingService.rebindOrgs(roleId, orgCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 解除指定角色和指定组织机构的绑定关系
   * @param roleId
   * @param orgCodes
   * @return
   */
  @ApiOperation(value = "解除指定角色和指定组织机构的绑定关系")
  @PatchMapping("unbindOrgs")
  public ResponseModel unbindOrgs(@RequestParam @ApiParam("角色ID") String roleId,
                                  @RequestParam @ApiParam("当前需要解绑的组织编码") String[] orgCodes) {
    try {
      this.roleOrgMappingService.unbindOrgs(roleId, orgCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定组织机构绑定一个或多个角色
   * @param orgCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定组织机构绑定一个或多个角色")
  @PatchMapping("bindOrgRoles")
  public ResponseModel bindOrgRoles(@RequestParam @ApiParam("组织机构编码") String orgCode,
                                    @RequestParam @ApiParam("一个或多个角色ID") String[] roleIds) {
    try {
      this.roleOrgMappingService.bindOrgRoles(orgCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定组织机构重新绑定角色
   * @param orgCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定组织机构重新绑定一个或多个角色")
  @PatchMapping("rebindOrgRoles")
  public ResponseModel rebindOrgRoles(@RequestParam @ApiParam("组织机构编码") String orgCode,
                                    @RequestParam(required = false) @ApiParam("一个或多个角色ID") String[] roleIds) {
    try {
      this.roleOrgMappingService.rebindOrgRoles(orgCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 解除指定组织机构和角色的绑定关系
   * @param orgCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "解除指定组织机构和角色的绑定关系")
  @PatchMapping("unbindOrgRoles")
  public ResponseModel unbindOrgRoles(@RequestParam @ApiParam("组织机构编码") String orgCode,
                                      @RequestParam @ApiParam("一个或多个角色ID") String[] roleIds) {
    try {
      this.roleOrgMappingService.unbindOrgRoles(orgCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色ID查询绑定的组织机构
   * @param roleId
   * @return
   */
  @GetMapping("findOrgsByRoleId")
  @ApiOperation("根据角色ID查询绑定的组织机构")
  public ResponseModel findByRoleId(@RequestParam @ApiParam("角色ID") String roleId) {
    try {
      List<OrganizationVo> orgs = roleOrgMappingService.findByRoleId(roleId);
      return this.buildHttpResultW(orgs);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
