package com.bizunited.platform.rbac.server.starter.service.init;

import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.init.InitProcessService;
import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.rbac.server.service.RbacSettingService;
import com.bizunited.platform.rbac.server.starter.entity.ButtonEntity;
import com.bizunited.platform.rbac.server.starter.entity.CompetenceEntity;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.entity.RoleUserMappingEntity;
import com.bizunited.platform.rbac.server.starter.repository.ButtonRepository;
import com.bizunited.platform.rbac.server.starter.repository.CompetenceRepository;
import com.bizunited.platform.rbac.server.starter.repository.RoleRepository;
import com.bizunited.platform.rbac.server.starter.repository.RoleUserMappingRepository;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 该初始化过程用于在系统启动时初始化角色相关的信息，如超级管理员角色、菜单信息、按钮信息、关联关系等
 * @author yinwenjie
 */
@Component("rbacInitProcess")
public class RbacInitProcess implements InitProcessService {
  private static final String ADMIN_NAME = "admin";
  private static final String ADMIN_ROLE_NAME = "ADMIN";
  private static final String BASEROLE_NAME = "BASEROLE";
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private ButtonRepository buttonRepository;
  @Autowired
  private CompetenceRepository competenceRepository;
  @Autowired
  private RbacSettingService rbacSettingService;
  @Autowired
  private RoleUserMappingRepository roleUserMappingRepository;

  @Override
  public boolean doProcess() {
    // 如果当前系统中能够找到一个名叫BASEROLE的角色，就认为整个基础权限的初始化做完了，那么就没有必要继续做了
    return this.roleRepository.findByRoleName(ADMIN_ROLE_NAME) == null
        || this.roleRepository.findByRoleName(ADMIN_NAME) == null
        || this.rbacSettingService.findRbacSetting() == null;
  }
  
  @Override
  public int sort() {
    // 要求在系统启动时，第一时间检验并完成权限基本信息的初始化
    return 1;
  }

  @Override
  public boolean stopOnException() {
    // 如果该初始化过程出现异常，则直接抛出，并终止后续处理
    return true;
  }

  @Override
  public void init() {
    // 如果条件成立，则初始化基本角色
    if (this.roleRepository.findByRoleName(BASEROLE_NAME) == null) {
      this.initBaseRole();
      this.initBaseRoleCompetence();
      this.initBaseRoleButtons();
    }
    // 如果条件成立，则初始化管理员角色
    if (this.roleRepository.findByRoleName(ADMIN_ROLE_NAME) == null) {
      this.initAdminRole();
    }
    rbacSettingService.init();
  }
  
  /** 初始化基本角色 */
  private void initBaseRole() {
    /*
     * 向系统初始化BASEROLE角色信息
     * 保证系统中至少固定持有BASEROLE（用户基础角色）角色
     * */
    RoleEntity role = new RoleEntity();
    role.setComment("用户基础角色");
    role.setCreateDate(new Date());
    role.setRoleName(BASEROLE_NAME);
    role.setRoleCode(BASEROLE_NAME);
    role.setTstatus(1);
    role.setIsDeny(true);
    this.roleRepository.saveAndFlush(role);
  }

  /** 初始化用户基础角色的按钮 */
  private void initBaseRoleButtons() {
    RoleEntity baserole = roleRepository.findByRoleName(BASEROLE_NAME);
    Validate.notNull(baserole, "初始化用户基础角色时未找到基础角色!");
    Set<String> buttonCodes = new HashSet<>();

    // 定义基础角色的按钮权限
    buttonCodes.add("home-init");
    buttonCodes.add("account-getPosition");
    buttonCodes.add("account-getRoles");
    buttonCodes.add("account-style");
    buttonCodes.add("account-reset");
    buttonCodes.add("account-search");
    buttonCodes.add("position-style");
    buttonCodes.add("position-reset");
    buttonCodes.add("position-search");
    buttonCodes.add("position-edit");
    buttonCodes.add("userGroup-search");
    buttonCodes.add("userGroup-style");
    buttonCodes.add("userGroup-reset");
    buttonCodes.add("org-style");
    buttonCodes.add("org-search");
    buttonCodes.add("org-reset");
    buttonCodes.add("role-search");
    buttonCodes.add("role-style");
    buttonCodes.add("role-reset");
    buttonCodes.add("role-authBind");
    buttonCodes.add("menuManage-roleRelation");
    buttonCodes.add("menuManage-buttonManage");
    buttonCodes.add("menuManage-reset");
    buttonCodes.add("menuManage-search");
    buttonCodes.add("menuManage-style");
    buttonCodes.add("btnManage-reset");
    buttonCodes.add("btnManage-search");
    buttonCodes.add("btnManage-roleRelation");
    buttonCodes.add("btnManage-style");
    buttonCodes.add("envVariable-view");
    buttonCodes.add("envVariable-search");
    buttonCodes.add("codeRule-view");
    buttonCodes.add("codeRule-search");
    buttonCodes.add("scheduler-search");
    buttonCodes.add("scheduler-log");
    buttonCodes.add("scheduler-detail");
    buttonCodes.add("dataviewmanage-search");
    buttonCodes.add("dataview-previewData");
    buttonCodes.add("remoteservicemanage-search");
    buttonCodes.add("remoteservice-search");
    buttonCodes.add("servicesource-search");
    buttonCodes.add("basicImport-search");
    buttonCodes.add("basicImport-analysisAll");
    buttonCodes.add("formTemplate-search");
    buttonCodes.add("formTemplate-reset");
    buttonCodes.add("formStyle-preview");
    buttonCodes.add("globalfunc-search");
    buttonCodes.add("instance-search");
    buttonCodes.add("instance-form");
    buttonCodes.add("instance-info");
    buttonCodes.add("instance-activity");
    buttonCodes.add("tableManage-search");
    buttonCodes.add("tableManage-reload");
    buttonCodes.add("listTemp-search");
    buttonCodes.add("listTemp-view");
    buttonCodes.add("listTemp-reset");
    buttonCodes.add("formImport-search");
    buttonCodes.add("formImport-analysisAll");
    buttonCodes.add("pageflow-search");
    buttonCodes.add("pageflow-view");
    buttonCodes.add("flowTemplate-search");
    buttonCodes.add("flowTemplate-look");
    buttonCodes.add("flowTemplate-check");
    buttonCodes.add("monitor-search");
    buttonCodes.add("operation-search");
    buttonCodes.add("listFlow-search");
    buttonCodes.add("listFlow-detail");
    buttonCodes.add("useableFlowTemp-search");
    buttonCodes.add("useableFlowTemp-look");
    buttonCodes.add("launchBySelf-search");
    buttonCodes.add("launchBySelf-look");
    buttonCodes.add("doneFlow-search");
    buttonCodes.add("doneFlow-look");
    buttonCodes.add("doneFlow-detail");
    buttonCodes.add("waitingFlow-search");
    buttonCodes.add("copyFlow-search");
    buttonCodes.add("dataAuth-search");
    buttonCodes.add("dictManage-search");
    buttonCodes.add("basicExport-search");

    for (String buttonCode : buttonCodes) {
      ButtonEntity button = buttonRepository.findByCodeAndProjectName(buttonCode, ApplicationContextUtils.getProjectName());
      if(button == null) {
        continue;
      }
      buttonRepository.bindButton(baserole.getId(), button.getId());
    }
  }

  /** 初始化用户基础角色的功能 */
  private void initBaseRoleCompetence() {
    RoleEntity baserole = roleRepository.findByRoleName(BASEROLE_NAME);
    Validate.notNull(baserole, "初始化用户基础角色时未找到基础角色!");
    Set<String> comments = new HashSet<>();
    comments.add("表单实例列表");
    comments.add("表单模板管理");
    comments.add("表单样式管理");
    comments.add("表单模板列表");
    comments.add("账户管理");
    comments.add("页面流管理");
    comments.add("全局函数管理");
    comments.add("用户与权限");
    comments.add("用户组管理");
    comments.add("表单引擎");
    comments.add("基础配置导入管理");
    comments.add("数据视图编辑");
    comments.add("监听器管理");
    comments.add("远端调用-地址管理");
    comments.add("列表功能");
    comments.add("功能菜单权限管理");
    comments.add("表单实例管理");
    comments.add("参数管理");
    comments.add("角色管理");
    comments.add("我的待办");
    comments.add("我的已办");
    comments.add("编码规则管理");
    comments.add("基础配置导出管理");
    comments.add("数据视图-数据源管理");
    comments.add("我可发起的流程");
    comments.add("业务数据表管理");
    comments.add("组织机构管理");
    comments.add("首页");
    comments.add("数据导入管理");
    comments.add("远端调用管理");
    comments.add("流程实例列表");
    comments.add("我的抄送");
    comments.add("数据导出管理");
    comments.add("功能按钮权限管理");
    comments.add("数据权限");
    comments.add("流程模板管理");
    comments.add("我发起的流程");
    comments.add("按钮功能操作管理");
    comments.add("数据字典管理");
    comments.add("数据源");
    comments.add("系统主题管理");
    comments.add("定时任务管理");
    comments.add("岗位管理");
    comments.add("流程模板列表");
    comments.add("基础工具套件");
    comments.add("流程引擎");
    comments.add("流程实例管理");
    for (String comment : comments) {
      List<CompetenceEntity> competences = competenceRepository.findByComment(comment);
      if(CollectionUtils.isEmpty(competences)) {
        continue;
      }
      for (CompetenceEntity competence : competences) {
        roleRepository.bindCompetence(competence.getId(), baserole.getId());
      }
    }
  }
  
  /**
   * 初始化管理员角色
   */
  private void initAdminRole() {
    /*
     * 向系统初始化role角色信息
     * 保证系统中至少固定持有ADMIN（管理员）角色
     * */
    RoleEntity role = new RoleEntity();
    role.setComment("管理员角色");
    role.setCreateDate(new Date());
    role.setRoleName(ADMIN_ROLE_NAME);
    role.setRoleCode(ADMIN_ROLE_NAME);
    role.setTstatus(NormalStatusEnum.ENABLE.getStatus());
    role.setIsDeny(true);
    this.roleRepository.saveAndFlush(role);
    // 管理员角色与管理员的绑定
    RoleUserMappingEntity entity = new RoleUserMappingEntity();
    entity.setRole(role);
    entity.setUserAccount(ADMIN_NAME);
    roleUserMappingRepository.save(entity);

  }

}
