package com.bizunited.platform.rbac.server.starter.service.internal;

import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.rbac.server.service.RbacSettingService;
import com.bizunited.platform.rbac.server.service.RolePositionMappingService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.entity.RolePositionMappingEntity;
import com.bizunited.platform.rbac.server.starter.repository.RolePositionMappingRepository;
import com.bizunited.platform.rbac.server.vo.RbacSettingVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

public class RolePositionMappingServiceImpl implements RolePositionMappingService {

  @Autowired
  private RolePositionMappingRepository rolePositionMappingRepository;
  @Autowired
  private RoleService roleService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private RbacSettingService rbacSettingService;
  
  @Override
  @Transactional
  public void bindPositions(String roleId, String[] positionCodes) {
    Validate.notBlank(roleId, "进行角色-岗位绑定时，必须指定当前的角色信息!!");
    Validate.isTrue(positionCodes != null && positionCodes.length > 0, "进行角色-岗位绑定时,至少传入一个岗位编号信息!!");
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到角色信息：%s", roleId);
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    for (String positionCode : positionCodes) {
      PositionVo positionVo = positionService.findByCode(positionCode);
      Validate.notNull(positionVo, "未找到岗位：%s", positionCode);
      this.validateBindPosition(rbacSetting, role, positionVo);
      RolePositionMappingEntity mapping = new RolePositionMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setPositionCode(positionCode);
      rolePositionMappingRepository.save(mapping);
    }
  }

  @Override
  @Transactional
  public void unbindPositions(String roleId, String[] positionCodes) {
    Validate.notBlank(roleId, "进行角色-岗位绑定时，必须指定当前的角色信息!!");
    Validate.isTrue(positionCodes != null && positionCodes.length > 0, "进行角色-岗位绑定时,至少传入一个岗位编号信息!!");
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到角色信息：%s", roleId);
    for (String positionCode : positionCodes) {
      RolePositionMappingEntity mapping = rolePositionMappingRepository.findByRoleAndPositionCode(role.getRoleCode(), positionCode);
      if(mapping!=null){
        rolePositionMappingRepository.delete(mapping);
      }
    }
  }

  /**
   * 解除指定角色和所有关联岗位的绑定关系
   * @param roleId
   */
  @Override
  @Transactional
  public void unbindAllByRoleId(String roleId) {
    Validate.notBlank(roleId,"角色主键不能为空");
    Set<PositionVo> positionVos = this.findByRoleId(roleId);
    if(CollectionUtils.isEmpty(positionVos)){
      return;
    }
    String[] positionIdArray = this.getPositionIdArray(positionVos);
    this.unbindPositions(roleId,positionIdArray);
  }

  /**
   * 将指定角色重新绑定一个或多个岗位
   * @param roleId
   * @param positionCodes
   */
  @Override
  @Transactional
  public void rebindPositions(String roleId, String[] positionCodes) {
    Validate.notBlank(roleId, "传入的角色id不能为空!!");
    RoleVo role = this.roleService.findById(roleId);
    Validate.notNull(role, "未找到角色信息：%s", roleId);
    //传入的岗位编号（去重）
    Set<String> currentPositionCodes = new HashSet<>();
    if (positionCodes != null){
      for (String  positionCode : positionCodes) {
        currentPositionCodes.add(positionCode);
      }
    }
    
    // 删除指定角色已绑定的所有岗位信息
    this.rolePositionMappingRepository.unbindRoleCode(role.getRoleCode());
    rolePositionMappingRepository.flush();
    
    //绑定岗位(注意，可能不绑定任何岗位)
    RbacSettingVo rbacSetting = this.rbacSettingService.findRbacSetting();
    for (String bindPositionCode : currentPositionCodes) {
      PositionVo positionVo = this.positionService.findByCode(bindPositionCode);
      Validate.notNull(positionVo, "未找到岗位：%s", bindPositionCode);
      this.validateBindPosition(rbacSetting, role, positionVo);
      RolePositionMappingEntity mapping = new RolePositionMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(roleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setPositionCode(bindPositionCode);
      this.rolePositionMappingRepository.save(mapping);
    }
    
  }

  /**
   * 验证绑定角色和岗位
   * @param role
   * @param positionVo
   */
  private void validateBindPosition(RbacSettingVo rbacSetting, RoleVo role, PositionVo positionVo) {
    long count = rolePositionMappingRepository.countByRoleAndPosition(role.getRoleCode(), positionVo.getCode());
    Validate.isTrue(count == 0, "在批量绑定时,指定岗位[%s]已经绑定了，不能重复绑定，请检查!!", positionVo.getName());
    //如果角色和岗位绑定关系为多对多,多对一，一对多，一对一分别控制判断
    if (rbacSetting.getRolePositionRelation() == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
      count = rolePositionMappingRepository.countByRoleAndPosition(role.getRoleCode(), positionVo.getCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和岗位多对多，指定岗位[%s]已经绑定了，不能重复绑定，请检查!!", positionVo.getName());
    }
    if (rbacSetting.getRolePositionRelation() == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
      count = rolePositionMappingRepository.countByRole(role.getRoleCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和岗位多对一，指定角色[%s]已经绑定了岗位，不能继续绑定，请检查!!", role.getRoleName());
    }
    if (rbacSetting.getRolePositionRelation() == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
      count = rolePositionMappingRepository.countByPositionCode(positionVo.getCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和岗位一对多，指定岗位[%s]已经绑定了角色，不能继续绑定，请检查!!", positionVo.getName());
    }
    if (rbacSetting.getRolePositionRelation() == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
      count = rolePositionMappingRepository.countByRole(role.getRoleCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和岗位一对一，指定角色[%s]已经绑定了岗位，不能继续绑定，请检查!!", role.getRoleName());
      count = rolePositionMappingRepository.countByPositionCode(positionVo.getCode());
      Validate.isTrue(count == 0, "在批量绑定时,角色和岗位一对一，指定岗位[%s]已经绑定了角色，不能继续绑定，请检查!!", positionVo.getName());
    }
  }
  
  @Override
  @Transactional
  public void bindPositionRoles(String positionCode, String[] roleIds) {
    Validate.notBlank(positionCode, "进行岗位-角色绑定时，必须指定当前的岗位信息!!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行岗位-角色绑定时,至少传入一个角色id信息!!");
    RbacSettingVo rbacSetting = rbacSettingService.findRbacSetting();
    PositionVo positionVo = positionService.findByCode(positionCode);
    Validate.notNull(positionVo, "未找到岗位：%s", positionCode);
    for (String roleId : roleIds) {
        RoleVo role = this.roleService.findById(roleId);
        Validate.notNull(role, "未找到角色信息：%s", roleId);
        this.validateBindPosition(rbacSetting, role, positionVo);
        RolePositionMappingEntity mapping = new RolePositionMappingEntity();
        RoleEntity currentRole = new RoleEntity();
        currentRole.setId(roleId);
        currentRole.setRoleCode(role.getRoleCode());
        mapping.setRole(currentRole);
        mapping.setPositionCode(positionCode);
        rolePositionMappingRepository.save(mapping);
      }
  }

  @Override
  @Transactional
  public void unbindPositionRoles(String positionCode, String[] roleIds) {
    Validate.notBlank(positionCode, "进行岗位-角色解绑时，必须指定当前的岗位信息!!");
    Validate.isTrue(roleIds != null && roleIds.length > 0, "进行岗位-角色解绑时,至少传入一个角色id信息!!");
    for (String roleId : roleIds) {
        RoleVo role = this.roleService.findById(roleId);
        Validate.notNull(role, "未找到角色信息：%s", roleId);
        RolePositionMappingEntity mapping = rolePositionMappingRepository.findByRoleAndPositionCode(role.getRoleCode(), positionCode);
        if(mapping!=null){
          rolePositionMappingRepository.delete(mapping);
        }
    }
  }

  /**
   * 将指定岗位重新绑定一个或多个角色
   * @param positionCode
   * @param roleIds
   */
  @Override
  @Transactional
  public void rebindPositionRoles(String positionCode, String[] roleIds) {
    Validate.notBlank(positionCode, "传入的岗位编号不能为空!!");
    PositionVo positionVo = this.positionService.findByCode(positionCode);
    Validate.notNull(positionVo, "未找到岗位：%s", positionCode);
    
    //传入的角色id（去重）
    Set<String> currentRoleIds = new HashSet<>();
    if (roleIds != null){
      for (String  roleId : roleIds) {
        currentRoleIds.add(roleId);
      }
    }
    //解绑指定岗位和所有角色的绑定关系
    this.rolePositionMappingRepository.unbindPositionCode(positionCode);
    
    //绑定角色
    RbacSettingVo rbacSetting = this.rbacSettingService.findRbacSetting();
    for (String bindRoleId : roleIds) {
      RoleVo role = this.roleService.findById(bindRoleId);
      Validate.notNull(role, "未找到角色：%s", bindRoleId);
      this.validateBindPosition(rbacSetting, role, positionVo);
      
      RolePositionMappingEntity mapping = new RolePositionMappingEntity();
      RoleEntity currentRole = new RoleEntity();
      currentRole.setId(bindRoleId);
      currentRole.setRoleCode(role.getRoleCode());
      mapping.setRole(currentRole);
      mapping.setPositionCode(positionCode);
      this.rolePositionMappingRepository.save(mapping);
    }
  }
  /**
   * 根据角色ID查询直接关联的岗位
   * @param roleId
   * @return
   */
  @Override
  public Set<PositionVo> findByRoleId(String roleId) {
    if(StringUtils.isBlank(roleId)) {
      return Sets.newHashSet();
    }
    RoleVo role = this.roleService.findById(roleId);
    if(role == null) {
      return Sets.newHashSet();
    }
    List<RolePositionMappingEntity> mappings = this.rolePositionMappingRepository.findByRole(role.getRoleCode());
    if(mappings == null) {
      return Sets.newHashSet();
    }
    
    // TODO 需要进行优化
    Set<PositionVo> positionVos = new HashSet<>();
    for (RolePositionMappingEntity mapping : mappings) {
      PositionVo positionVo = this.positionService.findPositionDetailByCode(mapping.getPositionCode());
      if (positionVo != null){
        positionVos.add(positionVo);
      }
    }
    return positionVos;
  }

  /**
   * 根据角色ID查询直接关联的岗位
   *
   * @param roleId
   * @param name
   * @param userName
   * @param pageable
   * @return
   */
  @Override
  @NebulaServiceMethod(name="RolePositionMappingService.findPageByRoleId" , desc="根据角色ID查询直接关联的岗位" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public Page<PositionVo> findPageByRoleId(@ServiceMethodParam(name = "id") String roleId, @ServiceMethodParam(name = "name") String name,
                                           @ServiceMethodParam(name = "userName") String userName, @ServiceMethodParam(name = "pageable") Pageable pageable) {
    if(StringUtils.isBlank(roleId)){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    Set<PositionVo> positionVosInRoleId = this.findByRoleId(roleId);
    if(CollectionUtils.isEmpty(positionVosInRoleId)){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    List<String> positionIdArray = this.getPositionIdList(positionVosInRoleId);
    HashMap<String, Object> condition = Maps.newHashMap();
    condition.put("ids", positionIdArray);
    condition.put("name", name);
    condition.put("userName", userName);
    condition.put("flag",1);
    condition.put("pageable", pageable);
    return this.positionService.findByConditions(condition,pageable);
  }

  /**
   * 根据角色ID查询没有关联该角色的岗位
   * @param roleId
   * @param name
   * @param userName
   * @param pageable
   * @return
   */
  @Override
  @NebulaServiceMethod(name="RolePositionMappingService.findNotBindRole" , desc="根据角色ID查询没有关联该角色的岗位" , returnPropertiesFilter="organization,positionLevel,parent" , scope= NebulaServiceMethod.ScopeType.READ)
  public Page<PositionVo> findNotBindRole(@ServiceMethodParam(name = "id") String roleId, @ServiceMethodParam(name = "name") String name,
                                          @ServiceMethodParam(name = "userName") String userName, @ServiceMethodParam(name = "pageable") Pageable pageable) {
    if(StringUtils.isBlank(roleId)){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    Set<PositionVo> positionVosInRoleId = this.findByRoleId(roleId);
    if(positionVosInRoleId == null){
      positionVosInRoleId = Sets.newHashSet();
    }
    List<String> positionIdArray = this.getPositionIdList(positionVosInRoleId);
    HashMap<String, Object> condition = Maps.newHashMap();
    condition.put("ids", positionIdArray);
    condition.put("name", name);
    condition.put("userName", userName);
    condition.put("flag", 0);
    condition.put("pageable", pageable);
    return this.positionService.findByConditions(condition,pageable);
  }

  /**
   * 查询没有绑定任何角色的职位
   *
   * @return
   * @param name
   * @param userName
   * @param pageable
   */
  @Override
  @NebulaServiceMethod(name="RolePositionMappingService.findNotBindAnyRole" , desc="分页查询没有绑定任何角色的职位" , returnPropertiesFilter="organization,positionLevel,parent" , scope= NebulaServiceMethod.ScopeType.READ)
  public Page<PositionVo> findNotBindAnyRole(@ServiceMethodParam(name = "name") String name, @ServiceMethodParam(name = "userName") String userName, @ServiceMethodParam(name = "pageable") Pageable pageable) {
    List<RolePositionMappingEntity> all = this.rolePositionMappingRepository.findAll();
    if(all == null){
      all = Lists.newLinkedList();
    }
    List<String> codes = this.getPositionCodeListFromMapping(all);
    HashMap<String, Object> condition = Maps.newHashMap();
    condition.put("codes", codes);
    condition.put("name", name);
    condition.put("userName", userName);
    condition.put("flag", 0);
    condition.put("pageable", pageable);
    return  this.positionService.findByConditions(condition,pageable);
  }

  /**
   * 根据角色主键、指定新的角色主键、职位编码列表，替换指定职位的绑定角色
   *
   * @param roleId        角色主键
   * @param rebinRoleId   新的角色主键
   * @param positionCodes 职位编码列表
   */
  @Override
  @Transactional
  public void rebindPositionsById(String roleId, String rebinRoleId, String[] positionCodes) {
    Validate.notBlank(roleId,"角色主键不能为空");
    Validate.notBlank(rebinRoleId,"替换的角色主键不能为空");
    Validate.notEmpty(positionCodes,"职位列表不能为为空");
    this.unbindPositions(roleId, positionCodes);
    this.bindPositions(rebinRoleId, positionCodes);
  }

  /**
   * 解析岗位集合，获取岗位Id 列表
   * @param positionVos
   * @return
   */
  private List<String> getPositionIdList(Set<PositionVo> positionVos){
    if(CollectionUtils.isEmpty(positionVos)){
      return Lists.newLinkedList();
    }
    LinkedList<String> list = Lists.newLinkedList();
    for(PositionVo positionVo : positionVos){
      list.add(positionVo.getId());
    }
    return list;
  }

  /**
   * 解析岗位集合，获取岗位Id 数组
   * @param positionVos
   * @return
   */
  private String[] getPositionIdArray(Set<PositionVo> positionVos){
    if(CollectionUtils.isEmpty(positionVos)){
      return new String[0];
    }
    String[] idArray=new String[positionVos.size()];
    Integer count = 0;
    for(PositionVo positionVo : positionVos){
      idArray[count] = positionVo.getId();
      count++;
    }
    return idArray;
  }

  /**
   * 从关联表中获取职位Code
   * @param rolePositionMappingEntities
   * @return
   */
  private List<String> getPositionCodeListFromMapping(List<RolePositionMappingEntity> rolePositionMappingEntities){
    if(CollectionUtils.isEmpty(rolePositionMappingEntities)){
      return Lists.newLinkedList();
    }
    LinkedList<String> list = Lists.newLinkedList();
    for(RolePositionMappingEntity rolePositionMappingEntity:rolePositionMappingEntities){
      list.add(rolePositionMappingEntity.getPositionCode());
    }
    return list;
  }
}
