package com.bizunited.platform.rbac.server.starter.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.ButtonService;
import com.bizunited.platform.rbac.server.service.CompetenceService;

import com.bizunited.platform.rbac.server.service.RoleExtendVoService;
import com.bizunited.platform.rbac.server.service.RolePositionMappingService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac.server.starter.entity.RoleExtendEntity;
import com.bizunited.platform.rbac.server.starter.repository.RoleExtendEntityRepository;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.rbac.server.vo.RoleExtendVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * RoleExtendEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("RoleExtendVoServiceImpl")
public class RoleExtendVoServiceImpl implements RoleExtendVoService {
  @Autowired
  private RoleExtendEntityRepository roleExtendEntityRepository;
  @Autowired
  private CompetenceService competenceService;
  @Autowired
  private ButtonService buttonService;
  @Autowired
  private RoleService roleService;
  @Autowired
  private RolePositionMappingService rolePositionMappingService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public RoleExtendVo create(RoleExtendVo roleExtendVo) {
    RoleExtendVo current = this.createForm(roleExtendVo);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public RoleExtendVo createForm(RoleExtendVo roleExtendVo) {
    this.createValidation(roleExtendVo);
    //首先保存RoleVo
    RoleVo roleVo = roleExtendVo.getRoleVo();
    roleVo=this.initRoleVo(roleVo);
    Validate.notBlank(roleVo.getRoleCode(),"编码不能为空");
    //原本的角色新增不能新增状态为0的角色，只能新增之后再禁用
    boolean flag = false;
    if(roleVo.getTstatus() == 0){
      flag = true;
      roleVo.setTstatus(1);
    }
    RoleVo currentRoleVo = this.roleService.create(roleVo);
    if(flag){
      this.roleService.disable(new String[]{roleVo.getId()});
    }
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    RoleEntity roleEntity = this.nebulaToolkitService.copyObjectByWhiteList(currentRoleVo, RoleEntity.class, HashSet.class, ArrayList.class);
    RoleExtendEntity entity = this.initRoleExtendEntity(roleExtendVo);
    entity.setRoleEntity(roleEntity);
    this.roleExtendEntityRepository.saveAndFlush(entity);
    // 返回最终处理的结果，里面带有详细的关联信息
    return this.extendCopy(entity);
  }

  /**
   * 在创建一个新的RoleExtendEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(RoleExtendVo roleExtendVo) {
    Validate.notNull(roleExtendVo , "进行当前操作时，信息对象必须传入!!");
    RoleVo roleVo = roleExtendVo.getRoleVo();
    Validate.notNull(roleVo,"角色不能为空");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(roleExtendVo.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    roleExtendVo.setId(null);
    String roleName = roleVo.getRoleName();
    Validate.notBlank(roleName, "添加信息时，角色名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    RoleExtendEntity entityByName = this.roleExtendEntityRepository.findByRoleName(roleName);
    Validate.isTrue(entityByName == null || entityByName.getDeleteFlag() == 1, "名称为[%s]的角色已存在,请检查",roleName);
  }

  private RoleExtendEntity initRoleExtendEntity(RoleExtendVo roleExtendVo){
    RoleExtendEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(roleExtendVo, RoleExtendEntity.class, HashSet.class, ArrayList.class);
    entity.setDeleteFlag(0);
    UserVo currentUser = SecurityUtils.getCurrentUser();
    Validate.notNull(currentUser,"当前登录用户异常，请重试");
    String account = currentUser.getAccount();
    entity.setCreateUser(account);
    return entity;
  }
  @Transactional
  @Override
  public RoleExtendVo update(RoleExtendVo roleExtendVo) {
    RoleExtendVo current = this.updateForm(roleExtendVo);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public RoleExtendVo updateForm(RoleExtendVo roleExtendVo) {
    this.updateValidation(roleExtendVo);
    // ===================基本信息
    String currentId = roleExtendVo.getId();
    Optional<RoleExtendEntity> op_currentRoleExtendEntity = this.roleExtendEntityRepository.findById(currentId);
    RoleExtendEntity currentRoleExtendEntity = op_currentRoleExtendEntity.orElse(null);
    Validate.notNull(currentRoleExtendEntity ,"未发现指定的原始模型对象信息");
    //首先更新RoleVo
    RoleVo roleVo = roleExtendVo.getRoleVo();
    RoleVo roleVoParent = roleVo.getParent();
    if(roleVoParent != null && StringUtils.isBlank(roleVoParent.getId())){
      roleVo.setParent(null);
    }
    RoleVo currentRoleVo = this.roleService.update(roleVo);
    RoleEntity roleEntity = this.nebulaToolkitService.copyObjectByWhiteList(currentRoleVo, RoleEntity.class, HashSet.class, ArrayList.class);
    // 开始重新赋值——一般属性
    currentRoleExtendEntity.setRoleEntity(roleEntity);
    this.roleExtendEntityRepository.saveAndFlush(currentRoleExtendEntity);
    return this.extendCopy(currentRoleExtendEntity);
  }
  /**
   * 在更新一个已有的RoleExtendEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(RoleExtendVo roleExtendVo) {
    RoleVo roleVo = roleExtendVo.getRoleVo();
    Validate.notNull(roleVo,"角色不能为空");
    String id = roleExtendVo.getId();
    Validate.isTrue(!StringUtils.isBlank(id), "修改信息时，当期信息的数据编号（主键）必须有值！");
    // 基础信息判断，基本属性，需要满足not null
    String roleName = roleVo.getRoleName();
    String roleVoId = roleVo.getId();
    Validate.notBlank(roleName, "修改信息时，表单名称不能为空！");
    Validate.notBlank(roleVoId, "修改信息时，主键不能为空！");
    // 重复性判断，基本属性，需要满足unique = true
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    RoleExtendEntity entityByName = this.roleExtendEntityRepository.findByRoleName(roleName);
    if(entityByName != null && entityByName.getDeleteFlag() == 0){
      Validate.isTrue(StringUtils.equals(entityByName.getId(),id), "名称为[%s]的角色已存在,请检查", roleName);
    }

  }

  /**
   * 初始化roleVo
   * @param roleVo
   * @return
   */
  private RoleVo initRoleVo(RoleVo roleVo){
    RoleVo roleVoParent = roleVo.getParent();
    if(roleVoParent != null && StringUtils.isBlank(roleVoParent.getId())){
      roleVo.setParent(null);
    }
    roleVo.setIsDeny(false);
    return roleVo;
  }
  @Override
  public RoleExtendVo findDetailsById(String id) {
    // 这是主模型下的明细查询过程
    // 1、=======
    if(StringUtils.isBlank(id)) {
      return null;
    }
    RoleExtendEntity current = this.roleExtendEntityRepository.findDetailsById(id);
    if(current == null) {
      return null;
    }
    RoleExtendVo roleExtendVo = this.nebulaToolkitService.copyObjectByWhiteList(current, RoleExtendVo.class, HashSet.class, ArrayList.class,"roleVo");
    return roleExtendVo;
  }
  @Override
  public RoleExtendVo findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    RoleExtendEntity roleExtendEntity = roleExtendEntityRepository.findById(id).orElse(null);
    return this.extendCopy(roleExtendEntity);
  }

  /**
   * 带分页的条件查询，根据名称模糊、编码模糊、状态分页查询角色
   * @param roleName
   * @param roleCode
   * @param tstatus
   * @param pageable
   * @return
   */
  public Page<JSONObject> queryPage(String roleName, String roleCode, Integer tstatus, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(roleName)) {
      conditions.put("roleName", roleName);
    }
    if (StringUtils.isNotBlank(roleCode)) {
      conditions.put("roleCode", roleCode);
    }
    if (tstatus != null) {
      conditions.put("tstatus", tstatus);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    Page<RoleExtendEntity> roleExtendEntities = this.roleExtendEntityRepository.queryPage(conditions, pageable);

    //自行转换分页数据
    List<RoleExtendEntity> list = roleExtendEntities.getContent();
    if(list.isEmpty()){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    List<JSONObject> jsonObjectList = this.extendEntityToJson(list);
    return new PageImpl<>(jsonObjectList, pageable, roleExtendEntities.getTotalElements());
  }
  /**
   * 根据角色编码删除角色拓展
   * @param roleCode 角色编码
   */
  @Override
  @Transactional
  public void deleteByRoleCode(String roleCode) {
    Validate.notBlank(roleCode, "参数不能为空");
    RoleExtendEntity roleExtendEntity = this.roleExtendEntityRepository.findByRoleCode(roleCode);
    Validate.notNull(roleExtendEntity, "未找到编码【%s】的角色拓展，请检查", roleCode);
    RoleEntity roleEntity = roleExtendEntity.getRoleEntity();
    Validate.notNull(roleEntity, "未找到编码【%s】的角色，请检查", roleCode);
    String id = roleEntity.getId();

    //解绑角色与按钮之间的关联
    this.buttonService.unbindAllByRoleId(id);
    //解绑角色与功能之间的关联
    this.competenceService.unbindAllByRoleId(id);
    //解绑角色与职位之间的关联
    this.rolePositionMappingService.unbindAllByRoleId(id);

    roleExtendEntity.setDeleteFlag(1);
    this.roleExtendEntityRepository.save(roleExtendEntity);
  }

  /**
   * 根据角色编码批量删除角色拓展，同时解绑全部菜单、按钮、职位
   *
   * @param roleCodes 角色编码
   */
  @Override
  @Transactional
  public void deleteByCodes(String[] roleCodes) {
    Validate.notEmpty(roleCodes,"参数不能为空");
    for (int i = 0; i < roleCodes.length; i++ ){
      this.deleteByRoleCode(roleCodes[i]);
    }
  }

  /**
   * 将角色拓展以及角色都转换为Vo
   * @param roleExtendEntity
   * @return
   */
  private RoleExtendVo extendCopy(RoleExtendEntity roleExtendEntity){
    Validate.notNull(roleExtendEntity,"角色拓展不能为空");
    RoleEntity roleEntity = roleExtendEntity.getRoleEntity();
    Validate.notNull(roleEntity,"角色不能为空");
    RoleExtendVo roleExtendVo = this.nebulaToolkitService.copyObjectByWhiteList(roleExtendEntity, RoleExtendVo.class, HashSet.class, ArrayList.class,"roleVo");
    RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, RoleVo.class, HashSet.class, ArrayList.class, "parent");
    roleExtendVo.setRoleVo(roleVo);
    return roleExtendVo;
  }

  /**
   * 将拓展列表转换为Json列表，其中角色与拓展字段变为同一层级
   * @param list
   * @return
   */
  private List<JSONObject> extendEntityToJson(List<RoleExtendEntity> list){
    LinkedList<JSONObject> jsonList = Lists.newLinkedList();
    if(CollectionUtils.isEmpty(list)){
      return jsonList;
    }
    list.forEach(roleExtendEntity -> {
      JSONObject jsonObject = new JSONObject();
      jsonObject.put("id",roleExtendEntity.getId());
      jsonObject.put("createTime",roleExtendEntity.getCreateTime());
      jsonObject.put("createUser",roleExtendEntity.getCreateUser());
      RoleEntity roleEntity = roleExtendEntity.getRoleEntity();
      if(roleEntity == null) {
        roleEntity = new RoleEntity();
      }
      jsonObject.put("roleId",roleEntity.getId());
      jsonObject.put("comment",roleEntity.getComment());
      jsonObject.put("roleCode",roleEntity.getRoleCode());
      jsonObject.put("roleName",roleEntity.getRoleName());
      jsonObject.put("roleType",roleEntity.getRoleType());
      jsonObject.put("tstatus",roleEntity.getTstatus());
      jsonList.add(jsonObject);
    });
    return jsonList;
  }

} 
