package com.bizunited.platform.rbac.server.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RoleUserGroupMappingService;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

@RestController
@RequestMapping("/v1/nebula/roleUserGroupMapping")
public class RoleUserGroupMappingController extends BaseController {
  @Autowired
  private RoleUserGroupMappingService roleUserGroupMappingService;
  /**
   * 配置的那些可以忽略方法级检查的，具有超级管理员性质的角色名
   */
  @Value("${rbac.ignoreMethodCheckRoles:ADMIN}")
  private String[] ignoreMethodCheckRoles;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RoleUserGroupMappingController.class);

  /**
   * 将指定角色绑定多个用户组
   * @param roleId
   * @param userGroupCodes
   * @return
   */
  @ApiOperation(value = "用户组与角色绑定")
  @PatchMapping(value = "/bindUserGroups")
  public ResponseModel bindUserGroups(@ApiParam(name = "roleId", value = "角色id") @RequestParam("roleId") String roleId,
                                     @ApiParam(name = "userGroupCodes", value = "用户组编码") @RequestParam("groupCodes") String[] userGroupCodes) {
    try {
      this.roleUserGroupMappingService.bindUserGroups(roleId, userGroupCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 解除指定角色和用户组的绑定关系
   * @param roleId
   * @param userGroupCodes
   * @return
   */
  @ApiOperation(value = "解除指定角色和用户组的绑定关系")
  @PatchMapping("/unbindUserGroups")
  public ResponseModel unbindUserGroups(@RequestParam @ApiParam("角色id") String roleId,
                                       @RequestParam @ApiParam("用户组编码") String[] userGroupCodes) {
    try {
      this.roleUserGroupMappingService.unbindUserGroups(roleId, userGroupCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 给指定id的角色重新进行用户组绑定，按照最新给定的用户组进行绑定
   * @param roleId
   * @param userGroupCodes
   * @return
   */
  @ApiOperation(value = "给指定id的角色重新进行用户组绑定，按照最新给定的用户组进行绑定")
  @PostMapping("/reBindUserGroups")
  public ResponseModel reBindUserGroups(@ApiParam(name = "roleId", value = "角色Id") @RequestParam(name = "roleId") String roleId,
                                        @ApiParam(name = "userGroupCodes", value = "用户组编码可以不传或者传空", required = false) @RequestParam(name = "userGroupCodes", required = false) String[] userGroupCodes) {
    try {
      this.roleUserGroupMappingService.reBindUserGroup(roleId, userGroupCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定用户组绑定一个或多个角色
   * @param userGroupCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定用户组绑定一个或多个角色")
  @PatchMapping("/bindUserGroupRoles")
  public ResponseModel bindUserGroupRoles(@RequestParam @ApiParam("用户组编码") String userGroupCode,
                                          @RequestParam @ApiParam("一个或多个角色ID") String[] roleIds) {
    try {
      this.roleUserGroupMappingService.bindUserGroupRoles(userGroupCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 解除指定用户组与角色的绑定
   * @param userGroupCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "解除指定用户组与角色的绑定关系")
  @PatchMapping("/unbindUserGroupRoles")
  public ResponseModel unbindUserGroupRoles(@RequestParam @ApiParam("用户组编码") String userGroupCode,
                                          @RequestParam @ApiParam("一个或多个角色ID") String[] roleIds) {
    try {
      this.roleUserGroupMappingService.unbindUserGroupRoles(userGroupCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 给指定编码的用户组重新进行角色绑定，按照最新给定的角色进行绑定
   * @param userGroupCode
   * @param roleCodes
   * @return
   */
  @ApiOperation(value = "给指定编码的用户组重新进行角色绑定，按照最新给定的角色进行绑定")
  @PostMapping("/reBindUserGroupRoles")
  public ResponseModel reBindUserGroupRoles(@ApiParam(name = "userGroupCode", value = "用户组编码") @RequestParam(name = "userGroupCode") String userGroupCode,
                                            @ApiParam(name = "roleCodes", value = "角色编码可以不传或者传空", required = false) @RequestParam(name = "roleCodes", required = false) String[] roleCodes) {
    try {
      this.roleUserGroupMappingService.reBindUserGroupRoles(userGroupCode, roleCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色编码查询绑定用户组
   * @param roleCode
   * @return
   */
  @GetMapping("findUserGroupByRoleCode")
  @ApiOperation("根据角色编码查询绑定用户组")
  public ResponseModel findByRoleCode(@RequestParam("roleCode") @ApiParam("角色编码") String roleCode) {
    try {
      List<UserGroupVo> userGroups = roleUserGroupMappingService.findByRoleCode(roleCode);
      return buildHttpResultW(userGroups);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
