package com.bizunited.platform.rbac.server.starter.repository;

import com.bizunited.platform.rbac.server.starter.entity.ButtonEntity;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.Set;

/**
 * 为按钮功能的服务提供数据库方法支持
 * @author: zengxingwang
 * @date: 2020/1/13 14:47
 */
@Repository("ButtonRepository")
public interface ButtonRepository  extends JpaRepository<ButtonEntity, String>, JpaSpecificationExecutor<ButtonEntity> {

  /**
   * 取消指定按钮和指定接口的绑定关系
   * @param buttonId 指定的按钮编号
   * @param competenceId 指定的接口编号
   */
  @Modifying
  @Query(value="delete from engine_button_competence_mapping where button_id = :buttonId and competence_id = :competenceId", nativeQuery=true)
  void unbindCompetence(@Param("buttonId") String buttonId, @Param("competenceId") String competenceId);

  /**
   * 根据按钮id删除competence绑定关系
   * @param id
   */
  @Modifying
  @Query(value="delete from engine_button_competence_mapping where button_id = :id", nativeQuery=true)
  void unbindCompetenceById(@Param("id") String id);

  /**
   * 建立指定按钮和指定接口的绑定关系
   * @param buttonId 指定的按钮编号
   * @param competenceId 指定的接口编号
   */
  @Modifying
  @Query(value="insert into engine_button_competence_mapping(button_id,competence_id) values (:buttonId,:competenceId)", nativeQuery=true)
  void bindCompetence(@Param("buttonId") String buttonId, @Param("competenceId") String competenceId);
  /**
   * 建立指定角色和指定按钮的绑定关系
   * @param roleId 指定的角色编号信息
   * @param buttonId 指定的按钮编号
   */
  @Modifying
  @Query(value="insert into engine_role_button_mapping(role_id,button_id) values (:roleId,:buttonId)", nativeQuery=true)
  void bindButton(@Param("roleId") String roleId, @Param("buttonId") String buttonId);

  /**
   * 取消指定角色和指定按钮的绑定关系
   * @param roleId 指定的角色编号信息
   * @param buttonId 指定的按钮编号
   */
  @Modifying
  @Query(value="delete from engine_role_button_mapping where role_id = :roleId and button_id = :buttonId", nativeQuery=true)
  void unbindButton(@Param("roleId") String roleId, @Param("buttonId") String buttonId);
  /**
   * 根据按钮id删除role绑定关系
   * @param id
   */
  @Modifying
  @Query(value="delete from engine_role_button_mapping where button_id = :id", nativeQuery=true)
  void unbindRoleById(@Param("id") String id);

  /**
   * 查询指定的按钮编号和接口编号的是否有且只有一个绑定关系
   * @param buttonId 指定的按钮编号
   * @param competenceId 指定的接口编号
   * @return
   */
  @Query(value="select count(*) from engine_button_competence_mapping where button_id = :buttonId and competence_id = :competenceId", nativeQuery=true)
  long countByCompetenceIdAndButtonId(@Param("buttonId") String buttonId, @Param("competenceId") String competenceId);

  /**
   * 根据编码统计
   * @param code
   * @return
   */
  @Query("select count(*) from ButtonEntity b where b.code = :code and b.projectName=:projectName ")
  long countByCodeAndProjectName(@Param("code") String code, @Param("projectName") String projectName);

  /**
   * 根据编码统计按钮数量，并排除ID
   * @param code
   * @param id
   * @return
   */
  @Query("select count(*) from ButtonEntity b where b.code = :code and b.id <> :id")
  long countByCodeWithoutId(@Param("code") String code,  @Param("id") String id);

  /**
   * 查询指定的角色编号和按钮编号的是否有且只有一个绑定关系
   * @param roleId 指定的角色编号
   * @param buttonId 指定的按钮编号
   * @return
   */
  @Query(value="select count(*) from engine_role_button_mapping where role_id = :roleId and button_id = :buttonId", nativeQuery=true)
  long countByRoleIdAndButtonId(@Param("roleId") String roleId, @Param("buttonId") String buttonId);
  
  /**
   * 根据按钮id查询按钮信息
   * @param id
   * @return
   */
  @Query(value = "select b from ButtonEntity b inner join fetch b.competence c left join fetch b.competences cs left join fetch b.roles r where b.id = :id")
  ButtonEntity findDateilsById(@Param("id") String id);

  /**
   * 根据角色id获取按钮信息
   * @param roleId
   * @return
   */
  @Query(value = "select distinct b from ButtonEntity b inner join b.roles r where r.id = :roleId and b.projectName=:projectName ")
  Set<ButtonEntity> findByRoleIdAndProjectName(@Param("roleId") String roleId, @Param("projectName") String projectName);

  /**
   * 根据编码查询
   * @param code
   * @return
   */
  ButtonEntity findByCodeAndProjectName(String code, String projectName);

  /**
   * 根据菜单ID查询按钮
   * @param competenceId
   * @return
   */
  @Query("select b from ButtonEntity b inner join b.competence c where c.id = :competenceId order by b.createTime desc")
  Set<ButtonEntity> findByCompetenceId(@Param("competenceId") String competenceId);

  /**
   * 根据角色code查询关联的按钮
   * @param roleCode
   * @return
   */
  @Query("select b from ButtonEntity b inner join b.roles r where r.roleCode = :roleCode and b.projectName=:projectName order by b.createTime desc")
  Set<ButtonEntity> findByRoleCodeAndProjectName(@Param("roleCode") String roleCode, @Param("projectName") String projectName);

  /**
   * 根据角色code和功能菜单查询关联的按钮
   * @param roleCode
   * @return
   */
  @Query("select b from ButtonEntity b " +
          "inner join b.roles r " +
          "inner join b.competence c " +
          "where r.roleCode = :roleCode and c.id = :competenceId " +
          "order by b.createTime desc")
  Set<ButtonEntity> findByRoleCodeAndCompetenceId(@Param("roleCode") String roleCode,@Param("competenceId") String competenceId);

}
