package com.bizunited.platform.rbac.server.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RolePositionLevelMappingService;
import com.bizunited.platform.user.common.vo.ReturnPositionLevelVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Map;


/**
 * @author: weikangning
 * @Date: 2020/8/16 21:28
 * @Description: 职位级别
 */
@RestController
@RequestMapping("/v1/nebula/rolePositionLevelMapping")
public class RolePositionLevelMappingController extends BaseController {

  @Autowired
  private RolePositionLevelMappingService rolePositionLevelMappingService;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RolePositionMappingController.class);

  /**
   * 将指定职位级别绑定一个或多个角色
   *
   * @param positionLevelCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定职位级别绑定一个或多个角色")
  @PatchMapping("bindPositionLevelRoles")
  public ResponseModel bindPositionLevelRoles(
          @RequestParam @ApiParam("职位级别编码") String positionLevelCode,
          @RequestParam @ApiParam("当前要绑定的一个或多个角色id") String[] roleIds) {
    try {
      rolePositionLevelMappingService.bindPositionLevelRoles(positionLevelCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定职位级别重新绑定一个或多个角色
   *
   * @param positionLevelCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定职位级别重新绑定一个或多个角色")
  @PatchMapping("rebindPositionLevelRoles")
  public ResponseModel rebindPositionLevelRoles(
          @RequestParam @ApiParam("职位级别编码") String positionLevelCode,
          @RequestParam(required = false) @ApiParam("当前要绑定的一个或多个角色id") String[] roleIds) {
    try {
      rolePositionLevelMappingService.rebindPositionLevelRoles(positionLevelCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 将指定职位级别解除一个或多个角色的绑定
   *
   * @param positionLevelCode
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "将指定职位级别解除一个或多个角色的绑定")
  @PatchMapping("unbindPositionLevelRoles")
  public ResponseModel unbindPositionLevelRoles(
          @RequestParam @ApiParam("职位级别编码") String positionLevelCode,
          @RequestParam @ApiParam("当前要解绑定的一个或多个角色id") String[] roleIds) {
    try {
      rolePositionLevelMappingService.unbindPositionLevelRoles(positionLevelCode, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * @Description: 通过编码数组解绑解绑一个或多个职位级别的全部角色
   * @Param:
   * @Return:
   * @Author: weikangning
   * @Date: 2020/8/27 17:37
   */
  @ApiOperation(value = "解绑一个或多个职位级别的全部角色")
  @PatchMapping("unbindPositionLevels")
  public ResponseModel unbindPositionLevels(
          @RequestParam @ApiParam("职位级别编码数组") String[] positionLevelCodes) {
    try {
      rolePositionLevelMappingService.unbindPositionLevels(positionLevelCodes);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 批量根据职级Id,删除职级、解绑角色
   * @param roleIds 角色Id列表
   */
  @ApiOperation(value = "批量根据职级Id,删除职级、解绑角色")
  @PostMapping("/deletePositionLevelsUnbindRole")
  public ResponseModel deletePositionLevelsUnbindRole(@RequestParam("roleIds") @ApiParam("角色Id列表") String[] roleIds) {
    try {
      this.rolePositionLevelMappingService.deletePositionLevelsUnbindRole(roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据职级Id,删除职级、解绑角色
   * @param roleId 角色Id
   */
  @ApiOperation(value = "根据职级Id,删除职级、解绑角色")
  @PostMapping("/deletePositionLevelUnbindRole")
  public ResponseModel deletePositionLevelUnbindRole(@RequestParam("roleId") @ApiParam("角色Id") String roleId) {
    try {
      this.rolePositionLevelMappingService.deletePositionLevelUnbindRole(roleId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 分页查询
   * @param conditions
   * @param pageable
   * @return
   */
  @ApiOperation(value = "分页查询")
  @PostMapping("/queryPage")
  public ResponseModel queryPage(@RequestBody(required = false) Map<String,Object> conditions,
                                 @PageableDefault(value = 50)@ApiParam(name = "pageable", value = "分页信息") Pageable pageable) {
    try {
      Page<ReturnPositionLevelVo> result = this.rolePositionLevelMappingService.queryPage(conditions, pageable);
      return this.buildHttpResult(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


}
