package com.bizunited.platform.rbac.server.service.redis;

import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RLock;
import org.redisson.api.RMapCache;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * 基于redission的redis客户端实现，这里提供的所有方法都可以直接被技术中台或者基于技术中台建设的业务系统使用
 * @author yinwenjie
 */
public class RedisMutexServiceDefaultImpl implements RedisMutexService {
  
  private static final Logger LOGGER = LoggerFactory.getLogger(RedisMutexServiceDefaultImpl.class);
  
  private static final String REDIS_ERROR = "RedisMutexService key is null!!";
  
  @Autowired
  private RedissonClient redissonClient;
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.redis.RedisMutexService#lock(java.lang.String)
   */
  @Override
  public void lock(String key) {
    Validate.notBlank(key , REDIS_ERROR);
    RLock rlock = redissonClient.getLock(key);
    rlock.lock();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.redis.RedisMutexService#tryLock(java.lang.String, java.util.concurrent.TimeUnit, int)
   */
  @Override
  public boolean tryLock(String key, TimeUnit unit, int timeout) {
    Validate.notBlank(key , REDIS_ERROR);
    Validate.notNull(unit , REDIS_ERROR);
    Validate.isTrue(timeout > 0 , REDIS_ERROR);
    RLock rlock = redissonClient.getLock(key);
    try {
      return rlock.tryLock(timeout, unit);
    } catch (InterruptedException e) {
      LOGGER.error(e.getMessage() , e);
    }
    
    return false;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.redis.RedisMutexService#unlock(java.lang.String)
   */
  @Override
  public void unlock(String key) {
    Validate.notBlank(key , "RedisMutexService key is null!!");
    RLock rlock = redissonClient.getLock(key);
    rlock.unlock();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.redis.RedisMutexService#islock(java.lang.String)
   */
  @Override
  public boolean islock(String key) {
    Validate.notBlank(key , "RedisMutexService key is null!!");
    RLock rlock = redissonClient.getLock(key);
    return rlock.isLocked();
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.redis.RedisMutexService#islock(java.lang.String)
   */
  @Override
  public long getAndIncrement(String code, long min) {
    Validate.notNull(code , "code必须传入");
    long currentMin;
    if(min <= 0) {
      currentMin = 0;
    } else {
      currentMin = min;
    }
    String lockCode = String.format("_async_", code);
    
    /*
     * -处理过程为：
     * 1、由于min的大小可能随时变化，所以一个code的请求，一次只能有一个进程操作
     * ——谁获得了lockCode，谁就进行code增量的操作
     * 2、如果当前获得的值小于currentMin，那么重新设定为currentMin后再进行操作
     * -否则就是正常进行数量新增即可
     * */
    long currentValue = 0l;
    try {
      // 1、====
      this.lock(lockCode);
      // 2、====
      currentValue = this.redissonClient.getAtomicLong(code).getAndIncrement();
      if(currentValue < currentMin) {
        this.redissonClient.getAtomicLong(code).set(currentMin + 1);
        currentValue = currentMin;
      }
    } finally {
      this.unlock(lockCode);
    }
    return currentValue;
  }
  @Override
  public String getAndIncrement(String code, long min, Integer mixStrLen) {
    Validate.isTrue(mixStrLen > 0 , "进行数字格式化的时候，设定的格式化最小长度必须大于0");
    long current = this.getAndIncrement(code, min);
    String currentValue = String.valueOf(current);
    int currentStrLen = currentValue.length();
    
    // 如果条件成立，说明当前返回的数字长度已经满足最小长度，不用再补位了
    if(currentStrLen >= mixStrLen) {
      return currentValue;
    }
    char[] fillChars = new char[mixStrLen];
    Arrays.fill(fillChars, 0, mixStrLen, '0');
    DecimalFormat format = new DecimalFormat(new String(fillChars));
    return format.format(current);
  }
  @Override
  public void setMCode(String mapName, String mapKey, String content, long min) {
    Validate.notBlank(mapName , "进行redission client虚拟map结构操作时，map name必须传入!!");
    Validate.notBlank(mapKey , "进行redission client虚拟map结构操作时，map name下的key值必须传入!!");
    Validate.notBlank(content , "进行redission client虚拟map结构操作时，map name下的key值对应的value内容必须传入!!");
    if(min <= 0) {
      throw new IllegalArgumentException("进行redission client虚拟map结构操作时，必须设定一个有效的过期时间!!");
    }
    
    RMapCache<Object , Object> currentMap = this.redissonClient.getMapCache(mapName);
    // 虽然这样做了判断，但正常情况下，这里不会为null
    Validate.notNull(currentMap , "未获取到redis相关信息，");
    currentMap.put(mapKey, content, min, TimeUnit.MILLISECONDS);
  }
  @Override
  public String getMCode(String mapName, String mapKey) {
    if(StringUtils.isBlank(mapName)) {
      return null;
    }
    if(StringUtils.isBlank(mapKey)) {
      return null;
    }
    
    RMapCache<Object , Object> currentMap = this.redissonClient.getMapCache(mapName);
    // 虽然这样做了判断，但正常情况下，这里不会为null
    if(currentMap == null) {
      return null; 
    }
    Object currentValue = currentMap.get(mapKey);
    return currentValue == null?null:currentValue.toString();
  }
}