package com.bizunited.platform.rbac.server.service.ccode;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.Random;

import javax.imageio.ImageIO;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.bizunited.platform.rbac.server.service.vcode.ValidateCodeService;
import com.bizunited.platform.rbac.server.service.vcode.ValidateCodeType;

/**
 * <b>校验码是一种特殊的验证码</b>，校验码只用在一部分登录场景下，只是用来保证当前的登录鉴权请求并非机器人操作</br>
 * .另外，校验码的外在表达方式，一定是一张指定长宽像素的png图片
 * @author yinwenjie
 */
public class CheckCodeServiceDefaultImpl implements CheckCodeService {
  /**
   *.检验码在系统中的“子系统”名称定义
   */
  private static final String CHECKCODE_SUBSYSTEM = "loginCheck";
  /**
   *.日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(CheckCodeServiceDefaultImpl.class);
  @Autowired
  private ValidateCodeService validateCodeService;
  
  @Override
  public byte[] generate(String userid, int size, int width, int height) {
    /*
     * .处理过程为：
     * 1、首先调用验证码服务，生成一个符合字符串长度要求的验证码：字母（大小写） + 数字
     * (注意：在这一步后，这个验证码就被记录到redis中，且固定的有效期为5分钟)
     * 2、然后按照设定的图片高度和宽度，生成带有颜色混淆层的png底图
     * 3、将得到的验证码嵌入到底图上
     * 4、加了十条随即线条。
     * 5、接着加入随机干扰点，作为图片实时噪点，得到最终的图片效果
     * 6、使用二进制流进行输入
     * */
    // 1、======
    String currentCode = this.validateCodeService.generate(userid, CHECKCODE_SUBSYSTEM, size, ValidateCodeType.IGNORE_CHAR, 5 * 60 * 1000);
    int currentSize = currentCode.length();
    // 2、======
    BufferedImage buffImg = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
    Graphics2D graphics = buffImg.createGraphics();
    graphics.setColor(new Color(240, 240, 240));
    graphics.fillRect(0, 0, width, height);
    
    // 3、======
    // 设置字体
    Font currentFont = new Font("Arial", Font.ITALIC, (int)(height * 0.75f));
    graphics.setFont(currentFont);
    int preWidth = width / (currentSize + 2);
    for (int index = 0; index < currentSize; index++) {
      Character currentChar = currentCode.charAt(index);
      // 设置字体颜色
      graphics.setColor(getRandomColor());
      // 设置字体位置
      graphics.drawString(currentChar.toString(), (index + 1) * preWidth, (int)(height * 0.75));
    }
    
    // 4、======
    Random pointRandom = new Random();
    for (int i = 0; i < 10; i++) {
      graphics.setColor(this.getRandomColor());
      graphics.drawLine(pointRandom.nextInt(width/2), pointRandom.nextInt(height/2), pointRandom.nextInt(width), pointRandom.nextInt(height));
    }
    
    // 5、======
    // 噪点数量为 width * height / 3，噪点坐标随机，颜色随机
    int many = width * height / 3;
    for (int i = 0; i < many; i++) {
      graphics.setColor(this.getRandomColor());
      graphics.fillRect(pointRandom.nextInt(width), pointRandom.nextInt(height), 1, 1);
    }
    
    // 6、=======
    byte[] result = null;
    try (ByteArrayOutputStream boutput = new ByteArrayOutputStream();
        BufferedOutputStream bos = new BufferedOutputStream(boutput)) {
      ImageIO.write(buffImg, "png", bos);
      result = boutput.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage() , e);
    }
    return result;
  }
  
  @Override
  public String generateByBase64(String userid, int size, int width, int height) {
    byte[] contents = this.generate(userid, size, width, height);
    if(contents == null) {
      return null;
    }
    
    // 进行base编码
    return Base64.getEncoder().encodeToString(contents);
  }
  
  @Override
  public boolean match(String userid, String original) {
    if(StringUtils.isBlank(userid)) {
      return false;
    }
    if(StringUtils.isBlank(original)) {
      return false;
    }
    
    // 这里的比对忽略大小写
    return this.validateCodeService.match(userid, CHECKCODE_SUBSYSTEM, original);
  }
  
  /**
   * .该私有方法用于获取一个随机颜色
   * @return
   */
  private Color getRandomColor() {
    Random random = new Random();
    int r = random.nextInt(255);
    int g = random.nextInt(255);
    int b = random.nextInt(255);
    return new Color(r, g, b);
  }
}